#!/bin/bash

MONITOR_NAME="arbitration-monitor"
ETCD_NAME="arbitration-etcd"

function getWorkPath()
{
    dir=$(dirname "$0")
    ispointstart=$(echo "${dir}"|awk -F/ '{print $1}')
    if [ "${ispointstart}" == "." ]
    then
        dir=$(echo "${dir}"|sed 's/^.//')
        curPath="$(pwd)${dir}"
    elif [ "${ispointstart}" == ".." ]
    then
        curPath="$(pwd)/${dir}"
    else
        curPath="${dir}"
    fi
    return 0
}
getWorkPath

function getExecParam()
{
    for i in {1..3}
    do
        [[ ${1#-} == "ip" ]] && {
            SEARCH_IP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "port" ]] && {
            SEARCH_PORT=$2
            shift 2
            continue
        }
        [[ ${1#-} == "index" ]] && {
            SEARCH_INDEX=$2
            shift 2
            continue
        }
    done
}


# 如果有输入IP，检查IP是否满足
function find_ip_match()
{
    local ips="$1"
    for find_ip_match_ip in ${SEARCH_IP[@]}
    do
        echo "${ips}" | grep -wF "${find_ip_match_ip}" > /dev/null 2>&1
        if [ $? -ne 0 ];then
            return 1
        fi
    done
    return 0
}

function find_index_match()
{
    local index="$1"
    for find_index_match_index in ${SEARCH_INDEX[@]}
    do
        echo "${index}" | grep -wF "${find_index_match_index}" > /dev/null 2>&1
        if [ $? -ne 0 ];then
            return 1
        fi
    done
    return 0
}

function find_port_match()
{
    local etcd_port=$1
    local server_port=$2
    local result1=
    local result2=
    for find_port_match_port in ${SEARCH_PORT[@]}
    do
        echo "${etcd_port}" | grep -wF "${find_port_match_port}" > /dev/null 2>&1
        result1=$?
        echo "${server_port}" | grep -wF "${find_port_match_port}" > /dev/null 2>&1
        result2=$?
        if [ ${result1} -ne 0 -a ${result2} -ne 0 ];then
            return 1
        fi
    done
    return 0
}

# 查询当前节点已经安装的仲裁软件的信息
function query_etcd_ips()
{
    ls /opt | grep "${MONITOR_NAME}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        echo "No arbiration software found."
        return 1
    fi
    print_title
    for item in $(ls /opt)
    do
        echo "${item}" | grep ${MONITOR_NAME} > /dev/null 2>&1
        if [ $? -eq 0 ];then
            echo "${item}" | grep "${MONITOR_NAME}_" > /dev/null 2>&1
            if [ $? -ne 0 ];then
                index=0
            else
                index=$(echo "${item}" | sed "s/${MONITOR_NAME}_//g")
            fi
            aam_properties=$(get_config_file ${index})
            if [ ! -f "${aam_properties}" ];then
                continue
            fi
            echo "${aam_properties}" | grep ${ETCD_NAME} > /dev/null 2>&1
            if [ $? -eq 0 ];then
                ips=$(get_ip_from_etcd_conf "${aam_properties}")
                etcd_port=$(get_etcd_port_from_etcd_conf "${aam_properties}")
                server_port=$(get_server_port_from_etcd_conf "${aam_properties}")
            else
                ips=$(get_ip_from_aam_conf "${aam_properties}")
                etcd_port=$(get_etcd_port_from_aam_conf "${aam_properties}")
                server_port=$(get_server_port_from_aam_conf "${aam_properties}")
            fi
            find_ip_match "${ips}" || continue
            find_port_match "${etcd_port}" "${server_port}" || continue
            find_index_match ${index} || continue
            print_info "${index}" "${ips}" "${etcd_port}" "${server_port}"
        fi
    done
}

function get_config_file() {
    index=$1
    if [ ${index} -eq 0 ];then
        etcd_path=${ETCD_NAME}
        monitor_path=${MONITOR_NAME}
    else
        etcd_path=${ETCD_NAME}_${index}
        monitor_path=${MONITOR_NAME}_${index}
    fi
    if [ -f "/opt/${etcd_path}/bin/etcd" ];then
        aam_properties=/opt/"${etcd_path}"/conf/etcd_server.yaml
        if [ ! -f "${aam_properties}" ];then
            aam_properties=/opt/"${etcd_path}"/conf/etcd_server.properties
        fi
    else
        aam_properties=/opt/"${monitor_path}"/conf/aam.properties
    fi
    echo ${aam_properties}
}


function print_title()
{
    printf "%-10s%-60s%-20s%-10s\n" "INDEX" "IP" "ETCD_PORT" "SERVER_PORT"
}

function print_info()
{
    local index=$1
    local ip=$2
    local etcd_port=$3
    local server_port=$4
    printf "%-10s%-60s%-20s%-10s\n" "${index}" "${ip}" " ${etcd_port}" "${server_port}"
}


# 从配置文件中获取仲裁软件使用到的IP
function get_ip_from_aam_conf()
{
    conf_file=$1

    local local_etcd_servers=$(cat ${conf_file} | grep "aam.local.etcdservers" | awk -F"=" '{print $2}')
    local remote_etcd_servers=$(cat ${conf_file} | grep "aam.remote.1.etcdservers" | awk -F"=" '{print $2}')
    local third_etcd_servers=$(cat ${conf_file} | grep "aam.public.etcdservers" | awk -F"=" '{print $2}')

    local_etcd_servers=$(echo "${local_etcd_servers}" | sed -E 's/:[0-9]+$//g' | sed -E 's/\[//g' | sed -E 's/\]//g')
    remote_etcd_servers=$(echo "${remote_etcd_servers}" | sed -E 's/:[0-9]+$//g' | sed -E 's/:[0-9]+,/,/g' | sed -E 's/\[//g' | sed -E 's/\]//g')
    third_etcd_servers=$(echo "${third_etcd_servers}" | sed -E 's/:[0-9]+$//g' | sed -E 's/\[//g' | sed -E 's/\]//g')

    local current_dc=$(cat "${conf_file}" | grep "aam.local.dcserial" | awk -F"=" '{print $2}')
    if [ "${current_dc}" == "public" ];then
        echo "${local_etcd_servers},${remote_etcd_servers}"
    else
        echo "${local_etcd_servers},${remote_etcd_servers},${third_etcd_servers}"
    fi

    return 0
}

# 从配置文件中获取仲裁软件使用到的ETCD端口
function get_etcd_port_from_aam_conf()
{
    conf_file=$1
    local port=$(cat ${conf_file} | grep "aam.public.etcdservers" | awk -F":" '{print $NF}')
    port=$(expr ${port} + 1)
    echo "${port}"
    return 0
}

# 从配置文件中获取仲裁软件使用到的服务端端口
function get_server_port_from_aam_conf()
{
    conf_file=$1
    local port=$(cat ${conf_file} | grep "aam.public.etcdservers" | awk -F":" '{print $NF}')
    echo "${port}"
    return 0
}

# 适配从老版本配置文件中获取仲裁软件使用到的IP
function get_ip_from_etcd_conf()
{
    conf_file=$1
    ips=

    echo "${conf_file}" | grep 'yaml$' > /dev/null 2>&1
    if [ $? -ne 0 ];then
        arr_conf_ips=$(cat ${conf_file} | grep 'initial_cluster=' | sed 's/initial_cluster=//g' | sed 's/,/ /g')
    else
        arr_conf_ips=$(cat ${conf_file} | grep 'initial-cluster:' | sed 's/initial_cluster://g' | sed 's/ //g' | sed 's/,/ /g')
    fi
    local port=
    for get_ip_from_aam_conf_item in ${arr_conf_ips[@]}
    do
        port=$(echo "${get_ip_from_aam_conf_item}"| awk -F":" '{print $NF}')
        temp_ip=$(echo "${get_ip_from_aam_conf_item}"| awk -F'//' '{print $2}' | awk -F":${port}" '{print $1}')
        ips=$(echo "${ips} ${temp_ip}")
    done
    echo "${ips}" | sed 's/^ //g'
    return 0
}

# 适配从老版本配置文件中获取仲裁软件使用到的ETCD端口
function get_etcd_port_from_etcd_conf()
{
    conf_file=$1
    local port=

    echo "${conf_file}" | grep 'yaml$' > /dev/null 2>&1
    if [ $? -ne 0 ];then
        port=$(cat ${conf_file} | grep 'initial_advertise_peer_urls=' | awk -F':' '{print $NF}')
    else
        port=$(cat ${conf_file} | grep 'listen-peer-urls:' | awk -F':' '{print $NF}')
    fi

    echo "${port}"
    return 0
}

# 适配从老版本配置文件中获取仲裁软件使用到的服务端端口
function get_server_port_from_etcd_conf()
{
    conf_file=$1
    local port=

    echo "${conf_file}" | grep 'yaml$' > /dev/null 2>&1
    if [ $? -ne 0 ];then
        port=$(cat ${conf_file} | grep 'listen_client_urls=' | awk -F':' '{print $NF}')
    else
        port=$(cat ${conf_file} | grep 'listen-client-urls:' | awk -F':' '{print $NF}')
    fi

    echo "${port}"
    return 0
}

function main()
{
    getExecParam "$@"
    query_etcd_ips || return 1
}

main "$@"