#!/bin/bash
#
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
#
. ./arbitration_opertate_lib
backup_path="/opt/backupArb"
upgrade_path="/opt/upgradeArb"

#################################################
#
# 生成check文件
#
#################################################
function create_check_file() {
    result_code=$1
    create_check_result_file ${result_code} "check_arbitration_os_mediation_result.sh"

    if [ $result_code -ne 100 ]; then
        # 注意echo的使用，可能会使这里的exit失效
        echo "exit"
    fi
}

#################################################
#
# 获取目标版本
#
#################################################
function get_target_version() {
    package_name=$1
    if [ -z ${package_name} ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "miss package name."
        $(create_check_file 200)
    fi

    target_version=$(echo ${package_name} | awk -F"_" '{print $2}')
    if [ -z ${target_version} ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "get target version fail."
        $(create_check_file 200)
    fi
}

#################################################
#
# 预处理sopuser创建的flag目录
#
#################################################
function init_flag_path() {
    create_sopuser_flag="${backup_path}/${backup_dir}_create_user.flag"
    change_group_flag="${backup_path}/${backup_dir}_change_group.flag"
    if [ ! -d "${backup_path}" ]; then
        mkdir -p "${backup_path}"
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "create dir: ${backup_path}."
    fi
}

#################################################
#
# 获取升级参数
#
#################################################
function get_upgrade_params() {

    echo "es_input: node_arbitrate_new_sopuser_pwd"
    read -s -r node_arbitrate_new_sopuser_pwd

    get_target_version $1

    package_base_path=$2
    if [ -z ${package_base_path} ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "miss package path."
        $(create_check_file 200)
    fi

    third_login_user=$3
    arbitration_log INFO "ARBITRATION_OS_MEDIATION" "upgrade third_login_user: ${third_login_user}"

    backup_dir=$4
    init_flag_path
}

#################################################
#
# 获取回滚参数
#
#################################################
function get_rollback_params() {
    get_target_version $1

    os_mediation_path="${upgrade_path}/NCEEngr/O2_path/upgrade_config_${target_version}"
    if [ ! -d ${os_mediation_path} ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "upgrade back path not exist"
        $(create_check_file 203)
    fi

    third_login_user=$2
    arbitration_log INFO "ARBITRATION_OS_MEDIATION" "rollback third_login_user: ${third_login_user}"

    backup_dir=$3
    init_flag_path
}

#################################################
#
# 升级配置前初始化：预置目录、解压包
#
#################################################
function upgrade_config_init() {
    if [ ! -d "${upgrade_path}" ]; then
        mkdir "${upgrade_path}"
        chown ossadm:ossgroup "${upgrade_path}"
        chmod 750 "${upgrade_path}"
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "create dir: ${upgrade_path}."
    fi

    package_path="${package_base_path}/${package_name}"
    os_mediation_path="${package_base_path}/os_mediation_${target_version}"
    if [ -e "${package_path}" ]; then
        su - ossadm -c "rm -rf ${os_mediation_path}"
        su - ossadm -c "unzip ${package_path} -d ${os_mediation_path}" >/dev/null 2>&1
        su - ossadm -c "chmod 400 ${os_mediation_path}/*"
        su - ossadm -c "chmod 500 ${os_mediation_path}/*.sh"
    else
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "OSMediation not exist"
        $(create_check_file 201)
    fi
}

#################################################
#
# 执行升级
#
#################################################
function exe_upgrade() {
    private_key_file="/root/.gnupg/secring.gpg"
    if [ -f "${private_key_file}" ]; then
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "delete the ${private_key_file} file."
        rm -f "${private_key_file}"
    fi
    exe_file="${os_mediation_path}/execute_upgrade.sh"
    su - ossadm -c "bash ${exe_file} 'single-upgrade' ${target_version} ${package_path} ${upgrade_path}" >/dev/null 2>&1
    result=$?
    if [ ${result} -ne 0 ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "upgrade fail on execute_upgrade. result: ${result}"
        $(create_check_file 202)
    fi
    arbitration_log INFO "ARBITRATION_OS_MEDIATION" "arbitration os mediation upgrade success."
}

#################################################
#
# 执行回滚
#
#################################################
function exe_rollback() {
    exe_file="${os_mediation_path}/execute_upgrade.sh"
    su - ossadm -c "bash ${exe_file} 'single-rollback' ${target_version} -- ${upgrade_path}" >/dev/null 2>&1
    result=$?
    if [ ${result} -ne 0 ]; then
        arbitration_log WARN "ARBITRATION_OS_MEDIATION" "rollback fail on execute_upgrade. result: ${result}"
        return 0
    fi
    arbitration_log INFO "ARBITRATION_OS_MEDIATION" "arbitration os mediation rollback success."
}

#################################################
#
# 创建sopuser用户
# 当前环境没有sopuser用户 且 三方登录用户为ossadm 且 有node_arbitrate_new_sopuser_pwd时,创建sopuser用户
#
#
#################################################
function create_sopuser() {
    id "sopuser" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "sopuser is exist."
        return
    fi
    if [ $third_login_user != "ossadm" ]; then
        return
    fi
    if [ -z $node_arbitrate_new_sopuser_pwd ]; then
        return
    fi

    # 强制删除username文件，防止注入密码失败
    rm -f /usr/local/os_tool/etc/username.properties
    local user="sopuser"
    # 注入用户密码
    echo -e "${node_arbitrate_new_sopuser_pwd}" | su - ossadm -c "bash /usr/local/osconfig/os/bin/injectpvalue.sh ${user}" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "user:${user} injectp failed"
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "user:${user} injectp failed"
        $(create_check_file 204)
    fi
    echo "user:${user} injectp success"

    # 创建用户
    su - ossadm -c "bash /usr/local/osconfig/os/bin/addosuser.sh ${user}" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "user:${user} create failed"
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "user:${user} create failed"
        $(create_check_file 204)
    fi
    echo "user:${user} create success"

    # 标记新增sopuser用户，用于回滚识别修改
    echo "sopuser" >$create_sopuser_flag
}

#################################################
#
# 修改ossadm用户组, 低版本升级时将ossadm加入sopgroup用户组
#
#################################################
function change_ossadm_group() {
    id "ossadm" | grep "sopgroup" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "ossadm already in sopgroup."
        return
    fi

    su - ossadm -c "bash /usr/local/osconfig/os/bin/changeusergroup.sh -type upgrade" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "change ossadm group success."
    fi

    # 标记修改ossadm用户组，用于回滚识别修改
    echo "ossadm" >"${change_group_flag}"
}

#################################################
#
# 升级
#
#################################################
function upgrade() {
    # 获取参数
    get_upgrade_params "$@"
    # 配置初始化
    upgrade_config_init
    # 执行升级
    exe_upgrade
    # 创建sopuser用户
    create_sopuser
    # 修改ossadm用户组
    change_ossadm_group
}

#################################################
#
# 删除本次升级创建的sopuser
#
#################################################
function remove_cur_sopuser() {
    if [ $third_login_user != "ossadm" ]; then
        return
    fi
    # 判断sopuser是否为本次升级创建
    grep "sopuser" $create_sopuser_flag >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        return
    fi

    # 共部署三方节点不做回滚删除，其他仲裁可能需要使用sopuser用户
    if [ $(ls /opt | grep -c "arbitration-monitor") -gt 1 ]; then
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "more arbitration in third, don't rollback sopuser."
        return
    fi

    # 删除sopuser用户
    local user="sopuser"
    for ((i = 1; i <= 3; i++)); do
        killall -u ${user} >/dev/null 2>&1
    done
    userdel -rf ${user}
    if [ $? -ne 0 ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "Failed to delete the ${user}."
        $(create_check_file 204)
    fi
    sed -i "/^${user}/d" /etc/security/opasswd
    arbitration_log INFO "ARBITRATION_OS_MEDIATION" "The ${user} information is cleared."
}

#################################################
#
# 回滚ossadm用户组变更
#
#################################################
function rollback_ossadm_group() {
    # 判断本次升级是否修改ossadm用户组
    grep "ossadm" $change_group_flag >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        return
    fi

    arbitration_log INFO "ARBITRATION_OS_MEDIATION" "rollback ossadm group change."

    su - ossadm -c "bash /usr/local/osconfig/os/bin/changeusergroup.sh -type rollback" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        arbitration_log INFO "ARBITRATION_OS_MEDIATION" "rollback ossadm group change success."
    fi

    echo "rollback" >$change_group_flag
}

#################################################
#
# 回滚
#
#################################################
function rollback() {
    # 获取参数
    get_rollback_params "$@"
    # 回滚ossadm用户组变更
    rollback_ossadm_group
    # 执行回滚
    exe_rollback
    # 删除本次升级创建的sopuser
    remove_cur_sopuser
}

#################################################
#
# 主函数
#
#################################################
function main() {
    action=$1 && shift 1
    if [ -z ${action} ]; then
        arbitration_log ERROR "ARBITRATION_OS_MEDIATION" "miss action."
        $(create_check_file 200)
    fi

    if [ ${action} == "upgrade" ]; then
        upgrade "$@"
    else
        rollback "$@"
    fi
}

main "$@"
$(create_check_file 100)
