#!/bin/bash
##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

# 包装错误信息，打印到easysuite前台界面
function echo_message()
{
    echo "$@" 1>&2
}

##################################################
# 校验执行用户
# 脚本要求使用root用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    if [ "${user}" != "root" ]
    then
        echo_message "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}


##################################################
# 日志记录进/opt/backupArb/log/${SELF_FILE}_*.log
##################################################
function LOG()
{
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]
    then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2" | tee -a ${LOG_FILE}
    if [ "${Level}" == "ERROR" ]
    then
        echo_message "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2"
    fi
}

# 获取输入参数
function init_params()
{
    local num=$#
    if [ $((${num}%2)) -eq 1 ]
    then
        num=$((${num}/2+1))
    else
        num=$((${num}/2))
    fi
    local count=1
    for((i=1;i<=${num};i++))
    do
        [[ ${1#-} == "iplist" ]] && { IPLIST=$2;shift 2; continue; }
        [[ ${1#-} == "backup_dir" ]] && { BACKUP_DIR=$2;shift 2; continue; }
        [[ ${1#-} == "timestamp" ]] && { TIMESTAMP=$2;shift 2; continue; }
        [[ ${1#-} == "thirdip" ]] && { THIRDIP=$2;shift 2; continue; }
        [[ ${1#-} == "action" ]] && { ACTION=$2;shift 2; continue; }
    done

    local log_path=/opt/backupArb/log
    if [ ! -d ${log_path} ];then
        mkdir -p /opt/backupArb/log > /dev/null 2>&1
    fi
    LOG_FILE="${log_path}/${SELF_FILE//.sh/}_${BACKUP_DIR}.log"
}

##################################################
# 校验入参
# @param
#   -iplist         --192.168.75.195,192.168.75.197,192.168.75.199
#   -pkg            --目标软件包名
#   -timestamp      --当前任务时间戳
#   -thirdip        --三方站点IP
#   -action         --任务动作
##################################################
function check_input()
{
    init_params "$@"
    if [ $? -ne 0 ]
    then
        echo "Example: bash ${SELF_FILE} -iplist 192.168.75.195,192.168.75.197,192.168.75.199"\
        "-pkg NCE_AAMonitor.zip -timestamp 1605078004164 -thirdip 192.168.75.199 -action stop"
        LOG "ERROR" "The input param is invalid."
        return 1
    fi
    
}


function get_index()
{
    index=""
    for check_installed_item in $(ls /opt | grep arbitration-etcd)
    do
        echo "${check_installed_item}" | grep "_" > /dev/null 2>&1
        if [ $? -ne 0 ];then
            tmp_index="0"
        else
            tmp_index=$(echo "${check_installed_item}" | sed "s/arbitration-etcd_//g")
        fi
        index="${index} ${tmp_index}"
    done

    for i in {0..10}
    do
        echo "${index}" | grep -w "${i}"
        if [ $? -ne 0 ];then
            echo "${i}"
            return 0
        fi
    done
}


# 初始化任务相关信息
function init_task_info()
{
    # 获取输入参数
    check_input "$@" || return 1
    
    # 确定回滚备份文件目录
    VERSION_PATH="/opt/backupArb/${BACKUP_DIR}"
    
    # 解析输入参数,确定当前站点类型
    # NCE:NCE主备 THIRD:三方仲裁
    ifconfig | grep -qw "${THIRDIP}"
    if [ $? -ne 0 ]
    then
        SITE_TYPE="NCE"
        LOG "INFO" "The current site is a NCE site."
    else
        SITE_TYPE="THIRD"
        LOG "INFO" "The current site is a third site."
    fi
    
    # 当前环境为三方站点时，确定尾缀参数
    if [ "${SITE_TYPE}" = "THIRD" ]
    then
        LOG "INFO" "Start to query index."
        local query_script="/opt/arb_upgrade/es_script/query_arbitration.sh"
        if [ -z "${query_script}" ]
        then
            LOG "WARN" "Failed to confirm query scripts."
            TAIL_INDEX=$(get_index)
            return 0
        fi
        bash ${query_script} > /var/log/arb_stie_info.log
        if [ $? -ne 0 ]
        then
            LOG "WARN" "Failed to query arb info."
            TAIL_INDEX=$(get_index)
            return 0
        fi
        local cmd="cat /var/log/arb_stie_info.log"
        local ips=$(echo ${IPLIST} | awk -F',' '{for(i=1;i<=NF;++i) print $i}')
        for ip in ${ips}
        do
            cmd="${cmd} | grep '${ip}'"
        done
        TAIL_INDEX=$(eval ${cmd} | awk '{print $1}')
        if [ -z "${TAIL_INDEX}" ]; then
            TAIL_INDEX=$(get_index)
        fi
        LOG "INFO" "Index is ${TAIL_INDEX}."
        LOG "INFO" "Finished to query index."
    else
        TAIL_INDEX="0"
    fi
    LOG "INFO" "Index is ${TAIL_INDEX}"
}

function init_path() {
    if [ "${TAIL_INDEX}" = "0" ]
    then
        TARGET="/opt/arbitration/"
        TARGET_M="/opt/arbitration-monitor/"
        TARGET_E="/opt/arbitration-etcd/"
    else
        TARGET="/opt/arbitration_${TAIL_INDEX}/"
        TARGET_M="/opt/arbitration-monitor_${TAIL_INDEX}/"
        TARGET_E="/opt/arbitration-etcd_${TAIL_INDEX}/"
    fi

    LOG "INFO" "TARGET is ${TARGET}."
    LOG "INFO" "TARGET_M is ${TARGET_M}."
    LOG "INFO" "TARGET_E is ${TARGET_E}."

    BACKUP_PATH="/opt/backupArb/${BACKUP_DIR}/${TAIL_INDEX}"
    mkdir -p "${BACKUP_PATH}" &>>${LOG_FILE}
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "Failed to create backup path:${BACKUP_PATH}."
        return 1
    fi
}

# 重启定时任务
function restart_crontab()
{
    cron_service_name="crond.service"
    systemctl -l | grep cron.service >/dev/null 2>&1
    if [ $? -eq 0 ]
    then
        cron_service_name="cron.service"
    fi
    LOG "INFO" "Start to restart ${cron_service_name}"
    systemctl restart ${cron_service_name} &>>${LOG_FILE}
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "Failed to restart ${cron_service_name}"
        return 1
    fi
    LOG "INFO" "Finished to restart ${cron_service_name}"
}

# 等待定时任务结束
function wait_cron_task_stop()
{
    sleep 5
    for i in {1..10} ; do
        ps -ef | grep $1 | grep -v grep >/dev/null 2>&1
        if [ $? -ne 0 ]; then
            return 0
        fi
        sleep 1
        LOG "INFO" "Waiting cron task to end: ${i}"
    done
    LOG "WARN" "cron task is running timeout"
}

# 删除定时任务并重启
function clean_restart_crontab()
{
    local aamonitor_watchdog_path_format=$(echo "${TARGET%/}/script/aamonitor_watchdog.sh" | sed 's/\//\\\//g')
    sed -i "/${aamonitor_watchdog_path_format}/d" /etc/crontab 2>/dev/null
    sed -i "/${aamonitor_watchdog_path_format}/d" /var/spool/cron/root 2>/dev/null
    
    restart_crontab || return 1

    wait_cron_task_stop ${aamonitor_watchdog_path_format}
}

function kill_process() {
    local process=$1
    pid=$(ps -ef | grep "${process}" | grep -v grep | awk '{print $2}')
    if [ -n "${pid}" ];then
        kill -9 "${pid}" &>> "${LOG_FILE}"
    fi
}

# 停止指定服务
function stop_specified_service()
{
    local index_file=$1
    local server_name=$2
    LOG "INFO" "Start to stop arbitration-${server_name}."
    if [ "${index_file}" == "0" ]
    then
        local tail=""
    else
        local tail="_${index_file}"
    fi
    if [ ! -f /opt/arbitration-${server_name}${tail}/script/service.sh ];then
        # 脚本不存在，直接kill
        LOG "WARN" "/opt/arbitration-${server_name}${tail}/script/service.sh is not exist, kill the process"
        kill_process "/opt/arbitration-${server_name}${tail}"
        return 0
    fi
    bash /opt/arbitration-${server_name}${tail}/script/service.sh stop &>>${LOG_FILE}
    if [ $? -ne 0 ]
    then
        sudo -u arbiter bash /opt/arbitration-${server_name}${tail}/script/service.sh stop&>>${LOG_FILE}
        if [ $? -ne 0 ]
        then
            LOG "ERROR" "Failed to stop arbitration-${server_name}."
            return 1
        fi
    fi
    LOG "INFO" "Finished to stop arbitration-${server_name}."
}

# 回滚仲裁
function rollback_application()
{
    LOG "INFO" "Start to rollback arbitration application."
    local index_file=$1
    BACKUP_PATH="${VERSION_PATH}/${index_file}"
    if [ ! -d "${BACKUP_PATH}" ]
    then
        LOG "ERROR" "Failed to find backup path."
        return 1
    fi
    if [ "${index_file}" == "0" ]
    then
        # 删除应用程序
        rm -rf /opt/arbitration
        rm -rf /opt/arbitration-common
        rm -rf /opt/arbitration-etcd
        rm -rf /opt/arbitration_file
        rm -rf /opt/arbitration-monitor
        
        # 恢复应用程序
        tar --warning=no-file-changed -xf ${BACKUP_PATH}/arbitrationBackup.tar.gz -C /opt &>>${LOG_FILE}
        result=$?
        if [ ${result} -ne 0 ]
        then
            LOG "ERROR" "Failed to unzip backup file:${BACKUP_PATH}/arbitrationBackup.tar.gz. Return Code:${result}"
            return 1
        fi
        
        chown -R root:root /opt/arbitration 2>/dev/null
        chown -R arbiter:ArbiterGroup /opt/arbitration-common 2>/dev/null
        chown -R arbiter:ArbiterGroup /opt/arbitration-etcd 2>/dev/null
        chown -R root:root /opt/arbitration_file 2>/dev/null
        chown -R arbiter:ArbiterGroup /opt/arbitration-monitor 2>/dev/null

        # 清理日志目录
        rm -rf /var/log/arbitration-etcd/*  >/dev/null 2>&1
        rm -rf /var/log/arbitration-monitor/*  >/dev/null 2>&1
    else
        # 删除应用程序
        rm -rf /opt/arbitration_${index_file}
        rm -rf /opt/arbitration-common_${index_file}
        rm -rf /opt/arbitration-etcd_${index_file}
        rm -rf /opt/arbitration_file
        rm -rf /opt/arbitration-monitor_${index_file}

        # 恢复应用程序
        tar --warning=no-file-changed -xf ${BACKUP_PATH}/arbitrationBackup.tar.gz -C /opt &>>${LOG_FILE}
        result=$?
        if [ ${result} -ne 0 ]
        then
            LOG "ERROR" "Failed to unzip backup file:${BACKUP_PATH}/arbitrationBackup.tar.gz. Return Code:${result}"
            return 1
        fi
        
        chown -R root:root /opt/arbitration_${index_file} 2>/dev/null
        chown -R arbiter:ArbiterGroup /opt/arbitration-common_${index_file} 2>/dev/null
        chown -R arbiter:ArbiterGroup /opt/arbitration-etcd_${index_file} 2>/dev/null
        chown -R root:root /opt/arbitration_file 2>/dev/null
        chown -R arbiter:ArbiterGroup /opt/arbitration-monitor_${index_file} 2>/dev/null

        # 清理日志目录
        rm -rf /var/log/arbitration-etcd/${index_file}/*  >/dev/null 2>&1
        rm -rf /var/log/arbitration-monitor/${index_file}/*  >/dev/null 2>&1
    fi

    # 恢复日志
    tar --warning=no-file-changed -xf ${BACKUP_PATH}/arbitrationLog.tar.gz -C /var/log &>>${LOG_FILE}
    result=$?
    if [ ${result} -ne 0 ]
    then
        LOG "ERROR" "Failed to unzip backup file:${BACKUP_PATH}/arbitrationLog.tar.gz. Return Code:${result}"
        return 1
    fi
    
    chown -R arbiter:root /var/log/arbitration
    chown -R arbiter:ArbiterGroup /var/log/arbitration-etcd
    chown -R root:root /var/log/arbitration_install
    chown -R arbiter:ArbiterGroup /var/log/arbitration-monitor
    
    # 恢复用户配置
    local old_config=$(cat ${BACKUP_PATH}/passwd.bak 2>/dev/null)
    if [ -n "${old_config}" ]
    then
        sed -i "s|^arbiter.*|${old_config}|" /etc/passwd
    fi
    LOG "INFO" "Finished to rollback arbitration application."
}

# 恢复定时任务
function restore_crontab()
{
    # 恢复定时任务
    cp -f ${BACKUP_PATH}/crontab.bak /etc/crontab 2>/dev/null
    
    if [ -f "/var/spool/cron/root" ]
    then
        cp -f ${BACKUP_PATH}/crontab_spool.bak /var/spool/cron/root 2>/dev/null
    fi
    
    restart_crontab || return 1
}

# 重启指定服务
function restart_specified_service()
{
    local index_file=$1
    local server_name=$2
    LOG "INFO" "Start to restart arbitration-${server_name}."
    if [ "${index_file}" == "0" ]
    then
        local tail=""
    else
        local tail="_${index_file}"
    fi

    # 兼容历史版本三方站点monitor不启动
    if [ ${server_name} == "monitor" ];then
        if [ ! -f /opt/arbitration-${server_name}${tail}/conf/aam.properties ];then
            LOG "WARN" "aam.properties not exist, skip restart"
            return 0
        fi
        local aam_local_role="$(cat /opt/arbitration-${server_name}${tail}/conf/aam.properties |grep "aam.local.role" | cut -d "=" -f 2)"
        if [ ! -n "${aam_local_role}" ] ;then
            LOG "INFO" "Skip to restart arbitration-${server_name}."
            return 0
        fi
    fi

    if [ ! -f /opt/arbitration-${server_name}${tail}/script/service.sh ];then
        LOG "WARN" "arbitration-${server_name}${tail}/script/service.sh not exist, skip restart"
        return 0
    fi
    local count=0
    while true
    do
        bash /opt/arbitration-${server_name}${tail}/script/service.sh restart &>>${LOG_FILE}
        if [ $? -ne 0 ]
        then
            sudo -u arbiter bash /opt/arbitration-${server_name}${tail}/script/service.sh restart&>>${LOG_FILE}
            if [ $? -ne 0 ]
            then
                let count++
            else
                break
            fi
        else
            break
        fi
        sleep 120
        if [ ${count} -eq 3 ]
        then
            LOG "ERROR" "Failed to restart arbitration-${server_name}."
            return 1
        fi
    done
    LOG "INFO" "Finished to restart arbitration-${server_name}."
}

# 清理仲裁备份,生成回滚完成标志
function clean_backup()
{
    echo "rollback:${TIMESTAMP}" >> /opt/backupArb/arb_backup_rollback.flag
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "Failed to echo backup timestamp:${TIMESTAMP}."
        return 1
    fi
    [ -d "${BACKUP_PATH}" ] && rm -rf "${BACKUP_PATH}"
    file_num=$(ls "${VERSION_PATH}" 2>/dev/null | wc -l)
    if [ ${file_num} -eq 0 ]
    then
        [ -d "${VERSION_PATH}" ] && rm -rf "${VERSION_PATH}"
    fi
}

# 如果上次已经回滚成功，重入直接跳过
function check_flag()
{
    grep -q "rollback:${TIMESTAMP}" /opt/backupArb/arb_backup_rollback.flag 2>/dev/null
    if [ $? -eq 0 ]
    then
        LOG "INFO" "The current project has been rolled back. Skip this step."
        return 0
    else
        return 1
    fi
}

# 停止仲裁软件定时任务和monitor+etcd服务
function stop_arb()
{
    clean_restart_crontab || return 1
    stop_specified_service "${TAIL_INDEX}" "etcd" || return 1
    stop_specified_service "${TAIL_INDEX}" "monitor" || return 1
    LOG INFO "Kill ${TARGET}"
    kill_process "${TARGET}"
    LOG INFO "Kill ${TARGET_M}"
    kill_process "${TARGET_M}"
    LOG INFO "Kill ${TARGET_E}"
    kill_process "${TARGET_E}"
}

# 回滚仲裁软件
function rollback_arb()
{
    check_flag && return 0
    rollback_application "${TAIL_INDEX}" || return 1
}

function main()
{
    check_user || return 1

    init_task_info "$@" || return 1

    init_path || return 1
    
    case ${ACTION} in
        "stop")
            stop_arb || return 1
        ;;
        "rollback")
            rollback_arb || return 1
        ;;
        "restart_etcd")
            restart_specified_service "${TAIL_INDEX}" "etcd" || return 1
        ;;
        "restart_monitor")
            restart_specified_service "${TAIL_INDEX}" "monitor" || return 1
            restore_crontab || return 1
            clean_backup
        ;;
        *)
            LOG "ERROR" "Unsupported action."
            return 1
        ;;
    esac
}

main "$@"
exit $?
