#!/bin/bash

# 获取脚本工作路径和脚本名
MODULE_NAME="$(basename $0)"
SCRIPT_PATH=$(cd $(dirname $0);pwd)

BACKUP_BAK_JSON="${SCRIPT_PATH}/server_bak_server.json"
BACKUP_TMP_JSON="${SCRIPT_PATH}/server_tmp_server.json"

# 生成获取结果脚本
function create_check_file()
{
    echo "#!/bin/bash
echo  RESULT:$1
    ">${SCRIPT_PATH}/backupdata_task_result.sh
    chmod 750 ${SCRIPT_PATH}/backupdata_task_result.sh
}

# 查询备份服务器
function get_backup_server_info()
{
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: Start to query backup server information."
    local backup_server_info="${SCRIPT_PATH}/get_backup_server_info.py"
    if [ ! -f "${backup_server_info}" ]
    then
        local backup_server_info="${SCRIPT_PATH}/get_backup_server_info.pyc"
    fi
    if [ -f "${backup_server_info}" ]
    then
        python ${backup_server_info} "${SCRIPT_PATH}" >"${SCRIPT_PATH}/get_backup_server.log"
        grep "get_backup_server_info_succeed" "${SCRIPT_PATH}/get_backup_server.log" &>/dev/null
        if [ $? -eq 0 ]
        then
            echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: Succeeded in querying the backup server information."
            return 0
        fi
    fi
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] ERROR: Failed to query the backup server information."
    return 1
}

# 设置备份服务器
function set_backup_server_with_path()
{
    local backup_server_json=$1
    local backup_server_path=$(grep "backuppath" ${backup_server_json} | head -n1 | awk -F'"' '{print $4}')
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: Start to set the backup server, backup path: ${backup_server_path}"
    if [ ! -f "${backup_server_json}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] ERROR: The backup server parameter file does not exist, json file: ${backup_server_json}"
        return 1
    fi
    
    cd ${OSS_ROOT}/tools/backuprestore && bash setglobalpara.sh -input "${backup_server_json}"
    if [ $? -eq 0 ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: The backup server is set successfully, backup path: ${backup_server_path}"
        return 0
    fi
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] ERROR: Failed to set the backup server, backup path: ${backup_server_path}"
    return 1
}

# 执行备份任务
function start_backup_initial_data()
{
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: Start backing up the initial data."
    # 创建临时目录
    if [ -d "${SCRIPT_PATH}/product" ]
    then
        rm -rf "${SCRIPT_PATH}/product"
    fi
    mkdir -p "${SCRIPT_PATH}/product"
    
    cd ${OSS_ROOT}/tools/resmgr && bash queryproduct.sh -pn all -output "${SCRIPT_PATH}/product" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] ERROR: Failed to export product information."
        return 1
    fi
    
    backup_flag="${SCRIPT_PATH}/backup.flag"
    echo "INIT" >${backup_flag}
    for product in $(ls ${SCRIPT_PATH}/product/product_*json)
    do
        local product_name=$(cat ${product} | sed 's/,/\n/g' | grep '"productname"' | awk -F'"' '{print $4}')
        local product_version=$(cat ${product} | sed 's/,/\n/g' | grep '"product_version"' | awk -F'"' '{print $4}')
        if [ -n "${product_name}" ]
        then
            echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: backup the initial data product_name: ${product_name}"
        fi
        if [ -n "${product_version}" ]
        then
            echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: backup the initial data product_version: ${product_version}"
        fi
        
        if [ -n "${product_name}" -a -z "${product_version}" ]
        then
            (
            cd ${OSS_ROOT}/agent/BackupService/bin && bash backupDynamicData.sh -product ${product_name} || echo "FAILUER" >${backup_flag}
            ) &
        elif [ -n "${product_name}" -a -n "${product_version}" ]
        then
            (
            cd ${OSS_ROOT}/agent/BackupService/bin && bash backupDynamicData.sh -product ${product_name} -remark ${product_version} || echo "FAILUER" >${backup_flag}
            ) &
        else
            echo "FAILUER" >${backup_flag}
            echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] ERROR: Failed to back up initial product data because the product information is not obtained."
        fi
    done
    wait

    grep "INIT" ${backup_flag} &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "backup_initial_data_done" >${backup_flag}
        echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: Initial product data backed up successfully."
        return 0
    fi
    echo "backup_initial_data_faid" >${backup_flag}
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] ERROR: Failed to back up initial product data."
    return 1
}

# 模块执行入口
function fn_main()
{
    # 判断是否备份初始数据
    local backup_server_file="/opt/install/get_backup_server_info.py"
    if [ ! -f "${backup_server_file}" ]
    then
        local backup_server_file="/opt/install/get_backup_server_info.pyc"
    fi
    cp -rf ${backup_server_file} /opt/backup/backuptmp/initdatabackup/
    backup_initial_data=$1
    if [ "${backup_initial_data}" != "id_backup_data_yes" ]
    then
        create_check_file "100"
        echo "[$(date +'%Y-%m-%d %H:%M:%S')] [$$] INFO: Do not backup initial product data."
        return 0
    fi
    
    # 开始任务
    create_check_file "1"
    # 导入环境变量
    source /opt/oss/manager/bin/engr_profile.sh || return 1
    
    # 导出备份服务器信息
    get_backup_server_info || return 1

    # 设置备份服务器:initdatabackup
    local is_update="False"
    set_backup_server_with_path "${BACKUP_TMP_JSON}" || return 1
    
    # 执行备份初始数据
    local is_update="True"
    start_backup_initial_data
    local backup_result=$?

    # 备份路径重新设置回原值
    if [ "${is_update}" == "True" ]
    then
        set_backup_server_with_path "${BACKUP_BAK_JSON}"
    fi
    
    # 记录执行结果
    create_check_file "100"
    return ${backup_result}
}

# 程序执行入口
function fn_ossadm_exec()
{
    id | grep "ossadm" &>/dev/null
    if [ $? -eq 0 ]
    then
        fn_main $*
        return $?
    else
        chmod -R 750 ${SCRIPT_PATH}
        chown -R ossadm:ossgroup ${SCRIPT_PATH}
        su - ossadm -c "cd ${SCRIPT_PATH} && bash ${MODULE_NAME} $*"
        return $?
    fi
}

fn_ossadm_exec $*
if [ $? -eq 0 ]
then
    logger -t ${USER} -p local0.info "${MODULE_NAME}; INFO: Succeeded to backup product initial data."
    exit 0
else
    logger -t ${USER} -p local0.err "${MODULE_NAME}; Error: Failed to backup product initial data."
    exit 1
fi


