#!/bin/bash

function fn_get_log_file_path()
{
    readlink -f "${current_path}/../../cmd.upgrade_local_process.log"
}
function fn_get_all_log_file_path()
{
    readlink -f "${current_path}/../../cmd.upgrade_local_process_all.log"
}

function fn_init()
{
    readonly current_path=$(dirname $(readlink -f $0 ))

    # Loading Modules
    source ${current_path}/../utils/load_all_util.sh
    readonly c_root_path=$(cd ${current_path}/../../ && pwd && cd - &>/dev/null)
    export PYTHONPATH=${c_root_path}/pyscript:${PYTHONPATH}

    # Initialization Log
    fn_sys_set_logging_verbose "$SYS_LOGGING_INFO"
    readonly c_log_file="$(fn_get_log_file_path)"

    # save history log file
    readonly all_log_file="$(fn_get_all_log_file_path)"
    fn_sys_create_log_file "$all_log_file"
    [ -f "${c_log_file}" ]  && cat "${c_log_file}" >> ${all_log_file}

    [[ -f ${c_log_file} ]] && rm -f ${c_log_file}
    fn_sys_create_log_file "$c_log_file"
    fn_sys_set_log_file "$c_log_file"

    source "${c_root_path}/shscript/profile/upgrade_profile.sh"
    # 运营商场景，python会放到此目录
    python_rtsp_path=$(ls ${c_root_path}/python-*.zip)

    if [[ -f ${python_rtsp_path} ]]; then
        rm -rf "${c_root_path}/python" >/dev/null 2>&1;
        unzip ${python_rtsp_path} -d "${c_root_path}/python" >/dev/null 2>&1;
        PYTHONHOME="${c_root_path}/python"
    fi

    [[ ! -f ${g_migrate_conf} ]] && echo '{}' > ${g_migrate_conf}
    return 0
}

function fn_prepare()
{
    fn_sys_log_info "Start to execute preparing."
    [[ ! -d /opt/backup/ncendp ]] && mkdir /opt/backup/ncendp && chown ossuser:ossgroup /opt/backup/ncendp && chmod 750 /opt/backup/ncendp
    fn_kill_useless_ndp_process
    fn_clear_step_flag
    rm -f ${g_namenode_copy_flag}

    [[ -f ${g_process_upgrade_flag} ]] && rm -f ${g_process_upgrade_flag}

    # 生成两个集合 meta_dir_arr:需要压缩的meta目录集合; skip_dir_arr:既不需要视为meta目录,也不视为数据目录,无需备份.
    final_dirs=""
    OLD_IFS="$IFS"
    IFS=","
    meta_dir_arr=(${need_tar_dir})
    for dir_str in ${meta_dir_arr[@]}
    do
        [[ -d ${dir_str} ]] && final_dirs="${final_dirs} ${dir_str}"
    done

    # 不备份的目录。
    skip_final_dirs=""
    skip_dir_arr=(${skip_dir})
    for dir_str in ${skip_dir_arr[@]}
    do
        [[ -d ${dir_str} ]] && skip_final_dirs="${skip_final_dirs} ${dir_str}"
    done
    IFS="${OLD_IFS}"

    fn_sys_log_info "End to execute preparing."
    return 0
}

function fn_check()
{
    fn_sys_log_info "Start to execute checking."

    fn_sys_assert_current_user_matched "root" || return 1
    fn_sys_assert_is_ip "omp_ip" || return 1
    fn_sys_assert_is_file "g_migrate_conf" || return 1
    fn_sys_assert_is_file "g_upgrade_param" || return 1
    fn_sys_assert_is_file "g_upgrade_cluster_id_file" || return 1
    fn_sys_assert_not_null "g_upgrade_cluster_id" || return 1
    fn_sys_assert_is_directory "g_fi_install_path" || return 1

    if [[ -d ${g_upgrade_Dir} ]]; then
        count_link=$(find ${g_upgrade_Dir} -type l | wc -l;)
        [[ ${count_link} -ne 0 ]] && fn_sys_error_info "A soft connection file is detected and exits." && return 1
    fi
    [[ ! -d ${PYTHONHOME} ]] && fn_sys_log_error "Could not find python home." && return 1
    fn_sys_log_info "End to execute checking."
    return 0
}

# 0.0 创建升级的flag。
function fn_process_upgrade_file()
{

    fn_sys_log_info "Start to create upgrade flag."

    [ -f "${g_upgrade_flag}" ] &&  chattr -i ${g_upgrade_flag} && rm -rf ${g_upgrade_Dir}
    [ -f "${g_upgrade_replace_flag}" ] &&  chattr -i ${g_upgrade_replace_flag} && rm -rf ${g_upgrade_replace_flag}
    rm ${g_upgrade_Dir} -rf
    [ $? -ne 0 ] && fn_sys_log_error "Fail to excute <rm -rf ${g_upgrade_Dir}>,please check!" && return 0
    mkdir -p ${g_upgrade_Dir}
    chmod -R 755 /export
    touch "${g_upgrade_flag}"
    touch "${g_upgrade_replace_flag}"
    chown ossuser:ossgroup -R ${g_upgrade_Dir}
    chown ossuser:ossgroup -R ${g_upgrade_replace_flag}
    touch "${g_upgrade_checkpoint}"
    chown ossuser:ossgroup -R ${g_upgrade_checkpoint}

    fn_sys_assert_is_file "g_upgrade_flag" && return 0
    return 1
}

# 2.0 将omm和odaeuser的目录都改成ossuser的。
function fn_process_srv_facl()
{
    fn_sys_log_info "Start to deal with the facl."
    local result_code=0
    local dn_data_dirs=`ls -d /srv/BigData/*`
    [ -z "${dn_data_dirs}" ] && fn_sys_log_info "There is no subdir,Skip" && return 0
    dn_data_dirs=(${dn_data_dirs})

    fn_sys_log_info "Start to backup the facl of /srv/BigData."
    # 声明数组，里面是所有的子目录。
    for dir in "${dn_data_dirs[@]}"; do
        fn_sys_log_info "Start to deal with ${dir}."
        if [[ "${skip_final_dirs[@]}" =~ "${dir}" ]];then
            fn_sys_log_info "It is a skip dir, skip."
            continue
        fi
        if [[ -f ${dir} ]];then
            fn_sys_log_info "It is a file, skip."
            continue
        fi
        if [[ -L ${dir} ]];then
            fn_sys_log_info "It is a file which is soft connection file, skip."
            continue
        fi
        if [[ "/srv/BigData/hadoop" =~ "${dir}" ]]; then
            fn_sys_log_info "It is /srv/BigData/hadoop, skip."
            continue
        fi
        dir_name=$(basename ${dir})
        local l_backup_srv_facl_path="${g_backup_path}/${dir_name}_backup_srv_facl.conf"
        [[ ! -f ${l_backup_srv_facl_path} ]] && getfacl -aR "${dir}" > ${l_backup_srv_facl_path}
        fn_sys_log_info "End to deal with ${dir}."
    done
    fn_sys_log_info "End to backup the facl of /srv/BigData."

    fn_sys_log_info "Start to change the facl of /srv."
    chown ossuser:ossgroup /srv/BigData/ >/dev/null 2>&1
    # 声明数组，里面是所有的子目录。
    for dir in "${dn_data_dirs[@]}"; do
        fn_sys_log_info "Start to deal with ${dir}."
        if [[ "${skip_final_dirs[@]}" =~ "${dir}" ]];then
            fn_sys_log_info "It is a skip dir, skip."
            continue
        fi
        if [[ -f ${dir} ]];then
            fn_sys_log_info "It is a file, skip."
            continue
        fi
        if [[ -L ${dir} ]];then
            fn_sys_log_info "It is a file which is soft connection file, skip."
            continue
        fi
        dir_name=$(basename ${dir})
        if [[ "/srv/BigData/hadoop" == "${dir}" ]]; then
            if [[ -d /srv/BigData/hadoop/data1/dn ]] && [[ ! -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag ]]; then
                find /srv/BigData/hadoop -maxdepth 2 | xargs -i chown ossuser:ossgroup {}
                fn_sys_log_info "The duration of this action depends on the disk I/O status."
                fn_sys_log_info "Start to modify the permission on the ${dir}."
                chown -R ossuser:ossgroup /srv/BigData/hadoop/data1/dn
                touch ${g_backup_path}/${dir_name}_backup_srv_facl.flag
            fi
            chown ossuser:ossgroup /srv/BigData/hadoop
            chown ossuser:ossgroup /srv/BigData/hadoop/data1
            fn_sys_log_info "End to deal with ${dir}."
            continue
        fi
        if [[ "/srv/BigData/hadoop" =~ "${dir}" ]]; then
            if  [[ ! -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag ]]; then
                fn_sys_log_info "The duration of this action depends on the disk I/O status."
                fn_sys_log_info "Start to modify the permission on the ${dir}."
                chown -R ossuser:ossgroup ${dir}
                touch ${g_backup_path}/${dir_name}_backup_srv_facl.flag
            fi
            fn_sys_log_info "End to deal with ${dir}."
            continue
        fi

        fn_sys_log_info "Start looking for file: ${dir_name}_backup_srv_facl.conf."
        [[ ! -f "${g_backup_path}/${dir_name}_backup_srv_facl.conf" ]] && fn_sys_log_error "No backup file found before change, please check." && result_code=1 && continue ;
        fn_sys_log_info "File found successfully: ${dir_name}_backup_srv_facl.conf."

        [[ -f "${g_backup_path}/${dir_name}_backup_srv_facl.flag" ]] && fn_sys_log_info "Success flag for chown is found, skip." && continue
        chown ossuser:ossgroup -R ${dir} >/dev/null 2>&1
        touch ${g_backup_path}/${dir_name}_backup_srv_facl.flag
        fn_sys_log_info "End to deal with ${dir}."
    done

    fn_sys_log_info "Wait for all task backup facl."
    wait
    fn_sys_log_info "All task backup facl finish."
    return ${result_code}
}

function fn_process_hadoop_yarn_facl()
{
  if [[ -d /sys/fs/cgroup/cpu,cpuacct/hadoop-yarn ]];then
        fn_sys_log_info "Start to change owner of hadoop-yarn."
        chown -R ossuser:ossgroup /sys/fs/cgroup/cpu,cpuacct/hadoop-yarn
        [[ $? != 0 ]] && fn_sys_log_error "Can not change owner of hadoop-yarn." && return 1
        chmod 750 /sys/fs/cgroup/cpu,cpuacct/hadoop-yarn
    fi
}

# 3.0 删除omm等无用用户的定时任务
function fn_precess_crontab()
{
    local l_return_value=0
    fn_sys_log_info "Start to backup crontab ..."
    count=$(crontab -l -u omm | wc -l)
    [[ $count -gt 0 ]] && crontab -l -u omm > ${g_backup_path}/crontab_backup_fi.txt
    fn_sys_log_info "End to backup crontab ..."

    fn_sys_log_info "Start to clean crontab before upgrade ..."
    rm -rf /tmp/empty.crontab
    touch /tmp/empty.crontab
    crontab -u omm /tmp/empty.crontab
    rm -rf /tmp/empty.crontab
    fn_sys_log_info "End to clean crontab before upgrade."

    return ${l_return_value};
}

# 4.0 备份原FI的三个配置文件（ldap）
function fn_precess_os_config()
{
    local l_return_value=0
    fn_sys_log_info "Start to backup os config ..."

    [[ -f /tmp/report.json ]] && fn_sys_log_info "Move report.json to backup path." && mv /tmp/report.json ${g_backup_path}/
    [[ ! -f ${g_backup_path}/sssd.conf ]] && fn_sys_log_info "Copy sssd.conf to backup path." && cp /etc/sssd/sssd.conf ${g_backup_path}/
    [[ ! -f ${g_backup_path}/ldap.conf ]] && fn_sys_log_info "Copy ldap.conf to backup path." && cp /etc/openldap/ldap.conf ${g_backup_path}/
    fn_sys_log_info "End to backup os config ..."

    sed -i '/^ossuser/d' /etc/cron.allow
    echo 'ossuser' >> /etc/cron.allow

    return ${l_return_value};
}

# 5.0 备份原FI的三个配置文件（ldap）
function fn_clear_share_config()
{
    local l_return_value=0

    fn_sys_log_info "Start to clear share config ..."
    share_path=`su - ossadm -c 'source /opt/oss/manager/bin/engr_profile.sh && echo ${SHAREPATH}'`
    fn_sys_log_info "share path is ${share_path}"
    rm -rf ${share_path}/*/NdpToolService
    rm -rf ${share_path}/*/NdpStatusService
    rm -rf ${share_path}/*/NdpLdapService
    rm -rf ${share_path}/*/NdpKerberosService
    rm -rf ${share_path}/*/NdpNodeManagerService
    rm -rf ${share_path}/*/NdpDiagnosService
    fn_sys_log_info "End to remove directory."

    return ${l_return_value};
}

deleteDirIfExists()
{
    local dir="$1"

    if [ -z "$dir" ] || [ ! -d "$dir" ]; then
        return 0
    fi

    dir=$(cd "$dir"; pwd)

    [ "$dir" == "/" ] && fn_sys_log_error "Delete dir is /, must be wrong" && exit 1

    fn_sys_log_info "start to delete if exists dir: ${dir}"

    rm -rf "$dir"
    [ -d "$dir" ] && fn_sys_log_error "Fail to delete dir: ${dir}" && exit 1

    fn_sys_log_info "Succeed to delete if exists dir: ${dir}"
    return 0
}
mv_nm_dir()
{
    local nmDir="$1"

    local nmDir_backup="${nmDir}_backup"
    [ ! -d "${nmDir}" ] && fn_sys_log_warn "current dir is not found: ${nmDir}" && return 0
    fn_sys_log_info "current dir is existed , mv ${nmDir} ${nmDir_backup}"

    deleteDirIfExists "${nmDir_backup}"
    mv "${nmDir}" "${nmDir_backup}"

    fn_sys_log_info "Success to backup current part by link of dir: ${nmDir}"
}
# 对传入的目录进行备份，备份逻辑是在对应目录下创建一个backup目录。
backupCurrentPartByLinkOfOneDir()
{
    local dataDir="$1"

    local srcDir="${dataDir}"
    local dstDir="${dataDir}/backup"

    fn_sys_log_info "backup current part by link of dir: ${dataDir}"

    [ ! -d "${srcDir}" ] && fn_sys_log_warn "current dir is not found: ${srcDir}" && return 0

    deleteDirIfExists "${dstDir}"

    diskSpaceInfo=$(df -m "${dataDir}" | grep -vw Filesystem)
    local -i availDiskSpace=$(echo $diskSpaceInfo | awk '{print $4}')

    if [ "$availDiskSpace" -lt "2048" ]; then
        fn_sys_log_error "Disk space is not enough(${dataDir}): need 2048MB, availabe ${availDiskSpace}M" && exit 1
    fi

    mkdir -p "${dstDir}"
    [[ ! -d ${dstDir} ]] && fn_sys_log_error "Create dir failed: "${dstDir}"." && exit 1

    fn_sys_log_info "Start to cp -apl ${srcDir} ${dstDir}"
    ls "${srcDir}" | grep -v "backup" | xargs -i cp -apl "${srcDir}/{}" "${dstDir}"
    if [ $? -ne 0 ]; then
        deleteDirIfExists "${dstDir}"
        fn_sys_log_error "Fail to copy ${srcDir} to ${dstDir}" && exit 1
    fi

    touch "${dstDir}/${g_source_version}_SUCCESS_FLAG"
    fn_sys_log_info "Success to backup current part by link of dir: ${dataDir}"
}

# 1.1  备份FI的srv目录 -- 元数据
function fn_backup_fi_srv()
{
    fn_sys_log_info "Start to backup Metadata stored in the /srv/BigData."
    local l_return_value=0;
    [[ ! -d ${g_backup_path} ]] && mkdir -p "${g_backup_path}"
    [[ -f ${g_backup_path}/${g_fi_backup_file%.*} ]] && fn_sys_log_info "If a backup file already exists, skip." && return 0

    fn_sys_log_info "Exec< tar --ignore-failed-read --atime-preserve -czpPf ${g_backup_path}/${g_fi_backup_file} ${final_dirs} >..."
    tar --ignore-failed-read --atime-preserve -czpPf ${g_backup_path}/${g_fi_backup_file} ${final_dirs} >/dev/null 2>&1;
    l_return_value=$?
    if [ ${l_return_value} -ne 0 -a ${l_return_value} -ne 1 ]; then
        fn_sys_log_warn "Exec< tar --ignore-failed-read --atime-preserve -czpPf ${g_backup_path}/${g_fi_backup_file} ${final_dirs} > Failed.l_return_value=${l_return_value}"
        tar tvf ${g_backup_path}/${g_fi_backup_file}
        if l_return_value=$? && [ ${l_return_value} -ne 0 ]; then
            fn_sys_log_error "Exec< tar tvf ${g_backup_path}/${g_fi_backup_file} > Failed.l_return_value=${l_return_value}"
            return 1;
        fi
    fi
    if [ ! -f "${g_backup_path}/${g_fi_backup_file}" ]; then
        fn_sys_log_error "${g_backup_path}/${g_fi_backup_file} does not exist"
        return 1;
    fi

    rm -f ${g_backup_path}/${g_fi_backup_file%.*}
    mv ${g_backup_path}/${g_fi_backup_file} ${g_backup_path}/${g_fi_backup_file%.*}

    fn_sys_log_info "End to backup Metadata stored in the /srv/BigData."

    # 后续删除。↓
    echo "exit 0" > "${g_upgrade_Dir}/upgrade_inherit.sh"
    chown ossuser:ossgroup "${g_upgrade_Dir}/upgrade_inherit.sh"

    return 0;
}

# 1.11  给微服务准备kafka的备份数据，属主改为ossuser，目录结构变化,已第一份备份的为主。
function fn_prepare_kafka()
{
    fn_sys_log_info "Preparing Kafka Backup Metadata."
    [[ -f ${tar_kafka_meta_back_file} ]] && fn_sys_log_info "The Kafka metadata has been prepared. Skip." && return 0
    readonly src_kafka_meta_back_file=$(ls -t /opt/huawei/Bigdata/LocalBackup/${g_upgrade_cluster_id}/backup_upgrade_*/Kafka*/*Kafka-Metadata*.tar.gz | head -1)
    [[ ! -f ${src_kafka_meta_back_file} ]] && fn_sys_log_info "No Kafka backup data exists. Skip." && return 0

    # 给微服务准备kafka备份数据，对目录名称和属主有要求。
    [[ ! -d ${g_upgrade_Dir}/dynamic_restore ]] && mkdir -p "${g_upgrade_Dir}/dynamic_restore" && fn_sys_log_info "Make directory for change data acl."
    tar -zxvf ${src_kafka_meta_back_file} -C ${g_upgrade_Dir}/dynamic_restore >/dev/null 2>&1;
    chown -R ossuser:ossgroup ${g_upgrade_Dir}/dynamic_restore && fn_sys_log_info "change facl for backup data."
    cd ${g_upgrade_Dir}; tar -zcvf Kafka-Metadata.tar.gz ./dynamic_restore >/dev/null 2>&1;
    chown ossuser:ossgroup ${tar_kafka_meta_back_file}
    [[ -d ${g_upgrade_Dir}/dynamic_restore ]] && cd ${g_upgrade_Dir}; rm -rf ${g_upgrade_Dir}/dynamic_restore
    fn_sys_log_info "Preparing Kafka metadata for backup ends."
}

# 1.2  备份FI的srv目录下的数据目录 -- 非元数据
function fn_backup_datanode()
{
    fn_sys_log_info "Start to backup business data in /srv/BigData."
    local l_return_value=0
    source ${g_upgrade_param}
    if [[ "X${dn_data_dirs}" == "X" ]]; then
        local dn_data_dirs=`ls -d /srv/BigData/*`
    fi
    [ -z "${dn_data_dirs}" ] && fn_sys_log_info "There is no datanode dir,Skip" && return 0
    dn_data_dirs=(${dn_data_dirs})

    # 声明数组，里面是所有的元数据目录。遍历非元数据目录的时候进行跳过即可。
    for dir in "${dn_data_dirs[@]}"; do
        fn_sys_log_info "Start to deal with ${dir}."
        if [ -f "${dir}/backup/${g_source_version}_SUCCESS_FLAG" ]; then
            fn_sys_log_info "Success flag found, skip."
            continue
        fi

        # 这种用法是不是有问题。
        if [[ "${meta_dir_arr[@]}" =~ "${dir}" ]];then
            fn_sys_log_info "It is a dir for meta data, skip."
            continue
        fi

        if [[ "${skip_final_dirs[@]}" =~ "${dir}" ]];then
            fn_sys_log_info "It is a skip dir, skip."
            continue
        fi

        if [[ -L ${dir} ]];then
            fn_sys_log_info "It is a dir which is soft connection file, skip."
            continue
        fi

        if [[ -f ${dir} ]];then
            fn_sys_log_info "It is a file, skip."
            continue
        fi

        if [[ "/srv/BigData/hadoop" == "${dir}" ]]; then
            backupCurrentPartByLinkOfOneDir "${dir}/data1/dn/" &
            mv_nm_dir "${dir}/data1/nm/containerlogs"
            mv_nm_dir "${dir}/data1/nm/localdir"
            fn_sys_log_info "There is a process with dir: ${dir}."
            continue
        fi
        backupCurrentPartByLinkOfOneDir "${dir}" &
        fn_sys_log_info "There is a process with dir: ${dir}."
    done

    fn_sys_log_info "Wait until the hard link task is complete. This task depends on the disk I/O status. If the disk is slow, this step takes a long time."
    wait
    fn_sys_log_info "Start to check whether the backup is complete."

    # 反过来校验一遍，是不是都已经备份过了。
    for dir in "${dn_data_dirs[@]}"; do
        fn_sys_log_info "Start to check ${dir}."
        if [[ "${meta_dir_arr[@]}" =~ "${dir}" ]];then
            continue
        fi

        if [[ "${skip_final_dirs[@]}" =~ "${dir}" ]];then
            continue
        fi
        if [[ -f ${dir} ]];then
            continue
        fi
        if [[ -L ${dir} ]];then
            continue
        fi
        if [[ "/srv/BigData/hadoop" == "${dir}" ]]; then
            if [[ ! -d "${dir}/data1/dn/" ]] ; then
                fn_sys_log_info "No dn subdir found in /srv/BigData/hadoop."
                fn_sys_log_info "End to check ${dir}."
                continue
            fi
            if [ ! -f "${dir}/data1/dn/backup/${g_source_version}_SUCCESS_FLAG" ]; then
                fn_sys_log_error "Fail to find the success flag: ${dir}/data1/dn/backup/${g_source_version}_SUCCESS_FLAG." && return 1
            fi
            fn_sys_log_info "End to check ${dir}."
            continue
        fi
        if [ ! -f "${dir}/backup/${g_source_version}_SUCCESS_FLAG" ]; then
            fn_sys_log_error "Fail to find the success flag: ${dir}/backup/${g_source_version}_SUCCESS_FLAG." && return 1
        fi
        fn_sys_log_info "End to check ${dir}."
    done

    # 清理previous目录
    fn_sys_log_info "Start to clear previous directory in /srv/BigData."
    find /srv/BigData/ -maxdepth 6 -type d -name "previous" | xargs -i rm -rf {}

    fn_sys_log_info "End to backup business data in /srv/BigData."
    return 0
}

# 1.3  kill进程，防止有没停掉的进程。
function fn_kill_fi_process()
{
    local l_return_value=0

    pkill -u omm

    pkill -u ommdba

    ps -efww | grep '^omm' | awk '{print $2}' | xargs -i kill -9 {}

    return ${l_return_value};
}

# 清理zk中的inode数据, 要在备份元数据之后做。
function fn_clear_zookeeper_inode()
{
    local l_return_value=0

    su - omm -c "rm /srv/BigData/zookeeper/version-* -rf"

    return ${l_return_value};
}

# 清理ldap中的inode数据, 要在备份元数据之后做。
function fn_clear_ldap_meta()
{
    su - omm -c "rm -f /srv/BigData/ldapData/ldapserver/init_flag"
    su - omm -c "rm -f /srv/BigData/ldapData/ldapserver/data/*"

    return 0;
}

# 清理redis中的meta数据, 要在备份元数据之后做。
function fn_clear_redis_data()
{
    fn_sys_log_info "Start to clear redis data."
    su - omm -c "rm -rf /srv/BigData/redis"
    su - omm -c "rm -rf /srv/BigData/redis_meta"
    fn_sys_log_info "End to clear redis data."

    return 0;
}

# 6.0 清理kafka中的log数据, 要在备份硬链接数据之后做。
function fn_clear_kafka_sys_consumer_dir()
{
    fn_sys_log_info "Start to clear kafka consumer offset data."
    [[ ! -f ${fi_cluster_json} ]] && fn_sys_log_error "Could not find fi_cluster.json." && return 1
    # 获取kafka的定制目录
    kafka_log_dirs=$(cat ${fi_cluster_json} | grep -A 3 "\"log.dirs\"" | grep "\"value\"" | awk -F "\"" '{print $4}' | grep -v "\.\." | grep "^/srv/BigData")
    kafka_log_dir_list=(${kafka_log_dirs//,/ })
    fn_sys_log_info "The directory collection is: ${kafka_log_dir_list[@]}."
    for dir_str in ${kafka_log_dir_list[@]}
    do
        fn_sys_log_info "Start to clear kafka consumer offset data in ${dir_str}."
        [[ -d ${dir_str} ]] && su - ossuser -c "rm -rf ${dir_str}/__consumer_offsets-*"
        [[ -d ${dir_str} ]] && su - ossuser -c "rm -f ${dir_str}/*checkpoint"
    done

    fn_sys_log_info "End to clear kafka consumer offset data."
    return 0;
}

# 1.4 调用工程框架的能力，将原FusionInsight的一些ID需要继承下来。
function fn_generate_fi_cluster_ini()
{
    local l_return_value=0
    fn_sys_log_info "Start to generate FI cluster_instance.ini"
    [[ ! -d ${PYTHONHOME} ]] && fn_sys_log_error "Could not find python home." && return 1
    local l_tsp_inst_cfg="${g_upgrade_Dir}/nce_inst.cfg"
    local l_generate_cluster_instance_ini_file="${c_root_path}/pyscript/actions/generate_cluster_instance_ini.pyc"
    [ ! -f "${l_generate_cluster_instance_ini_file}" ] && fn_sys_log_error "File \"${l_generate_cluster_instance_ini_file}\" does not exist,please check" && return 1

    # 加入到环境变量中。
    local l_result_conf=${g_upgrade_Dir}/cluster_instance.ini

    ${PYTHONHOME}/bin/python ${l_generate_cluster_instance_ini_file} "${l_tsp_inst_cfg}" ${l_result_conf} ${g_migrate_conf} ${c_log_file} >> ${c_log_file} 2>&1
    l_return_value=$?
    chown ossuser:ossgroup -R ${g_upgrade_Dir}/cluster_instance.ini
    chown ossuser:ossgroup -R ${g_upgrade_Dir}/nce_inst.cfg
    [ ${l_return_value} -ne 0 ] && fn_sys_log_error "Failed to generate FI cluster_instance.ini" && return ${l_return_value}
    fn_sys_log_info "End to generate FI cluster_instance.ini"
    [[ -f ${python_rtsp_path} ]] && rm -rf "${c_root_path}/python" >/dev/null 2>&1;
    return ${l_return_value};
}

# 1.0 备份操作
function fn_process_fi()
{
    # 1.1  备份FI的srv目录 -- 元数据   第一份为准。
    fn_backup_fi_srv || return 1

    # 1.21 清理zk中的inode数据, 要在备份元数据之后做。
    fn_clear_zookeeper_inode || return 1

    # 1.22 清理ldap中的inode数据, 要在备份元数据之后做。
    fn_clear_ldap_meta || return 1

    # 1.23清理redis中的meta数据, 要在备份元数据之后做。
    fn_clear_redis_data || return 1

    # 2.0 调用工程框架的能力，将原FusionInsight的一些ID需要继承下来。
    fn_generate_fi_cluster_ini || return 1

    # 3.0 将omm和odaeuser的目录都改成ossuser的。
    fn_process_srv_facl || return 1

    # 3.1 修改hadoop_yarn 文件权限
    fn_process_hadoop_yarn_facl || return 1

    # 4.0  给微服务准备kafka的备份数据库，属主改为ossuser，目录结构变化,已第一份备份的为主。
    fn_prepare_kafka || return 1

    # 5.0  备份FI的srv目录下的数据目录 -- 非元数据
    fn_backup_datanode || return 1

    # 6.0  kill进程，防止有没停掉的进程。
    fn_kill_fi_process || return 1

    return 0
}

function fn_do()
{
    # 0.0 创建升级的flag。
    fn_process_upgrade_file || return 1

    # 1.0 备份操作
    fn_process_fi || return 1

    # 3.0 删除omm等无用用户的定时任务
    fn_precess_crontab || return 1

    # 4.0 备份原FI的三个配置文件（ldap）
    fn_precess_os_config || return 1

    # 5.0 清理share目录下的配置文件，适配切换加密算法
    fn_clear_share_config || return 1

    # 8.0 清理kafka中的log数据, 要在备份硬链接数据之后以及修改成ossuser属主之后做。
    fn_clear_kafka_sys_consumer_dir || return 1

    return 0
}

function exit_create_flag()
{
    err_msg=$1
    fn_sys_log_error "${err_msg}"
    touch ${g_process_upgrade_flag}
    return 1
}

function fn_main()
{
    c_start_time="$(date +%s)"
    fn_init || exit_create_flag "Start to create the flag file and exit." || return 1
    fn_check || exit_create_flag "Start to create the flag file and exit." || return 1
    fn_prepare || exit_create_flag "Start to create the flag file and exit." || return 1
    fn_do
    local l_result=$?
    touch ${g_process_upgrade_flag}
    local l_elapsed_seconds="$(fn_sys_get_elapsed_time "$c_start_time")"
    fn_sys_log_info "Leave post_install with status [${l_result}]. It takes [${l_elapsed_seconds}] seconds"
    [[ ${l_result} -ne 0 ]] && fn_sys_log_error "Failed to run upgrade_local script."
    return "$l_result"
}

###############################################################################################
fn_main
result="$?"
exit "$result"
