# ! /usr/bin/env python
# coding=utf-8

#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import os.path
import subprocess
import sys

import easyhttputil
from util import common

if hasattr(common, 'getDefaultLogger'):
    LOGGER = common.getDefaultLogger()
elif hasattr(common, 'get_default_logger'):
    LOGGER = common.get_default_logger()
else:
    import logging
    LOGGER = logging.getLogger("default")


def _get_dbsvc_adm() -> str:
    if hasattr(common, 'getDBAgentPath'):
        return common.getDBAgentPath('bin/dbsvc_adm')
    if hasattr(common, 'get_dbagent_path'):
        return common.get_dbagent_path('bin/dbsvc_adm')
    LOGGER.error('get_dbagent_path and getDBAgentPath not found in common')
    return ''


def _load_json_file(file_path: str) -> dict:
    if hasattr(common, 'loadJsonFile'):
        return common.loadJsonFile(file_path)
    if hasattr(common, 'load_json_file'):
        return common.load_json_file(file_path)
    LOGGER.error('load_json_file and loadJsonFile not found in common')
    return {}


def _load_json_from_string(content: str) -> dict:
    if hasattr(common, 'loadJsonFromString'):
        return common.loadJsonFromString(content)
    if hasattr(common, 'load_json_from_string'):
        return common.load_json_from_string(content)
    LOGGER.error('loadJsonFromString and load_json_from_string not found in common')
    return {}


def _get_tenant_name(plan_data_file: str) -> str:
    file_content = _load_json_file(plan_data_file)
    if not file_content:
        return ""
    return file_content.get("productname", "")


def _get_unused_insts_dbs(dbs: dict, curdbs: dict) -> (list, dict):
    unused_insts = [instid for instid in curdbs if instid not in dbs]
    unused_inst_ids = [curdbs[inst].get('dbInstanceID') for inst in unused_insts]
    unused_inst_ids = list(set(unused_inst_ids))
    unused_db_maps = _get_unused_db_maps(dbs, curdbs)
    return unused_inst_ids, unused_db_maps


def _get_unused_db_maps(dbs: dict, curdbs: dict) -> dict:
    dbmaps = {}
    for instid in [instid for instid in curdbs if instid in dbs]:
        realinstid = curdbs[instid].get('dbInstanceID')
        for dbname in curdbs[instid].get('dbList', {}):
            if dbname in dbs[instid].get('dbList', {}):
                continue
            if realinstid not in dbmaps:
                dbmaps[realinstid] = []
            dbmaps[realinstid].append(dbname)
    return dbmaps


def _download_containerlist(tenant: str) -> str:
    url = '/rest/plat/deploy-proxy/v1/containerlist/action?action-id=export-containerlist'
    data = {'tenant': tenant, 'userList': ['adminUser', 'dbUser', 'readUser', 'rplUser']}
    if hasattr(easyhttputil, 'httppostWithRetry'):
        response = easyhttputil.httppostWithRetry(url, data, printlog=False)
    elif hasattr(easyhttputil, 'http_post_with_retry'):
        response = easyhttputil.http_post_with_retry(url, data, printlog=False)
    else:
        LOGGER.error('httppostWithRetry and http_post_with_retry not found in easyhttputil')
        response = None
    if response and len(response) > 8:
        return response
    return ''


def _delete_instance(instance_ids: list) -> bool:
    if not instance_ids:
        LOGGER.info("delete instance is empty")
        return True
    dbsvc_adm = _get_dbsvc_adm()
    if not dbsvc_adm:
        LOGGER.info(f'get dbsvc_adm file failed')
        return False
    for instance_id in instance_ids:
        LOGGER.info(f'delete instance {instance_id} begin')
        cmd = f"bash {dbsvc_adm} -cmd delete-db-instance -instid {instance_id}"
        LOGGER.info(f'execute cmd: {cmd}')
        code, output = subprocess.getstatusoutput(cmd)
        if code != 0:
            LOGGER.error(f"delete instance {instance_id} failed, output={output}")
            return False
        LOGGER.info(f'delete instance {instance_id} success')
    return True


def _delete_db(db_map: dict) -> bool:
    if not db_map:
        LOGGER.info("delete db map is empty")
        return True
    dbsvc_adm = _get_dbsvc_adm()
    if not dbsvc_adm:
        LOGGER.info(f'get dbsvc_adm file failed')
        return False
    for instance_id, dbs in db_map.items():
        if not dbs:
            continue
        LOGGER.info(f'delete instance {instance_id} dbs {dbs} begin')
        cmd = f"bash {dbsvc_adm} -cmd remove-database -instid {instance_id} -dbnames {','.join(dbs)}"
        LOGGER.info(f'execute cmd: {cmd}')
        code, output = subprocess.getstatusoutput(cmd)
        if code != 0:
            LOGGER.error(f"delete instance {instance_id} dbs {dbs} failed, output={output}")
            return False
        LOGGER.info(f'delete instance {instance_id} dbs {dbs} success')
    return True


def main(argv) -> int:
    if len(argv) < 3:
        LOGGER.error(f"invalid argv {argv}")
        return 1

    work_path = argv[1]
    tenant_name = _get_tenant_name(os.path.join(work_path, 'plandata.json'))
    if not tenant_name:
        LOGGER.error(f"get productname from {os.path.join(work_path, 'plandata.json')} failed")
        return 1

    LOGGER.info(f"delete diff db instance for tenant {tenant_name}")
    old_container_list = _load_json_file(argv[2])
    if not old_container_list:
        LOGGER.error(f"load old container list content from {argv[2]} failed")
        return 1

    container = _download_containerlist(tenant_name)
    new_container_list = _load_json_from_string(container)
    if not new_container_list:
        LOGGER.error(f"load new container list content from {container} failed")
        return 1

    unused_insts, unused_db_maps = _get_unused_insts_dbs(old_container_list, new_container_list)
    LOGGER.info(f"find unused instances {unused_insts}")
    LOGGER.info(f"find unused dbs {unused_db_maps}")

    if not _delete_instance(unused_insts):
        LOGGER.error("delete unused instance failed")
        return 1

    if not _delete_db(unused_db_maps):
        LOGGER.error("delete unused db failed")
        return 1

    LOGGER.info('delete unused instance and db success')
    return 0


if __name__ == "__main__":
    sys.exit(main(sys.argv))
