# -*- coding:utf-8 -*-
"""检查操作系统分区"""

import os
import logging
import json
import time
from utils.log_util import EasysuiteLogger
from utils.sshUtil import CsshCmd
from utils.thread_util import MultiThreadPool
from func.upgrade.common.upgrade_local_ssh import UpgradeLocalSsh
from func.upgrade.upgrade_operation_mgr.common.upgrade_util import UpgradeUtil

local_logger = logging.getLogger(__name__)
# sudo包位置
sudo_pkg_path = os.path.split(os.path.realpath(__file__))[0]
# sudo包名
sudo_pkg_name = 'sudobin_pkg.tar'
# 执行脚本
sudo_script = "check_os_partition.sh"
# 上传路径
upload_path = "/tmp"
# sudo包临时解压目录
tmp_path = f"/opt/check_os_partition_{time.time()}"
# sudo脚本执行结果存放位置
os_partition_check_result = f"{upload_path}/os_partition_check_result"
# ssh命令参数
ssh_para = "-o ConnectTimeout=30 -o stricthostkeychecking=no -o ConnectionAttempts=5 -o " \
           "ServerAliveInterval=3 -o ServerAliveCountMax=20"


def execute_cmd(cmd, ssh_node):
    """
    功能描述: 执行shell命令，返回结果值
    :return: 执行结果
    """
    if UpgradeLocalSsh.is_es_node(ssh_node.get("login_ip")):
        return UpgradeLocalSsh.send_admin_local_cmd(ssh_node.get("su_pwd"), cmd,
                                                    node_ip=ssh_node.get("login_ip"))
    return CsshCmd.exec_shell_cmd_su_exec(ssh_node.get("login_ip"), ssh_node.get("login_user"),
                                          ssh_node.get("login_pwd"), cmd, ssh_node.get("su_user"),
                                          ssh_node.get("su_pwd"), proxy_param=None)


def execute_task_check(kvs):
    """
    功能描述: 执行检查任务
    :return: 执行结果
    """
    try:
        # 登录信息
        ssh_nodes_param = get_node_connect_info(kvs)
        logger = EasysuiteLogger.get_logger(kvs, "task_check_os_partition")
        logger.easysuite_start("begin to check")
        # 获取sudobin安装位置，普通环境/usr/local/uniepsudobin，企业环境/opt/sudobin
        sudobin_home_path = get_sudobin_file(ssh_nodes_param,logger)
        execute_thread_pool = MultiThreadPool()
        upload_thread_pool = MultiThreadPool()
        # 上传提权包到管理节点
        logger.easysuite_info("Start to upload the package to management node")
        upload_result = upload_sudo_pkg_to_manager(ssh_nodes_param)
        if not upload_result:
            logger.easysuite_error(f"Upload the package failed, please check the environment.")
            return False
        logger.easysuite_info("Upload completed")
        # 执行检查
        for site, ssh_node in ssh_nodes_param.items():
            if not ssh_node.get("login_ip"):
                continue
            result, node_ip_list = get_ip_list(site, ssh_node)
            if not result:
                logger.easysuite_error("get remote ips failed.")
                return False
            for ip in node_ip_list:
                upload_thread_pool.add_thread(upload_sudo_pkg_to_node, kvs, ssh_node, ip, logger)
                execute_thread_pool.add_thread(execute_check_sshd_config, kvs, ssh_node, logger, ip,sudobin_home_path)

        # 传包
        logger.easysuite_info("Start to upload the package to other node")
        upload_thread_pool.execute_with_result()
        # 处理检查结果信息
        logger.easysuite_info("Start to check")
        results = execute_thread_pool.execute_with_result()
        if all(results):
            logger.easysuite_finish("The os partition on all nodes is checked successfully.")
            return True
        logger.easysuite_error("The os partition is checked failed.")
        return False
    except Exception as e:
        logger.easysuite_error("Execute check task exception.{}".format(e))
        return False

def get_sudobin_file(ssh_nodes_param,logger):
    ssh_node=ssh_nodes_param.get("master")
    # 获取sudobin安装位置，普通环境/usr/local/uniepsudobin，企业环境/opt/sudobin
    get_sudobin_file_cmd= f"su - ossadm -c \"[ -f /opt/oss/manager/bin/engr_profile.sh ] && . /opt/oss/manager/bin/engr_profile.sh ; if [ -z \${{SUPER_ROOT}} ];then  echo '/usr/local/uniepsudobin' ; else echo \${{SUPER_ROOT}} ; fi\" 2>>/dev/null"
    result, output = execute_cmd(get_sudobin_file_cmd, ssh_node)
    if not result:
        logger.easysuite_info("get sudobin path failed")
        return False
    logger.easysuite_info("get sudobin path success")
    return output.replace("\n", "").replace("\r", "")

def execute_check_sshd_config(kvs, ssh_node, logger, ip, sudobin_home_path):
    """
    在节点上执行检查
    :return:
    """
    if not ip:
        return True
    cmd = f"sudo {sudobin_home_path}/execute.sh {upload_path}/{sudo_pkg_name} {tmp_path} {sudo_script}"
    if ip == UpgradeUtil.get_value_from_main(kvs.get("easysuite.task_id"), "arbitrate_ip"):
        exe_check_cmd = f"su - ossadm -c \"{cmd}\" 2>>/dev/null"
        get_check_result_cmd = f"su - ossadm -c \"cat {os_partition_check_result} 2>>/dev/null\""
    else:
        exe_check_cmd = f"su - ossadm -c \"ssh {ssh_para} {ip} {cmd}\" 2>>/dev/null"
        get_check_result_cmd = f"su - ossadm -c \"ssh {ssh_para} {ip} cat {os_partition_check_result} 2>>/dev/null\""
    # 执行检查
    check_result, stdout = execute_cmd(exe_check_cmd, ssh_node)
    if not check_result:
        local_logger.error(f"execute cmd:{exe_check_cmd} failed; stdout:{stdout}")
        logger.easysuite_error(f"Failed to execute the sudo script on node {ip}.")
        return False
    # 获取检查结果
    result, message = execute_cmd(get_check_result_cmd, ssh_node)
    # 打印日志时避免主备的127.0.0.1混淆
    if ip == "127.0.0.1":
        ip = ssh_node.get("login_ip")
    if not result or "check" not in message:
        local_logger.info(f"execute 'cat {os_partition_check_result}':{message}")
        logger.easysuite_error(f"Failed to obtain the check result on {ip}")
        return False
    if "check failed" in message:
        logger.easysuite_error(f"The os partition on node {ip} is checked failed.")
        logger.easysuite_error("check result:" + message)
        return False
    logger.easysuite_info(f"The os partition on node {ip} is checked successfully.")
    return True


def upload_sudo_pkg_to_node(kvs, ssh_node, ip, logger):
    """
    将提权包上传到其他节点上
    :return:
    """
    # ip为空跳过
    if not ip:
        return True
    # 仲裁节点跳过
    if ip == UpgradeUtil.get_value_from_main(kvs.get("easysuite.task_id"), "arbitrate_ip"):
        return True
    upload_cmd = f"su - ossadm -c \"scp {ssh_para} {upload_path}/{sudo_pkg_name} [{ip}]:{upload_path} \" 2>>/dev/null"
    result, output = execute_cmd(upload_cmd, ssh_node)
    if result:
        local_logger.info(f"upload sudo package to {ip} success")
        return True
    local_logger.error(f"upload sudo package to {ip} failed, output: {output}")
    return False


def upload_sudo_pkg_to_manager(ssh_nodes_param):
    """
    将提权包上传到管理节点上
    :return:
    """
    for site, ssh_node in ssh_nodes_param.items():
        if not ssh_node.get("login_ip"):
            continue
        clean_cmd = f"rm -rf {upload_path}/{sudo_pkg_name}"
        execute_cmd(clean_cmd, ssh_node)
        if UpgradeLocalSsh.is_es_node(ssh_node.get("login_ip")):
            cmd = f"cp -arf {sudo_pkg_path}/{sudo_pkg_name} {upload_path}"
            result, output = execute_cmd(cmd, ssh_node)
        else:
            result = CsshCmd.sftp_put(ssh_node.get("login_ip"), ssh_node.get("login_user"),
                                      ssh_node.get("login_pwd"),
                                      sudo_pkg_path, upload_path, sudo_pkg_name, connect_port=ssh_node.get("sftp_port"))
        if not result:
            local_logger.error(f"upload sudo package to {ssh_node.get('login_ip')} failed")
            return False
        # 修改提权包权限
        modify_cmd = f"chown ossadm:ossgroup {upload_path}/{sudo_pkg_name}"
        modify_result, output = execute_cmd(modify_cmd, ssh_node)
        if not modify_result:
            local_logger.error("Failed to modify the package permission.")
            return False
    return True


def get_ip_list(site, ssh_node):
    if site == "arbit":
        return True, [ssh_node.get("login_ip")]
    if UpgradeLocalSsh.is_es_node(ssh_node.get("login_ip")):
        return True, _get_all_node_ips()
    result, out_put = execute_cmd(_get_remote_ips_cmd(), ssh_node)
    return result, out_put.strip().split(',')


def _get_all_node_ips():
    """
    本站点执行获取节点ip列表的命令
    :return:
    """
    node_ip_list = []
    with open(f'/opt/oss/manager/etc/sysconf/nodelists.json', mode='r') as node_info_file:
        data = json.load(node_info_file)
        for k, v in data.get('nodeList').items():
            for item in v.get('IPAddresses'):
                if 'maintenance' in item.get('usage'):
                    node_ip_list.append(item.get('IP'))
    return node_ip_list


def _get_remote_ips_cmd():
    """
    远端站点执行获取节点ip列表的命令
    :return:
    """
    return f'''su - ossadm -c ". /opt/oss/manager/bin/engr_profile.sh;python -c \\"
import json
ips = ''
with open('/opt/oss/manager/etc/sysconf/nodelists.json', mode='r') as node_info_file:
  data = json.load(node_info_file)
  for k, v in data.get('nodeList').items():
    for item in v.get('IPAddresses'):
      if 'maintenance' in item.get('usage'):
       ips += item.get('IP') + ',' 
  print(ips)
\\""'''


def get_node(kvs, node_name, is_arbit=None):
    """
    封装节点信息
    """
    if is_arbit:
        login_ip = UpgradeUtil.get_value_from_main(kvs.get("easysuite.task_id"), "arbitrate_ip")
        login_user = kvs.get("{}_login_user".format(node_name))
        login_pwd = kvs.get("{}_login_pwd".format(node_name))
    else:
        login_ip = kvs.get('{}_ip'.format(node_name))
        login_user = kvs.get("standby_login_user")
        login_pwd = kvs.get('{}_{}_pwd'.format(node_name, login_user))
    return {
        'login_ip': login_ip,
        'login_user': login_user,
        'login_pwd': login_pwd,
        'su_user': "root",
        'su_pwd': kvs.get('{}_root_pwd'.format(node_name)),
        'ssh_port': kvs.get('{}_ssh_port'.format(node_name)),
        'sftp_port': kvs.get('{}_sftp_port'.format(node_name)),
        'node_name': node_name
    }


def get_node_connect_info(kvs):
    """
    获取节点登录信息
    """
    ssh_node_dict = {}
    # 非单管场景
    if kvs.get("single_mgr_domain") == "no":
        master_node = get_node(kvs, "node_omp01")
        ssh_node_dict.update({"master": master_node})
        # 异地容灾场景
        if kvs.get("protection") == 'id_protection_hot':
            standby_node = get_node(kvs, "node_standby_omp01")
            ssh_node_dict.update({"standby": standby_node})
    # 单管场景
    else:
        master_node = get_node(kvs, "node_nmsserver")
        ssh_node_dict.update({"master": master_node})
        # 异地容灾场景
        if kvs.get("protection") == 'id_protection_hot':
            standby_node = get_node(kvs, "node_standby_nmsserver")
            ssh_node_dict.update({"standby": standby_node})
    if kvs.get("is_arbitration_exist") == "yes":
        arbit_node = get_node(kvs, "node_arbitrate", True)
        ssh_node_dict.update({"arbit": arbit_node})
    return ssh_node_dict


def check(kvs):
    """
    功能描述：判断此任务是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    return True


def exe(kvs):
    return [[('', '', execute_task_check(kvs))]]
