# -*- coding:utf-8 -*-
"""检查SUSE系统的nscd组件版本"""

import logging
from utils.log_util import EasysuiteLogger
from utils.sshUtil import CsshCmd
from func.upgrade.common.upgrade_local_ssh import UpgradeLocalSsh


local_logger = logging.getLogger(__name__)


def execute_cmd(cmd, ip, ssh_node):
    """
    功能描述: 执行shell命令，返回结果值
    :return: 执行结果
    """
    if UpgradeLocalSsh.is_es_node(ip):
        return UpgradeLocalSsh.send_admin_local_cmd(ssh_node.get("su_pawd"), cmd,
                                                    node_ip=ip)
    return CsshCmd.exec_shell_cmd_su_root(ip, ssh_node.get("login_user"),
                                          ssh_node.get("login_pawd"), cmd, ssh_node.get("su_pawd"),
                                          proxy_param=None)


def execute_task_check(kvs):
    """
    功能描述: 执行检查任务
    :return: 执行结果
    """
    # 登录信息
    ssh_nodes_param = get_node_connect_info(kvs)
    logger = EasysuiteLogger.get_logger(kvs, "task_check_suse_nscd_version")
    logger.easysuite_start("start to check nscd version")

    # 执行任务
    for site, ssh_node in ssh_nodes_param.items():
        ip = ssh_node['login_ip']
        if not ip:
            continue
        local_logger.info("check nscd in {}".format(ip))

        get_nscd_version_cmd = "rpm -q nscd"
        _, stdout = execute_cmd(get_nscd_version_cmd, ssh_node['login_ip'],
                                              ssh_node)
        local_logger.info("nscd version: {}".format(stdout))
        nscd_version = stdout.split("nscd-")[1].split(".x86_64")[0]
        local_logger.info("version num: {}".format(nscd_version))

        check_suse_version_cmd = "grep PATCHLEVEL /etc/SuSE-release"
        _, suse_version = execute_cmd(check_suse_version_cmd, ssh_node['login_ip'],
                                              ssh_node)
        local_logger.info("suse version: {}".format(suse_version))
        suse_version = suse_version.split("=")[1].strip()
        local_logger.info("version:{}".format(suse_version))
        if "4" in suse_version:
            os_version = "SUSE 12SP4"
            target_version = "2.22-114.8.3"
        elif "5" in suse_version:
            os_version = "SUSE 12SP5"
            target_version = "2.22-109.2"
        else:
            return True

        error_info = "The version:{} of the nscd component of the {} system is " \
                     "lower than {}. During the upgrade process, there is a low " \
                     "probability that a known bug in nscd will be triggered, " \
                     "which will cause nscd to deny service and cause the upgrade " \
                     "to fail. It is recommended to upgrade nscd to: {} or above, " \
                     "please contact the SUSE vendor for support.".format(
            nscd_version, os_version, target_version, target_version)

        success_info = "The version:{} of the nscd component in {} system " \
                       "meet the requirements.".format(nscd_version, os_version)

        if nscd_version < target_version:
            logger.easysuite_error(error_info)
            return False
        else:
            logger.easysuite_finish(success_info)
            return True


def get_node_connect_info(kvs):
    """
    获取管理节点登录信息
    """
    ssh_node_dict = {}
    login_user = kvs.get("standby_login_user")
    su_user = "root"
    # 非单管场景
    if kvs.get("single_mgr_domain") == "no":
        master_node = {
            'login_ip': kvs.get('node_omp01_ip'),
            'login_user': login_user,
            'login_pawd': kvs.get('node_omp01_%s_pwd' % login_user),
            'su_user': su_user,
            'su_pawd': kvs.get('node_omp01_%s_pwd' % su_user),
            'ssh_port': kvs.get('node_omp01_ssh_port'),
            'sftp_port': kvs.get('node_omp01_sftp_port')
        }
        ssh_node_dict.update({"node": master_node})
        # 异地容灾场景
        if kvs.get("protection") == 'id_protection_hot':
            standby_node = {
                'login_ip': kvs.get('node_standby_omp01_ip'),
                'login_user': login_user,
                'login_pawd': kvs.get('node_standby_omp01_%s_pwd' % login_user),
                'su_user': su_user,
                'su_pawd': kvs.get('node_standby_omp01_%s_pwd' % su_user),
                'ssh_port': kvs.get('node_standby_omp01_ssh_port'),
                'sftp_port': kvs.get('node_standby_omp01_sftp_port')
            }
            ssh_node_dict.pop("node")
            ssh_node_dict.update({"master": master_node})
            ssh_node_dict.update({"standby": standby_node})
    # 单管场景
    else:
        master_node = {
            'login_ip': kvs.get('node_nmsserver_ip'),
            'login_user': login_user,
            'login_pawd': kvs.get('node_nmsserver_%s_pwd' % login_user),
            'su_user': su_user,
            'su_pawd': kvs.get('node_nmsserver_%s_pwd' % su_user),
            'ssh_port': kvs.get('node_nmsserver_ssh_port'),
            'sftp_port': kvs.get('node_nmsserver_sftp_port')
        }
        ssh_node_dict.update({"node": master_node})
        # 异地容灾场景
        if kvs.get("protection") == 'id_protection_hot':
            standby_node = {
                'login_ip': kvs.get('node_standby_nmsserver_ip'),
                'login_user': login_user,
                'login_pawd': kvs.get('node_standby_nmsserver_%s_pwd' % login_user),
                'su_user': su_user,
                'su_pawd': kvs.get('node_standby_nmsserver_%s_pwd' % su_user),
                'ssh_port': kvs.get('node_standby_nmsserver_ssh_port'),
                'sftp_port': kvs.get('node_standby_nmsserver_sftp_port')
            }
            ssh_node_dict.pop("node")
            ssh_node_dict.update({"master": master_node})
            ssh_node_dict.update({"standby": standby_node})
    return ssh_node_dict


def get_os_type(kvs):
    """
    功能描述：判断此任务是否在操作系统上是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    ssh_nodes_param = get_node_connect_info(kvs)
    for site, ssh_node in ssh_nodes_param.items():
        # 某个节点来判断系统类型
        ip = ssh_node['login_ip']
        if not ip:
            continue
        local_logger.info("check os type in {}".format(ip))

        check_suse_cmd = "[ -f /etc/SuSE-release ] && echo suse"
        _, stdout = execute_cmd(check_suse_cmd, ssh_node['login_ip'],
                                              ssh_node)
        local_logger.info("check suse:{}".format(stdout))
        if not "suse" in stdout:
            return "NO-INVOlVED-OS"
        check_suse_version_cmd = "grep PATCHLEVEL /etc/SuSE-release"
        _, stdout = execute_cmd(check_suse_version_cmd, ssh_node['login_ip'],
                                              ssh_node)
        local_logger.info("check suse version:{}".format(stdout))
        suse_version = stdout.split("=")[1].strip()
        if suse_version.find("4") or suse_version.find("5"):
            return "suse-12sp{}".format(suse_version)
        else:
            return "NO-INVOLVED-SUSE-VERSION"


def check(kvs):
    """
    功能描述：判断此任务是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    os_type = get_os_type(kvs)
    local_logger.info("success to get os: {}".format(os_type))
    if not os_type:
        return False
    return "suse" in os_type


def exe(kvs):
    ret = list()
    result = execute_task_check(kvs)
    if result:
        ret.append([('', '', True)])
    else:
        ret.append([('', '', False)])
    return ret