#!/bin/bash
LOG_FILE="/var/log/complete_match_block.log"
[ -f ${LOG_FILE} ] && rm -f ${LOG_FILE}
# 日志函数
function print_log()
{
    echo -e "$(date +%Y-%m-%d) $(date +%H:%M:%S) | ${1} | ${2}" >> ${LOG_FILE}
}

# 打印脚本正确调用方法
function print_usage()
{
    print_log "ERROR" "Parameter error, the correct usage is : complete_match_block.sh query/add_match_all"
    return 1
}

# 参数校验
function check_param()
{
    local usage=$1
    # 校验参数用途
    local usage_list=("query" "add_match_all")
    echo "${usage_list[@]}" | grep -qw "${usage}" || print_usage
}

# 检查配置，确定是否需要处理
function query_match_config()
{
    # 若环境上不存在Match User dbuser配置，则无需处理
    grep -q "Match User dbuser" "${CONFIG_FILE}" || return 0

    # 若环境上已有Match All # For dbuser配置，则无需重复处理
    grep -q "Match All # For dbuser" "${CONFIG_FILE}" && return 0

    print_log "INFO" "The match block is not closed."
    return 1
}

# sshd -t检查sshd配置是否正确
function check_sshd_config()
{
    local res=$(sshd -t -f "$1" 2>&1)
    if [ $? -ne 0 ]
    then
        print_log "ERROR" "The sshd configuration file is incorrect. Error message:\n${res}"
        return 1
    fi
}

# 添加Match all配置
function add_match_all()
{
    check_sshd_config "${CONFIG_FILE}" || return 1

    print_log "INFO" "Start to add Match all configuration."
    print_log "INFO" "SSH config file:"
    cat "${CONFIG_FILE}" >> ${LOG_FILE}

    cp -f "${CONFIG_FILE}" "${CONFIG_FILE_BAK}"
    local match_dbuser_line_num=$(grep -n "Match User dbuser" ${CONFIG_FILE_BAK} | tail -1 | awk -F':' '{print $1}')
    # 获取从Match User dbuser开始到文件结尾的所有内容
    local match_dbuser_text=$(cat -n ${CONFIG_FILE_BAK} | tail -n +"${match_dbuser_line_num}")
    # 查询Match User dbuser之后的第一个Match all
    local match_all_line_num=$(echo "${match_dbuser_text}" | grep "Match All" | head -1 | awk '{print $1}')

    # 在原有的Match all前插入Match all，若原先没有Match all，则直接追加到文件末尾
    if [ -n "${match_all_line_num}" ]
    then
        sed -i "${match_all_line_num}i Match All # For dbuser" "${CONFIG_FILE_BAK}"
    else
        echo "Match All # For dbuser" >> "${CONFIG_FILE_BAK}"
    fi

    if [ $? -eq 0 ]
    then
        print_log "INFO" "Success to add Match all configuration."
    else
        print_log "ERROR" "Failed to add Match all configuration."
        return 1
    fi

    check_sshd_config "${CONFIG_FILE_BAK}" || return 1

    for _ in $(seq 3)
    do
        mv -f "${CONFIG_FILE_BAK}" "${CONFIG_FILE}"
        return_code=$?
        [ "${return_code}" -eq 0 ] && break
        sleep 5
    done
    if [ "${return_code}" -ne 0 ]
    then
        print_log "ERROR" "Unable to overwrite sshd_config."
        return 1
    fi
}

function main()
{
    local usage=$1
    check_param "$@" || return 1

    CONFIG_FILE="/etc/ssh/sshd_config"
    CONFIG_FILE_BAK="/etc/ssh/sshd_config_$(date +%Y%m%d%H%M%S)"

    case ${usage} in
        "query")
            query_match_config
            return $?
        ;;
        "add_match_all")
            query_match_config && return 0
            add_match_all
            return $?
        ;;
    esac
}

main "$@"
exit $?