#!/bin/bash

SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
TMP_PATH="/opt/oss/log/manager/easysuite_upgrade_config"


##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    if [ "${user}" != "ossadm" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##################################################
# 日志记录进/opt/oss/log/manager/easysuite_upgrade/scriptlo/${SELF_FILE}.log
##################################################
function LOG()
{
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]
    then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date +%Y)-$(/bin/date +%m)-$(/bin/date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2" | tee -a ${LOG_FILE}
}

# 入参校验
function check_param()
{
    local check_tmp=$1
    [ -z "${check_tmp}" ] && return 1
    echo "${check_tmp}" | fgrep -q "../"
    if [ $? -eq 0 ]
    then
        LOG "ERROR" "The input param \"${check_tmp}\" is invalid."
        return 1
    fi
    if [ ! -d "${check_tmp}" ]
    then
        LOG "ERROR" "The input param \"${check_tmp}\" is invalid."
        return 1
    fi
}

# 获取输入参数
function init_params()
{
    local num=$#
    if [ $((${num}%2)) -eq 1 ]
    then
        num=$((${num}/2+1))
    else
        num=$((${num}/2))
    fi
    local count=1
    for((i=1;i<=${num};i++))
    do
        [[ ${1#-} == "input" ]] && { INPUTPATH=$2;shift 2;check_param "${INPUTPATH}" || return 1; continue; }
    done
}

function check_input()
{
    init_params "$@"
    if [ $? -ne 0 ]
    then
        echo "Example: bash ${SELF_FILE} -input /opt/upgrade/easysuite_upgrade/workpath/V100R019C00SPC602-V100R020C00/workpath-NCE/"
        return 1
    fi
}

# 查询当前产品节点列表
function get_node_list() {
    PRODUCTNAME=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['productname'])" 2>/dev/null)
    if [ -z "${PRODUCTNAME}" ]
    then
        LOG "ERROR" "Failed to obtain product information."
        return 1
    fi
    NODE_FILE="${INPUTPATH}/nodes_${PRODUCTNAME}.json"
    NODELIST=$(python 2>/dev/null<< PEND
import json
path="${NODE_FILE}"
try:
    with open(path, 'r') as file:
        data=json.load(file)
    hostlist=data.get('hostlist',[])
    node_list=list()
    for host in hostlist:
        node_list.append(host.get("nodemgrip",""))
    print(" ".join(node_list).strip())
except Exception as e_msg:
    print("")
PEND
)
    LOG "INFO" "NODELIST:${NODELIST}"
    if [ -z "${NODELIST}" ]
    then
        LOG "ERROR" "Failed to obtain product nodelist."
        return 1
    fi
}

function fresh_one_node() {
    local node_ip=$1
    LOG "INFO" "Start to fresh ${node_ip}."
    SSH_CMD="ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o ConnectionAttempts=3 -o ServerAliveInterval=10 ${node_ip} "
    local result=$(${SSH_CMD} "bash ${OSS_ROOT}/agent/bin/ipmc_adm -cmd statusapp -tenant ${PRODUCTNAME} >/dev/null 2>&1;echo SSHD_CONNECT_FLAG")
    echo ${result} | grep -qw 'SSHD_CONNECT_FLAG'
    if [ $? -ne 0 ]
    then
        LOG "INFO" "Connection ${node_ip} failed."
        return 1
    fi
    LOG "INFO" "Finished to fresh ${node_ip}."
}

# 检查节点状态
function fresh_node_status()
{
    LOG "INFO" "Start to fresh node status."
    local fail_flag="${TMP_PATH}/${SELF_FILE//.sh/}_$(date +%Y%m%d%H%M%S)"
    for ip in ${NODELIST}
    do
        {
            if [ ! -f "${fail_flag}" ]
            then
                fresh_one_node "${ip}"
                [ $? -ne 0 ] && touch "${fail_flag}"
            fi
        }&
    done
    wait
    if [ -f "${fail_flag}" ]
    then
        LOG "ERROR" "Failed to fresh node status."
        rm -f "${fail_flag}"
        return 1
    fi
    LOG "INFO" "Finished to fresh node status."
}

function main()
{
    check_user || return 1
    # 创建日志目录
    [ ! -d "${LOG_PATH}" ] && mkdir -p ${LOG_PATH}
    # 创建临时目录
    [ ! -d "${TMP_PATH}" ] && mkdir -p ${TMP_PATH}
    source /opt/oss/manager/bin/engr_profile.sh
    check_input "$@" || return 1
    get_node_list || return
    fresh_node_status || return 1
}

main "$@"
exit $?