# encoding=utf-8
"""
功 能：记录日志
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
"""
import inspect
import json
import os
import shlex
import subprocess
import sys
import threading
import time
import copy
from datetime import datetime

from commonlog import Logger
from deployment import nodelist
from taskmgr_util import Taskmgrutil
from uniep_taskmgr import Unieptaskmgr
from upload_health_check_zip import import_easy_ops

LOGGER = Logger().getinstance(sys.argv[0])


def check_is_active_node():
    """
    功能说明:检查当前站点是否active节点
    :return: True or False
    """
    if hasattr(nodelist.NodeListMgr(), 'localNodeIsActiveManagerNode'):
        return nodelist.NodeListMgr().localNodeIsActiveManagerNode()
    return nodelist.NodeListMgr().local_node_is_active_manager_node()


def get_function_name():
    '''获取正在运行函数(或方法)名称'''
    return inspect.stack()[1][3]


class OperateScanPkg:
    """
    校验、扫描软件包
    """

    def __init__(self, script_id):
        task_mgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
        self.scan_path = "/opt/oss/manager/var/easysuite_tmp/"
        self.task_path = os.path.join(task_mgr_path, script_id)
        self.res = True
        self.msg_list = []
        self.res_dict = Taskmgrutil.get_res_dict()
        self.task_mgr_function = Taskmgrutil()
        self.task_mgr_function.init_e_taskmgr(self.task_path)
        self.local_lang = Taskmgrutil.get_local_lang()

    def do_exit_finish(self):
        """
        功能说明:成功退出
        :return:
        """
        Taskmgrutil.set_e_taskstatus(self.task_path, "finish")
        Taskmgrutil.set_e_taskprogress(self.task_path, "100")
        return True

    @staticmethod
    def get_remote_ip():
        """
        获取备节点ip
        :return:
        """
        master_ip = ""
        standby_ip = ""
        local_node_ip = ""
        manager_ip_conf = "/opt/oss/manager/var/agent/managerip.conf"
        with open(manager_ip_conf, 'r') as file_obj:
            while True:
                line = file_obj.readline()
                if not line:
                    break
                if line.startswith("managerip"):
                    master_ip = line.split("=")[1].split(",")[0].strip()
                    standby_ip = line.split("=")[1].split(",")[1].strip()
                elif line.startswith("localip"):
                    local_node_ip = line.split("=")[1].strip()
        if not master_ip or not standby_ip or not local_node_ip:
            return False
        if local_node_ip == master_ip:
            return standby_ip
        if local_node_ip == standby_ip:
            return master_ip
        return False

    @staticmethod
    def run_switch_cmd(remote_ip):
        """
        主备倒换
        :param remote_ip:
        :return:
        """
        ret_code = 0
        log_file = "/opt/oss/log/manager/easysuite_upgrade/%s.log" % sys.argv[0].split("/")[-1]
        ssh_cmd = "ssh -o ConnectTimeout=300 -o stricthostkeychecking=no -o ConnectionAttempts=3 " \
                  "-o ServerAliveInterval=10"
        ommha_script = "/opt/oss/manager/apps/OMMHAService/bin/switchover.sh"
        hyper_ha_script = "/opt/oss/manager/apps/HyperHAAgent/bin/hactl"
        switch_cmd = f"bash {ommha_script}"
        # 优先hyper_ha
        if os.path.isfile(hyper_ha_script):
            switch_cmd = f"bash {hyper_ha_script} switch resourcegroup -p manager"
        for try_count in range(0, 3):
            ret_code = os.system(
                "%s %s '%s'>>%s 2>&1 " % (ssh_cmd, remote_ip, switch_cmd, log_file))
            if ret_code == 0:
                LOGGER.info("Finished to run %s" % switch_cmd)
                break
            else:
                LOGGER.info("Fail to run switchover.sh %s times" % try_count)
            time.sleep(1)
        if ret_code != 0:
            LOGGER.info("Failed to run %s" % switch_cmd)
        return True

    def switch_float_ip(self):
        """
        倒换浮动ip
        :return:
        """
        self.record_log("check_float_ip.start")
        is_active = check_is_active_node()
        if is_active:
            self.record_log("check_float_ip.finish")
            LOGGER.info("Local node is active.")
            return True
        self.record_log("check_float_ip.error")

        self.record_log("switch_float_ip.start")
        remote_ip = OperateScanPkg.get_remote_ip()
        if not remote_ip:
            self.record_log("switch_float_ip.error", "error")
            LOGGER.error("Failed to get remote ip.")
            return False
        OperateScanPkg.run_switch_cmd(remote_ip)
        for check_count in range(0, 40):
            is_active = check_is_active_node()
            if is_active:
                LOGGER.info("Local node is active.")
                break
            time.sleep(3)
            LOGGER.info("Local node is not active.%s" % check_count)
        if is_active:
            self.record_log("switch_float_ip.finish")
            return True
        self.record_log("switch_float_ip.error", "error")
        Taskmgrutil.set_e_taskprogress(self.task_path, "10")
        return False

    @staticmethod
    def check_is_management_domain():
        """
        功能输出：检查当前环境是否单管
        :return: True;False
        """
        manager_nodes_file = "/opt/oss/manager/etc/sysconf/nodelists.json"
        with open(manager_nodes_file, 'r') as file_obj:
            file_data = json.load(file_obj)
            for node in file_data.get("nodeList").values():
                if node.get("hostname").lower().startswith("nms_server"):
                    return True
        return False

    def copy_pkg_with_sign(self, pkg_file):
        """
        拷贝软件包至扫描软件路径
        :param pkg_file:
        :return:
        """
        os.system("mkdir -p %s" % self.scan_path)
        if os.path.isfile(pkg_file) and os.path.isfile("%s.cms" % pkg_file) and os.path.isfile( \
                "%s.crl" % pkg_file):
            ret_code = os.system("cp %s %s" % (pkg_file, self.scan_path))
            ret_code_cms = os.system("cp %s.cms %s" % (pkg_file, self.scan_path))
            ret_code_crl = os.system("cp %s.crl %s" % (pkg_file, self.scan_path))
            if ret_code != 0 or ret_code_cms != 0 or ret_code_crl != 0:
                return False
        elif os.path.isfile(pkg_file) and os.path.isfile("%s.asc" % pkg_file):
            ret_code_pkg = os.system("cp %s %s" % (pkg_file, self.scan_path))
            ret_code_asc = os.system("cp %s.asc %s" % (pkg_file, self.scan_path))
            if ret_code_pkg != 0 or ret_code_asc != 0:
                return False
        elif os.path.isfile(pkg_file) and os.path.isfile("%s.p7s" % pkg_file):
            ret_code_pkg = os.system("cp %s %s" % (pkg_file, self.scan_path))
            ret_code_asc = os.system("cp %s.p7s %s" % (pkg_file, self.scan_path))
            if ret_code_pkg != 0 or ret_code_asc != 0:
                return False
        else:
            return False
        return True

    def copy_scan_pkg(self, pkg_path, pkgs_str):
        """
        解析需求拷贝的软件包列表，执行拷贝
        :param pkg_path:
        :param pkgs_str:
        :return:
        """
        omp_pub_path = "/opt/pub/software/mgr-installdisk/"
        pkg_list = pkgs_str.split(",")
        for one_pkg in pkg_list:
            self.res_dict.update({"pkg": one_pkg})
            if one_pkg.find("OSMediation") > 0:
                LOGGER.info("Skip the {} package.".format(one_pkg))
                continue
            if one_pkg.find("_OMP") > 0:
                omp_pub_file = os.path.join(omp_pub_path, one_pkg.split(".")[0],
                                            "software_define.yaml")
                if os.path.isfile(omp_pub_file):
                    self.record_log("pkg_exit")
                    LOGGER.info("The omp package is already exist in the software repository.("
                                "%s)" % omp_pub_file)
                    continue
            if OperateScanPkg.check_is_management_domain():
                if one_pkg.find("-GaussDBPatch-") > 0 or one_pkg.find("_server_dbpatch_linux") > 0:
                    LOGGER.info("The node is NMS_Server.")
                    LOGGER.info("Skip dbpath package in scan step.")
                    continue
            self.record_log("copy_pkg.start")
            if not self.copy_pkg_with_sign(os.path.join(pkg_path, one_pkg)):
                LOGGER.error("Failed to copy pkg.(%s)" % one_pkg)
                self.record_log("copy_pkg.error", "error")
                return False
            self.record_log("copy_pkg.finish")
        Taskmgrutil.set_e_taskprogress(self.task_path, "20")
        return True

    def get_res_value(self, key):
        """
        更新软件包名
        :param key:
        :return:
        """
        value = self.res_dict.get(key)
        # 进行两次宏替换
        for one_key in self.res_dict:
            mac_one_key = "{%s}" % one_key
            if mac_one_key in value:
                value = value.replace(mac_one_key, self.res_dict.get(one_key))
        if value.find("{") >= 0:
            for one_key in self.res_dict:
                mac_one_key = "{%s}" % one_key
                if mac_one_key in value:
                    value = value.replace(mac_one_key, self.res_dict.get(one_key))
        return value

    def record_log(self, msg, log_type="", i18=True):
        """
        更新任务process、status、和日志
        :param msg:
        :param log_type:
        :param i18:
        :return:
        """
        now_time = datetime.strftime(datetime.now(), '%Y-%m-%d %H:%M:%S')
        if i18:
            str_msg = "[%s] [%s] %s" % (now_time, os.getpid(), self.get_res_value(msg))
        else:
            str_msg = "[%s] [%s] %s" % (now_time, os.getpid(), msg)
        self.msg_list.append(str_msg)
        self.task_mgr_function.set_e_taskmsg(self.task_path, "\n".join(self.msg_list))
        if log_type == "error":
            Taskmgrutil.set_e_taskstatus(self.task_path, "error")
            Taskmgrutil.set_e_taskprogress(self.task_path, "100")
        return True

    def thread_scan(self, cmd):
        """
        执行软件包扫描，实时更新进展
        :param cmd:
        :return:
        """
        ret_code, ret_msg = Taskmgrutil.execute_cmd(cmd)
        if ret_msg:
            LOGGER.info("Scan pkg result:%s" % ret_msg)
        if ret_code != 0:
            self.record_log("scan_pkg.error", "error")
            self.res = False
            return False
        self.record_log("scan_pkg.finish")
        return True

    def scan_pkg(self):
        """
        创建启动软件包扫描线程
        :return:
        """
        self.record_log("scan_pkg.start")
        scan_cmd = "bash /opt/oss/manager/tools/deployapp/scanpackage.sh -path %s" % self.scan_path
        back_thread = threading.Thread(target=self.thread_scan, args=(scan_cmd,))
        back_thread.start()
        progress = 20
        while back_thread.is_alive():
            time.sleep(12)
            progress = progress + 1
            if progress >= 70:
                progress = 70
            Taskmgrutil.set_e_taskprogress(self.task_path, str(progress))
        return self.res

    def get_features(self, work_path):
        """
        生成特性集信息
        :return:
        """
        result_data = []
        query_cmd = "bash /opt/oss/manager/tools/resmgr/queryproduct.sh -pn all -output {}".format(work_path)
        if os.system(query_cmd) != 0:
            LOGGER.error("Failed to query the product information(1), Retry...")
            time.sleep(3)
            if os.system(query_cmd) != 0:
                LOGGER.error("Failed to query the product information(2).")
                return result_data
        for json_file in os.listdir(work_path):
            if not json_file.startswith("product_") \
                    or not json_file.endswith(".json"):
                continue

            features_data = {"productname": "", "name": "", "features": []}
            with open("{}/{}".format(work_path, json_file), 'r') as file_obj:
                product_data = json.load(file_obj)
            features_data["productname"] = product_data.get("productname")
            features_data["name"] = product_data.get("productext").get("featureGroup")
            features = product_data.get("productext").get("features").split(";")
            for feature in features:
                if not feature.strip():
                    continue
                features_data["features"].append({"name": feature})
            result_data.append(copy.deepcopy(features_data))
        return result_data

    def get_packages(self, pkgs_str):
        """
        生成软件包信息
        :return:
        """
        result_data = []
        pkgs_url = "/rest/unideploywebsite/v1/product/pkgs"
        flag, response = Unieptaskmgr().send_get_request(pkgs_url)
        if not flag:
            LOGGER.error("Failed to query the pkgs information(1), Retry...")
            time.sleep(3)
            flag, response = Unieptaskmgr().send_get_request(pkgs_url)
            if not flag:
                LOGGER.error("Failed to query the pkgs information(2).")
                return result_data
        response = json.loads(response)
        for one_pkg in pkgs_str.split(","):
            for pkg_info in response["entity"]:
                if one_pkg == pkg_info["packageName"]:
                    result_data.append({
                        "name": pkg_info["softwareName"],
                        "version": pkg_info["softwareVersion"]
                    })
        return result_data

    def distribut_service_package(self, pkgs_str, is_r19="false"):
        """
        调用平台脚本，预分发的服务包
        :return:
        """
        # 单管不执行分发任务,升级源版本是R19C00不执行
        if is_r19 == "true" or OperateScanPkg.check_is_management_domain():
            self.do_exit_finish()
            return True
        self.record_log(msg="Start the predistribution service package.", i18=False)
        work_path = "/opt/upgrade/easysuite_upgrade/predistribution"
        if not os.path.exists(work_path):
            os.mkdir(work_path)
        packages = self.get_packages(pkgs_str)
        result_datas = self.get_features(work_path)
        if len(result_datas) == 0 or len(packages) == 0:
            LOGGER.info("No software package or product data. Skip this step.")
            self.record_log(msg="Service package predistribution completed.", i18=False)
            self.do_exit_finish()
            return True
        try:
            count_step = int(29/(2*len(result_datas)))
        except ZeroDivisionError:
            LOGGER.error("You can't divide by 0!")

        progress = 70
        for result_data in result_datas:
            input_data = {
                "user_isolation": "true",
                "productname": result_data["productname"],
                "featuregrouplist": [{
                    "name": result_data["name"],
                    "features": result_data["features"]
                }],
                "packages": packages
            }
            # 输出input文件
            input_file_path = "{}/input_{}.json".format(work_path, result_data["productname"])
            try:
                json_data = json.dumps(input_data, indent=4, ensure_ascii=False)


                with os.fdopen(os.open(input_file_path, os.O_CREAT | os.O_WRONLY | os.O_TRUNC, mode=0o660),
                               'w', encoding='utf-8') as json_file:
                    json_file.write(json_data)
                LOGGER.info("The {} configuration file is exported.".format(input_file_path))
            except Exception as e:
                LOGGER.error("Failed to write file %s, Exception:%s." % (input_file_path, e))
                self.record_log(msg="Failed to write file %s. Skip this step.", i18=False)
                break
            # 准备调用平台接口预分发服务包
            pre_upgrade_file_src = "/opt/upgrade/easysuite_upgrade/scripts/common/NCE-Common/pyscripts/pre_upgrade.py"
            pre_upgrade_file_dst = "/opt/oss/manager/agent/tools/pyscript/tools/pre_upgrade.py"
            cp_pre_upgrade_cmd = "cp -f {} {}".format(pre_upgrade_file_src, pre_upgrade_file_dst)
            if not os.path.isfile(pre_upgrade_file_dst) and not os.path.isfile("{}c".format(pre_upgrade_file_dst)):
                if not os.path.isfile(pre_upgrade_file_src):
                    self.record_log(msg="The pre_upgrade.py* module cannot be found. Skip this step.", i18=False)
                    break
                if os.system(cp_pre_upgrade_cmd) != 0:
                    LOGGER.error("Failed to copy the pre_upgrade module(1), Retry...")
                    if os.system(cp_pre_upgrade_cmd) != 0:
                        LOGGER.error("Failed to copy the pre_upgrade module(2).")
                        self.record_log(msg="Failed to copy the predistributed file. Skip this step.", i18=False)
                        break
            # 预分发服务包，步骤1
            env_cmd = "source /opt/oss/manager/bin/engr_profile.sh"
            task_cmd_one = "ipmc_tool -cmd pre_upgrade -o fuzzy_calculate_pkgs -input_file {}".format(input_file_path)
            res_code_one = os.system("{};{}".format(env_cmd, task_cmd_one))
            progress = progress + count_step
            Taskmgrutil.set_e_taskprogress(self.task_path, str(progress))
            if res_code_one != 0:
                LOGGER.error("Failed to execute the {} command, ErrorCode:{}".format(task_cmd_one, res_code_one))
                self.record_log(msg="Pre-distribution is abnormal. Skip this step.", i18=False)
                break
            # 预分发服务包，步骤2
            task_cmd_two = "ipmc_tool -cmd pre_upgrade -o prepare_pkgs"
            res_code_two = os.system("{};{}".format(env_cmd, task_cmd_two))
            progress = progress + count_step
            Taskmgrutil.set_e_taskprogress(self.task_path, str(progress))
            if res_code_two != 0:
                LOGGER.error("Failed to execute the {} command, ErrorCode:{}".format(task_cmd_two, res_code_two))
                self.record_log(msg="Pre-distribution is abnormal. Skip this step.", i18=False)
                break
        self.record_log(msg="Service package predistribution completed.", i18=False)
        self.do_exit_finish()
        return True

    def check_omp_tasks(self):
        # 检查管理面任务分页接口
        query_omp_tasks_url = f'/rest/plat/omp/v1/main/common/taskmgr/pagingtasks?' \
                              'jobType=ui&page-length=20&page-size=1'
        status, response = Unieptaskmgr().send_get_request(query_omp_tasks_url)
        if not status:
            self.record_log("check_omp_task.failed", "error")
            return False
        omp_tasks = json.loads(response).get("taskList", [])
        for task in omp_tasks:
            current_state = task.get('currentState')
            is_enabled = task.get('isEnabled')
            is_schedule_job = task.get('isScheduleJob')
            if current_state in ['RUNNING', 'TERMINATING']:
                self.record_log("check_omp_task.error", "error")
                return False
            if current_state == "WAITING" and ((is_schedule_job == "yes" and is_enabled == "yes") or
                                               is_schedule_job == "no"):
                self.record_log("check_omp_task.error", "error")
                return False
        return True

    def call_platform_shell(self):
        # 调用平台规避脚本
        platform_cmd = "bash /opt/upgrade/easysuite_upgrade/scripts/CloudSop/fix_repo_tmp_path.sh"
        try:
            process = subprocess.Popen(shlex.split(platform_cmd), shell=False, encoding="utf-8",
                                       stderr=subprocess.PIPE,
                                       stdout=subprocess.PIPE)
            stdout, stderr = process.communicate(timeout=3600)
            code = process.returncode
            if code != 0:
                self.record_log(stderr, 'error')
                return False
            else:
                self.record_log("call platform shell scripts success", i18=False)
                return True
        except Exception as e:
            self.record_log("call_platform_shell.exception", i18=False)
            return False

    def import_dfs_pkg(self, pkg_path, easy_ops_path):
        """
        功能说明:导入巡检包
        :param pkg_path: 包路径
        :param easy_ops_path: easy_ops包路径
        """
        if easy_ops_path == "None":
            # 无Easy Ops包不导入
            return True
        self.record_log("Start to import easy_ops pkg", i18=False)
        result = import_easy_ops([sys.argv[0], pkg_path, easy_ops_path])
        if not result:
            self.record_log("Failed to import easy_ops pkg", "error", i18=False)
            return False
        self.record_log("Finished to import easy_ops pkg", i18=False)
        return True


def main(argv):
    """
    升级流程扫描软件包
    :param argv:
    :return:
    """
    pkg_path = argv[1]
    pkgs_str = argv[2]
    script_id = argv[3]
    is_r19 = argv[4]
    easy_ops_path = argv[5]
    if not pkg_path or not pkgs_str or not script_id:
        LOGGER.error("Input params is wrong.")
        return False

    check_function = OperateScanPkg(script_id)
    # 表驱动进行调度
    work_flow = [{"function": check_function.import_dfs_pkg, "param": [pkg_path, easy_ops_path]},
                 {"function": check_function.call_platform_shell, "param": []},
                 {"function": check_function.check_omp_tasks, "param": []},
                 {"function": check_function.switch_float_ip, "param": []},
                 {"function": check_function.copy_scan_pkg, "param": [pkg_path, pkgs_str]},
                 {"function": check_function.scan_pkg, "param": []},
                 {"function": check_function.distribut_service_package,
                  "param": [pkgs_str, is_r19]}]
    for one_step in work_flow:
        try:
            result = one_step.get('function')(*one_step.get('param', []))
        except Exception as e_msg:
            check_function.record_log(f"Failure due to exception:{e_msg}", "error", i18=False)
            return False
        if result is False:
            return False
    return True


if __name__ == '__main__':
    main(sys.argv)
