#!/usr/bin/python
# -*- coding: utf-8 -*-
import time
import traceback
import uuid

from getDBConnection import get_zenith_session
import common_tasks.const_sql as const_sql
from common_tasks.base_task import BaseTask

ONE_PACKAGE = 500


class UpgradeQxPara(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradeQxPara, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeQxPara init product_name is %s" % product_name)
        self.src_db_session = get_zenith_session('MCDB', 'MCDB', product_name)
        if self.src_db_session is None:
            self.error("MCDB is None")
            return
        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('neresdb', 'neresdb', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "qxPara"
        self.src_table_cols = ("cNEID","cNeUser", "cPassword", "purpose")

        self.src_table_cols_index = {y:x for x, y in enumerate(self.src_table_cols)}
        self.dst_table = "QxPara"
        self.dst_table_cols = ("modelId", "fModelId", "tenantId", "neResId", "neId", "purpose", "connectStatus",
                               "userName", "password", "createAt", "updateAt")
        self.idmapping_dic = {}

    def get_resid_from_idmapping(self, paras):
        self.debug("get_resid_from_idmapping::start")
        idmappingdb_sess = get_zenith_session('idmappingdb', 'idmappingdb', self.product_name)
        if idmappingdb_sess is None:
            self.error("get idmappingdb_sess session fail")
            return
        idmappingdb_sess.autocommit(True)
        id_mapping_cursor = idmappingdb_sess.cursor()

        tmp_table_name = "tmp_neid_%s" % self.dst_table
        id_mapping_cursor.execute(const_sql.DROP_TEMP_TABLE % tmp_table_name)
        id_mapping_cursor.execute(const_sql.CREATE_TEMP_TABLE % tmp_table_name)
        insert_stmt = "insert into tmp_%s values(:1)" % tmp_table_name

        nativeIds = []
        for para in paras:
            data = []
            nativeId = "NE=%s" % para[self.src_table_cols_index.get("cNEID")]
            data.append(nativeId)
            tuple_data = tuple(data)
            nativeIds.append(tuple_data)
            if len(nativeIds) == ONE_PACKAGE:
                id_mapping_cursor.executemany(insert_stmt, nativeIds)
                self.debug("one package:%s" % nativeIds)
                nativeIds = []

        if len(nativeIds) != 0:
            id_mapping_cursor.executemany(insert_stmt, nativeIds)
            self.debug("last package datas: %s" % nativeIds)

        query_stmt = const_sql.INNER_JOIN_TEMP_TABLE % tmp_table_name
        self.debug("query sql: %s" % query_stmt)
        id_mapping_cursor.execute(query_stmt)
        result = id_mapping_cursor.fetchall()

        for r in result:
            l = list(r)
            self.idmapping_dic[l[1].lstrip("NE=")] = l[0]
            self.debug("result: %s: %s" % (l[1], l[0]))

        id_mapping_cursor.execute(const_sql.DROP_TEMP_TABLE % tmp_table_name)
        id_mapping_cursor.close()
        idmappingdb_sess.close()

    def convert_data(self, paras):
        modelId = str(uuid.uuid1())
        fModelId = None
        tenantId = "default-organization-id"
        neResId = self.idmapping_dic.get(str(paras[self.src_table_cols_index.get("cNEID")]))
        neId = paras[self.src_table_cols_index.get("cNEID")]
        # Done: 字段值待定--已经确定
        purpose = paras[self.src_table_cols_index.get("purpose")]
        connectStatus = -1
        userName = paras[self.src_table_cols_index.get("cNeUser")]
        password = paras[self.src_table_cols_index.get("cPassword")]

        createAt = int(time.time() * 1000)
        updateAt = int(time.time() * 1000)

        return tuple(None if x is None else str(x) for x in (
            modelId, fModelId, tenantId, neResId, neId, purpose, connectStatus,
            userName, password, createAt, updateAt))

    def to_UpgradePara(self, stelnet_templates):
        self.get_resid_from_idmapping(stelnet_templates)
        col_names = "`" + ("`, `".join(self.dst_table_cols)) + "`"
        val_ids = ":" + (",:".join((str(x+1) for x in range(len(self.dst_table_cols)))))
        insert_stmt = "INSERT INTO `%s` (%s) VALUES(%s)" % (self.dst_table, col_names, val_ids)
        self.debug("%s" % insert_stmt)
        list_datas = []
        dst_cols_len = len(self.dst_table_cols)
        for tempalte in stelnet_templates:
            data = self.convert_data(tempalte)
            if len(data) == dst_cols_len:
                tuple_data = tuple(data)
                list_datas.append(tuple_data)
            else:
                self.warning("coverted data length not equals dst table cols, to be ignored.")
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def get_src_data_by_db_inst(self, db_inst):
        try:
            db_sess = get_zenith_session(db_inst, db_inst, self.product_name)
            if db_sess is None:
                # 找不到数据库，可能是当前场景不需要升级
                self.warning("%s db session is None" % db_inst)
                return []
        except TypeError as te:
            self.warning("can not get db session for db: %s, err is :%s"% (db_inst, str(te)))
            return []
        except BaseException as be:
            self.warning("can not get db session for db: %s, err is :%s"% (db_inst, str(be)))
            return []

        db_sess.autocommit(True)
        db_cur = db_sess.cursor()
        # 这个SQL语句的字段顺序与self.src_table_cols必须保持一致
        select_stmts = [
            # Done：通过表的来源，提供purpose字段的值
            "select  cLogicNeId, cNEUser, cEncryptPassword, 'default' from tTECommuNe",
            # "select  cNEID, cNeUser, cPassword, 'DC' from tTENEDCUser",
            # "select  cNEID, cNeUser, cPassword, 'uTraffic' from tTENEPMSUser",
            # "select  cNEID, cNeUser, cPassword, 'MML' from tTENEMmlUser",
        ]
        templates = []
        for select_stmt in select_stmts:
            self.debug("execute sql: %s" % select_stmt)
            try:
                db_cur.execute(select_stmt)
                temp = db_cur.fetchall()
            except BaseException as be:
                self.warning("execute sql failed, sql: %s, error: %s" % (select_stmt, be))
                continue

            if temp:
                temp = list(temp)
                templates.extend(temp)
            self.info('get qx para data success count:%d' % len(temp))
        self.info('get qx para data success total count:%d' % len(templates))

        db_cur.close()
        db_sess.close()

        return templates

    def get_src_data_by_db_inst_for_v8ptn(self, db_inst):
        try:
            db_sess = get_zenith_session(db_inst, db_inst, self.product_name)
            if db_sess is None:
                # 找不到数据库，可能是当前场景不需要升级
                self.warning("%s db session is None" % db_inst)
                return []
        except TypeError as te:
            self.warning("can not get db session for db: %s, err is :%s"% (db_inst, str(te)))
            return []
        except BaseException as be:
            self.warning("can not get db session for db: %s, err is :%s"% (db_inst, str(be)))
            return []

        db_sess.autocommit(True)
        db_cur = db_sess.cursor()
        # 这个SQL语句的字段顺序与self.src_table_cols必须保持一致
        select_stmt = "select devID, devNEUser, devEncryptPassword, 'default' from t_dev_commune"
        self.debug("execute sql: %s" % select_stmt)
        try:
            db_cur.execute(select_stmt)
        except BaseException as be:
            self.warning("execute sql failed, err is: %s"% str(be))
            return []
        templates = db_cur.fetchall()
        if templates:
            templates = list(templates)
        self.info('get qx para data success count:%d' % len(templates))

        # dcn_java不需要根据网关的物理ID获取逻辑ID
        db_cur.close()
        db_sess.close()

        return templates

    def merge_cpp_java_inst_data(self, dcn_cpp, dcn_java):
        # 若qxDcn_JAVA存在devID在qxDcn_C++中不存在，则需要在QxDcn和QxPara中新增这条数据
        # 若qxDcn_JAVA存在devId在qxDcn_C++中存在，则刷新QxDcn中iCommuProType字段数据。（不要遗漏这一步操作）
        java_inst_devid_index = dict((paras[self.src_table_cols_index.get("cNEID")], i) for i, paras in enumerate(dcn_java))
        java_inst_dev_id_set = set(java_inst_devid_index.keys())
        cpp_inst_devid_index = dict((paras[self.src_table_cols_index.get("cNEID")], i) for i, paras in enumerate(dcn_cpp))
        cpp_inst_dev_id_set = set(cpp_inst_devid_index.keys())

        # 若qxDcn_JAVA存在devID在qxDcn_C++中不存在，则需要在QxDcn和QxPara中新增这条数据
        to_add_devids = java_inst_dev_id_set - cpp_inst_dev_id_set
        for devid in to_add_devids:
            java_idx = java_inst_devid_index.get(devid)
            dcn_cpp.append(dcn_java[java_idx])

        return dcn_cpp

    def do_db_inst(self, process_type, inst_id):
        db_inst_map = {
            "nemgr_trans": "nemgr_transDB_%s",
            "nemgr_v8trans": "nesvc_v8transDB_%s",
            "nemgr_marine": "nemgr_marineDB_%s",
            "nemgr_v8ptn": "nesvc_v8ptnDB_%s", # v8ptn还有一个java进程的数据库实例需要额外处理
        }
        if process_type in db_inst_map:
            db_inst = db_inst_map.get(process_type) % inst_id
        else:
            self.warning("not valid process_type: %s" % process_type)
            return
        self.info('UpgradeQxPara::do_db_inst(%s) start!' % db_inst)
        if process_type == "nemgr_v8ptn":
            dcn_cpp = self.get_src_data_by_db_inst(db_inst)
            dcn_java = self.get_src_data_by_db_inst_for_v8ptn("ptn_v8_db_%s" % inst_id)
            # 合并数据, 按照规则，将dcn_java中的数据合并到dcn_cpp，并返回合并后的dcn_cpp
            templates = self.merge_cpp_java_inst_data(dcn_cpp, dcn_java)
        else:
            templates = self.get_src_data_by_db_inst(db_inst)
        self.to_UpgradePara(templates)

        self.info('UpgradeQxPara::do_db_inst(%s) Done!' % db_inst)

    def do(self):
        try:
            self.info('UpgradeQxPara::do start!')
            # 先从MCDB查出多个数据库实例，再用数据库实例的连接进行操作：
            sql_get_db_inst = "select DISTINCT process_type, handle " \
                              "from currenthandle " \
                              "where process_type = '%s' order by handle asc"
            for process_type_name in ("nemgr_trans", "nemgr_v8trans", "nemgr_marine", "nemgr_v8ptn"):
                select_stmt = sql_get_db_inst % process_type_name
                ret = self.exec_query_sql(self.src_db_cursor, select_stmt)
                if ret:
                    self.debug("process_type, handle: %s" % str(ret))
                    for process_type, handle in ret:
                        self.do_db_inst(process_type_name, handle)
                else:
                    self.debug("get no record, use default")
                    self.do_db_inst(process_type_name, 1)
            self.info('to_UpgradeQxPara done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeQxPara got exception')
            self.error(traceback.format_exc())
            return -1
        self.info('UpgradeQxPara::do done')
        return 0


if __name__ == '__main__':
    tool = UpgradeQxPara()
    print('[INFO] UpgradeQxPara start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeQxPara finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
