import json
import os
import re
import sys
import time

from backup_product_url import backup_product_service as backup_product_func
from backup_product_url import BackupProductData
from check_building_db import main as check_build_func
from commonlog import Logger
from uniep_taskmgr import Unieptaskmgr
from taskmgr_util import Taskmgrutil

logger = Logger().getinstance(sys.argv[0])
RUNNING_STATE = ["INIT", "RUNNING", "WAITING"]
FAILURE_STATE = ["FAILURE", "PARTIAL_SUCCESS", "TERMINATING", "TERMINATED"]


class BackupProductWithDiff:
    """
    增量备份
    """
    def __init__(self, e_task_path):
        self.task_path = e_task_path
        self.start_time = int(time.time())
        self.time_out = 3600

    def set_task(self, task_path='', status='', progress='', msg=''):
        """
        功能说明：设置任务状态和信息
        :param task_path:
        :param status:
        :param progress:
        :param msg:
        :return:
        """
        if task_path:
            Taskmgrutil().init_e_taskmgr(task_path)
        if status:
            Taskmgrutil.set_e_taskstatus(self.task_path, status)
        if progress:
            Taskmgrutil.set_e_taskprogress(self.task_path, progress)
        if msg:
            msg = f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] [{os.getpid()}] {msg}"
            Taskmgrutil.set_msg_append(self.task_path, msg)

    def get_response(self, response):
        """
        功能说明：请求响应
        :return:
        """
        if not response:
            return False
        try:
            json.loads(response)
        except Exception:
            logger.error(f"Failed to load json from \"{response}\"")
            self.set_task(msg=f'The task response is \"{response}\"')
            return False
        return True

    def query_backup_progress(self, url):
        """
        查询恢复产品数据任务进度
        :param url: 查询任务执行状态所需的url
        :return:
        """
        result = False
        self.set_task(msg="Start to query backup product progress.")
        max_time = self.start_time + self.time_out
        while True:
            # 超时报错退出
            if int(time.time()) >= max_time:
                logger.error("Time_out")
                self.set_task(status='ERROR', progress='100', msg=f'The task execution timeout exceeds {self.time_out} '
                                                                  f'seconds.')
                break

            status, response = Unieptaskmgr().send_get_request(url)

            # 请求失败
            if not status or not response:
                self.set_task(status='ERROR', progress='100', msg='Failed to send get request.')
                break

            # json解析失败重试
            if not self.get_response(response):
                self.set_task(status='ERROR', progress='100', msg='Failed to get response.')
                break

            execute_result = json.loads(response)[0].get("currentState")
            progress = json.loads(response)[0].get('progress')
            if not execute_result or execute_result in FAILURE_STATE:
                self.set_task(status='ERROR', progress='100', msg=f'Backup product is error, progress is {progress}%.')
                break
            elif execute_result in RUNNING_STATE:
                # 运行中状态
                self.set_task(status='RUNNING', progress=f"{progress}", msg=f'Backup product is running, progress is '
                                                                            f'{progress}%.')
                time.sleep(5)
            else:
                self.set_task(status='RUNNING', progress='99', msg=f'Backup product is finish, progress is '
                                                                   f'{progress}%.')
                result = True
                break
        return result

    def write_backup_timestamp_file(self, url, task_id):
        """
        写备份产品数据的时间戳到OMP后台文件
        :param url:
        :param task_id:
        :return:
        """
        logger.info("Start to write backup timestamp file")
        try:
            status, response = Unieptaskmgr().send_get_request(url)
            details = json.loads(response)[0].get("details")
            timestamp = re.search("(20[2-3]\d{11}(:?\d{3})?)", details).group(1)
            timestamp_str = "backup_product_with_diff_%s: %s" % (str(task_id), timestamp)
            # 创建备份目录
            initial_path_cmd = '[ ! -d "/opt/upgrade/backup" ] ' \
                               '&& mkdir -p /opt/upgrade/backup ' \
                               '&& chown ossadm:ossgroup /opt/upgrade/backup ' \
                               '&& chmod 750 /opt/upgrade/backup'
            ret_code = os.system(initial_path_cmd)
            if ret_code != 0:
                logger.error("Make upgrade directory failed")
            # 生成备份任务对应的时间戳文件
            backup_timestamp = "/opt/upgrade/backup/backup_timestamp.properties"
            write_timestamp_cmd = "echo %s >>%s; chmod 640 %s; echo success$?" % (timestamp_str, backup_timestamp,
                                                                                  backup_timestamp)
            ret_code = os.system(write_timestamp_cmd)
            if ret_code != 0:
                logger.error("Wirte backup timestamp file failed")
                return False
        except Exception as e:
            logger.error("Wirte backup timestamp file throw an exception: {}".format(e))
            return False
        logger.info("Write backup timestamp file success, backup timestamp:{}.".format(timestamp))
        return True


def main(argv):
    """
    功能说明：功能说明
    :return:
    """
    product_name = argv[1]
    ssh_node = {
        "ip": argv[2],
        "username": argv[3]
    }
    uniep_version = argv[4]
    upgrade_path = argv[5]
    remote_backup = argv[6]
    task_id = argv[7]

    e_task_id = argv[8]
    e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
    e_task_path = os.path.join(e_taskmgr_path, e_task_id)

    # 初始化任务
    backup_product_with_diff = BackupProductWithDiff(e_task_path)
    backup_product_with_diff.set_task(task_path=e_task_path, status="RUNNING", progress="0")

    # 检查数据库实例角色
    check_build_func(argv)

    # 构造任务下发参数,并下发请求
    backup_product_result = backup_product_func(ssh_node, product_name, uniep_version, upgrade_path, remote_backup,
                                                "PHYSICAL_DIFF_BACKUP")
    if backup_product_result.get("status") != 200:
        return False

    # 查询请求结果，根据url获取任务执行进度
    back_url = backup_product_result.get("url")
    if not backup_product_with_diff.query_backup_progress(back_url):
        return False

    # 关闭 BCT
    if not BackupProductData.gauss_db_bct("off"):
        return False

    # 记录备份url返回的备份时间戳到后台配置文件/opt/upgrade/backup/backup_timestamp.properties
    if not backup_product_with_diff.write_backup_timestamp_file(back_url, task_id):
        return False

    backup_product_with_diff.set_task(status="FINISH", progress="100")
    return True


if __name__ == "__main__":
    main(sys.argv)
