#!/bin/bash
#***************************************************************#
# Name             : get_os_info.sh                             #
# Description      : 获取操作系统类型和版本                         #
# Platform         : Linux                                      #
# Creation date    : 2022-04-26                                 #
# usage            : source /opt/tools/get_os_info.sh           #
# log :            : /var/log/messages                          #
#***************************************************************#

#***************************************************************#
# 输出OS类型：[euler, suse, centos, kylin]
# 输出采取 export变量 的方式
# 仅支持euler, suse, centos, kylin
#***************************************************************#
function get_os_type() {
    local os_type
    if [ "$(cat /proc/version | grep -i suse)" ];then
        os_type="suse"
        local suse_type
        lsblk | grep root_bak > /dev/null 2>&1
        if [ $? -eq 0 ];then
            suse_type="hw-delivered"
        else
            suse_type="non-hw-delivered"
        fi
    elif [ "$(cat /proc/version | grep -i euler)" ];then
        os_type="euler"
    elif [ "$(cat /proc/version | grep -i centos)" ];then
        os_type="centos"
    elif [ "$(cat /proc/version | grep -i kylin)" ];then
        os_type="kylin"
    else
        os_type="unknown"
    fi
    export OS_TYPE="${os_type}"
    [ ! -z "${suse_type}" ] && export SUSE_TYPE="${suse_type}"
}

#***************************************************************#
# 输出硬件平台类型：[x86_64, aarch64]
# 输出采取 export变量 的方式
# 支持x86_64, aarch64
#***************************************************************#
function get_arch_type() {
    local arch_type
    arch_type="$(uname -i)"
    if [[ "${arch_type}" =~ "x86_64" ]];then
        arch_type="x86_64"
    elif [[ "${arch_type}" =~ "aarch64" ]];then
        arch_type="aarch64"
    else
        arch_type="unknown"
    fi
    export ARCH_TYPE="${arch_type}"
}

#***************************************************************#
# 输出操作系统版本：[euler:[v2r7..], suse:[12SP5..], kylin:[ky10..]]
# 输出采取 export变量 的方式
# 仅支持euler, suse, kylin
#***************************************************************#
function get_os_version() {
    local os_version
    if [ "${OS_TYPE}" = "euler" ];then
        os_version=$(uname -r | awk -F"euleros" '{print $2}' | awk -F"." '{print $1}')
    elif [ "${OS_TYPE}" = "suse" ];then
        local version=$(cat /etc/SuSE-release | grep ^VERSION | awk -F"=" '{print $2}' | xargs)
        local patch_level=$(cat /etc/SuSE-release | grep ^PATCHLEVEL | awk -F"=" '{print $2}' | xargs)
        os_version="${version}SP${patch_level}"
    elif [ "${OS_TYPE}" = "kylin" ]; then
        os_version="ky$(uname -r | awk -F ".ky" '{print $2}' | awk -F "." '{print $1}')"
    else
        os_version="unknown"
    fi
    export OS_VERSION="${os_version}"
}

#***************************************************************#
# 入口函数
# 每次执行，logger命令记录执行时间和结果
#***************************************************************#
function main() {
    logger -p local0.info "start to run get_os_info.sh to get info."
    get_os_type
    get_arch_type
    get_os_version
    logger -p local0.info "success to get info, os type:${OS_TYPE}, arch type:${ARCH_TYPE}, os version:${OS_VERSION}."
}

main