#!/bin/bash
#get absolute path
FileName="`basename $0`"
RELATIVE_PATH="$(dirname $0)"
cd "${RELATIVE_PATH}"
ABSOLUATELY_PATH=`pwd`

MODULE_NAME="`basename $0 | sed 's/\.sh$//'`"
#import common variable..
. ./common.inc
cd - &> /dev/null

backup_path=""
oldinstall_path=""
newinstall_path=""
version=""

function fn_check_params()
{
    PRINT_LOG "INFO" "Start to check input params."

    if [ ! -d "${backup_path}" ]
    then
        PRINT_LOG "FATAL" "The backup path is not exist."
        return 3
    fi

    if [ -z "${version}" ]
    then
        PRINT_LOG "FATAL" "The version is empty."
        return 3
    fi

    if [[ ! "${version}" =~ ^[a-zA-Z0-9]+$ ]]
    then
        PRINT_LOG "FATAL" "The ${version} is invalid, the version only contain the chars [a-zA-Z0-9]."
        return 3
    fi

    backup_path="${backup_path}/${version}"
    [ ! -d "${backup_path}" ] && mkdir "${backup_path}"

    if [ ! -z "${old_version}" ]
    then
        if [[ ! "${old_version}" =~ ^[a-zA-Z0-9]+$ ]]
        then
            PRINT_LOG "FATAL" "The ${old_version} is invalid, the version only contain the chars [a-zA-Z0-9]."
            return 3
        fi
    fi

    if [ -z "${oldinstall_path}" -o -z "${newinstall_path}" ]
    then
        PRINT_LOG "FATAL" "The install path is empty."
        return 3
    fi

    fn_check_file "${oldinstall_path}" || return $?

    PRINT_LOG "INFO" "Check input params done."
    return 0
}

function fn_check_file()
{
    local check_dir="$1"

    if [ ! -d "${check_dir}" ]
    then
        PRINT_LOG "WARN" "The parameter <${check_dir}> is not directory, please check"
        return 0
    fi

    local root_file=$(find ${check_dir} -user root -o -group root)
    if [ ! -z "${root_file}" ]
    then
        ERROR_CODE="${ERROR_CODE} UpgradeOsconfig000004 "
        PRINT_LOG "FATAL" "The root file <${root_file}> is exist in osconfig path,please check and clean it."
        return 1
    fi
    return 0
}

#*************************************************************#
# Name:         fn_backup_precheck
# Description:  backup precheck function
# Parameters:   NULL
# return:       0: no need to backup
#               1: need to backup
#*************************************************************#
function fn_backup_precheck()
{
    local nobackup_flag="${backup_path}/nobackup_${version}"

    #Note: install path is not exist, no need to backup
    if [ ! -d "${oldinstall_path}" -o ! -f "${oldinstall_path}/os/bin/osprofile" ]
    then
        PRINT_LOG "INFO" "The old install path is not exist, osconfig is not install, no need to backup."

        if [ ! -f "${nobackup_flag}" ]
        then
            PRINT_LOG "INFO" "The file <${nobackup_flag}> is not exist, create it."
            touch "${nobackup_flag}" || return $?
            chmod 600 "${nobackup_flag}"
        fi
        return 0
    fi

    #Note: 防止C00升级C10场景，多次升级，第二次升级重复备份的问题
    local is_required="false"
    if [ ! -z "${old_version}" ]
    then
        local old_version_flag="${backup_path}/src_version_${old_version}"
        ls "${old_version_flag}" &> /dev/null || is_required="true"
    fi

    if [ -f "${nobackup_flag}" -a "${is_required}" = "false" ]
    then
        PRINT_LOG "INFO" "The no backup flag file is exist, no need to backup."
        return 0
    fi

    return 1
}

function fn_backup_ostool()
{
    PRINT_LOG "INFO" "Start to backup os_tool"

    local sudo_pkg="${ABSOLUATELY_PATH}/sudobin_pkg.tar"
    local sudo_tmpPath="/tmp/osconfig_${RANDOM}"
    if [ ! -f "${sudo_pkg}" ]
    then
        PRINT_LOG "FATAL" "The sudobin_pkg.tar is not exist."
        return 1
    fi

    local sudobin_path=$(cat /usr/local/osconfig/os/conf/common.cfg | grep "SUDOBIN_PATH=" | awk -F'=' '{print $2}' | xargs)
    if [ -z "${sudobin_path}" -o ! -d "${sudobin_path}" ]
    then
        PRINT_LOG "FATAL" "The sudobin path <${sudobin_path}> does not exist."
        return 1
    fi

    sudo "${sudobin_path}/execute.sh" "${sudo_pkg}" "${sudo_tmpPath}" "upgrade.sh" "backup" "${version}" "${old_version}" &> "${TMP_CONTENT_FILE}"
    local ret_code=$?
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
    if [ $ret_code -ne 0 ]
    then
        PRINT_LOG "FATAL" "Exec the cmd <sudo \"${sudobin_path}/execute.sh\" \"${sudo_pkg}\" \"${sudo_tmpPath}\" \"upgrade.sh\" \"backup\" \"${version}\"> failure, backup os_tool failed."
        return 1
    fi

    PRINT_LOG "INFO" "End to backup os_tool"
    return 0
}


function fn_compress_osconfig()
{
    local package_file="osconfig_${version}.tar.gz"

    cd "${oldinstall_path}"
    tar -czvf "${package_file}" * &> "${TMP_CONTENT_FILE}"
    local retCode=$?
    if [ $retCode -ne 0 -a $retCode -ne 1 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "FATAL" "Exec cmd<tar -czvf ${package_file} * >failure, retCode is: $retCode."
        return 1
    fi
    cd - &>/dev/null

    mv "${oldinstall_path}/${package_file}" "${backup_path}" || return $?
    tar -tvf "${backup_path}/${package_file}" &> "${TMP_CONTENT_FILE}"
    local retCode=$?
    if [ $retCode -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "FATAL" "Exec cmd<tar -tvf ${backup_path}/${package_file}>failure, retCode is: $retCode."
        return 1
    fi

    chmod 600 "${backup_path}/${package_file}"
    PRINT_LOG "INFO" "Exec cmd<tar -czvf ${package_file} *> success." &>/dev/null
    return 0
}


function fn_backup_osconfig()
{
    PRINT_LOG "INFO" "Start to backup osconfig."
    local is_required="false"

    package_file="osconfig_${version}.tar.gz"
    if [ ! -z "${old_version}" ]
    then
        local old_version_flag="${backup_path}/src_version_${old_version}"
        ls "${old_version_flag}" &> /dev/null || is_required="true"
    fi

    #Note: 备份文件存在，不用重复备份
    if [ -f "${backup_path}/${package_file}" -a "${is_required}" = "false" ]
    then
        PRINT_LOG "INFO" "The backup file<${backup_path}/${package_file}> exist, no need to backup."
        return 0
    else
        rm -f "${backup_path}/${package_file}"
        rm -f "${backup_path}"/src_version_*
    fi

    for i in {1..3}
    do
        PRINT_LOG "INFO" "Try backup ${i} time. Please wait..."
        fn_compress_osconfig "${version}" "${backup_path}" "${oldinstall_path}" && break
        rm -f "${oldinstall_path}/${package_file}" &> /dev/null
        rm -f  "${backup_path}/${package_file}" &> /dev/null
        sleep 5
        [ "$i" = 3 ] && return 1
    done

    [ -z "${old_version_flag}" ] || touch "${old_version_flag}"

    PRINT_LOG "INFO" "Backup osconfig success."

    return 0
}

#*************************************************************#
# Name:         fn_backup
# Description:  backup function
# Parameters:   NULL
# return:       0: no need to backup
#               1: need to backup
#*************************************************************#
function fn_backup()
{
    fn_backup_precheck
    if [ $? -eq 0 ]
    then
        PRINT_LOG "INFO" "Backup precheck success, no need to backup." &>/dev/null
        return 0
    fi

    fn_backup_osconfig || return $?

    fn_backup_ostool || return $?

    return 0
}

#*************************************************************#
# Name:         fn_main                                       #
# Description:  main function                                 #
# Parameters:   NULL                                          #
#*************************************************************#
function fn_main()
{
    PRINT_LOG "INFO" "bash osconfig_backup.sh $*" &>/dev/null

    if [ $# -lt 8 ]
    then
        PRINT_LOG "FATAL" "Usage: osconfig_backup.sh -backuppath <path> -oldinstallpath <path> -newinstallpath <path> -version <version> -oldversion <old verson>"
        return 3
    fi

    if [ "$1" != "-backuppath" -o "$3" != "-oldinstallpath" -o "$5" != "-newinstallpath" -o "$7" != "-version" ]
    then
        PRINT_LOG "FATAL" "Usage: osconfig_backup.sh -backuppath <path> -oldinstallpath <path> -newinstallpath <path> -version <version> -oldversion <old verson>"
        return 3
    fi

    backup_path="$2"
    oldinstall_path="$4"
    newinstall_path="$6"
    version="$8"
    old_version="${10}"

    PRINT_LOG "INFO" "backup_path:<${backup_path}>, oldinstall_path:<${oldinstall_path}>, newinstall_path:<${newinstall_path}>, version:<${version}>,oldversion<${old_version}>."
    fn_check_env || return $?

    fn_check_params || return $?

    fn_backup || return $?

    return 0
}


fn_main "$@"
RET_MAIN=$?
if [[ "$RET_MAIN" == "0" ]]
then
    logger -t $USER -p local0.info "$FileName;Successful;${OPERIP:-127.0.0.1};Excute execution $FileName to backup osconfig."
else
    ERROR_CODE=$(echo ${ERROR_CODE} | xargs -n1 | sort -u | xargs)
    [ ! -z "${ERROR_CODE}" ] && PRINT_LOG "INFO" "Error code: ${ERROR_CODE}"
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Excute execution $FileName to backup osconfig."
fi
[ -f "${TMP_CONTENT_FILE}" ] && rm -f "${TMP_CONTENT_FILE}" &> /dev/null
FUNC_CLEAN_TMP_FILE &> /dev/null
exit ${RET_MAIN}