#!/bin/bash
#get absolute path 
FileName="`basename $0`"
RELATIVE_PATH="$(dirname $0)"
cd "${RELATIVE_PATH}"
ABSOLUATELY_PATH=`pwd`

MODULE_NAME="`basename $0 | sed 's/\.sh$//'`"
#import common variable..
. ./common.inc
cd - &> /dev/null


function fn_check_params()
{
    local backup_path="$1"
    local newinstall_path="$2"
    local oldsudobin_path="$3"
    local version="$4"
    
    PRINT_LOG "INFO" "Start to check input params." &>/dev/null
    
    if [ -z "${newinstall_path}" ]
    then
        PRINT_LOG "FATAL" "The install path is empty."
        return 3
    fi
    
    if [ -z "${version}" ]
    then
        PRINT_LOG "FATAL" "The version<${version}> is empty."
        return 3
    fi

    if [[ ! "${version}" =~ ^[a-zA-Z0-9]+$ ]]
    then
        PRINT_LOG "FATAL" "The ${version} is invalid, the version only contain the chars [a-zA-Z0-9]."
        return 3
    fi
    
    #Note：如果备份路径不存在，不回滚. R7版本的备份路径${backup_path}/xxx.tar.gz, R21之后备份路径为${backup_path}/${version}/xxx.tar.gz
    roll_to_r7=True
    [ ! -f "${backup_path}/osconfig_${version}.tar.gz" ] && { backup_path="${backup_path}/${version}"; roll_to_r7=False; }

    if [ ! -d "${backup_path}" ]
    then
        PRINT_LOG "WARN" "The backup path is not exist." &>/dev/null
        return 1
    fi
    
    tar -tvf "${backup_path}/osconfig_${version}.tar.gz" &> "${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "FATAL" "Exec cmd<tar -tvf ${backup_path}/osconfig_${version}.tar.gz>failure, retCode is: ${ret_code}."
    fi

    PRINT_LOG "INFO" "Check input params done." &>/dev/null
    return 0
}


function fn_clean_sudo_script()
{
    local sudopath="$1"
    local tmp_flag="$2"
    PRINT_LOG "INFO" "Start to clean install sudo script." &> /dev/null
    sudo "${sudopath}"/clean_osconfig_gpg_check_sudo.sh "${tmp_flag}" &> "${TMP_CONTENT_FILE}"
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
    PRINT_LOG "INFO"  "Clean install sudo script done." &> /dev/null
}


function fn_uninstall_osconfig()
{
    fn_disabl_features "${newinstall_path}"
    bash "${newinstall_path}/os/bin/uninstall.sh" "upgrade" &> ${TMP_CONTENT_FILE}
    if [ $? -ne 0 ]
    then
        PRINT_LOG "INFO" "Start to clear the old data of osconfig... "
        tmp_content=$(find /usr/local/osconfig | xargs -I {} stat -c "%A, %U:%G @@ access=%x, modify=%y,  change=%z %n" {} 2>&1)
        PRINT_LOG "INFO" "file list=${tmp_content}" &> /dev/null
        ls /usr/local/osconfig | grep -vw -e "logs" -e "etc" | xargs -I {} chmod -R 750 /usr/local/osconfig/{}
        ls /usr/local/osconfig | grep -vw -e "logs" -e "etc" | xargs -I {} rm -rf /usr/local/osconfig/{}

        if [ $? -ne 0 ]
        then
            ERROR_CODE="${ERROR_CODE} UpgradeOsconfig000004 "
            PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
            PRINT_LOG "INFO" "Exec< rm -rf /usr/local/osconfig/files> failure." 
            PRINT_LOG "FATAL" "Execute osconfig uninstall failed."
            return 1
        fi
        
        PRINT_LOG "INFO" "The osconfig old data is cleared. "
    fi

    PRINT_LOG "INFO" "Uninstall new install osconfig."
    return 0
}


function fn_rollback()
{
    local backup_path="$1"
    local oldinstall_path="$2"
    local newinstall_path="$3"
    local oldsudobin_path="$4"
    local version="$5"
    
    PRINT_LOG "INFO" "Start to rollback osconfig."
    
    #Note:uninstall newinstall osconfig
    PRINT_LOG "INFO" "Start to uninstall new install osconfig."
    
    #Note:避免反复回滚报错C00->C10
    if [ ! -d "${newinstall_path}" ]
    then
        PRINT_LOG "WARN" "The osconfig install path is not exist, no need to rollback."
        return 0
    fi

    local is_null_dir=$(ls -A "${newinstall_path}")
    if [ -z "${is_null_dir}" ]
    then
        PRINT_LOG "WARN" "The installation directory is empty. "
        return 0
    fi

    #Note: no backup flag file exist, no need to rollback osconfig
    local nobackup_flag="${backup_path}/nobackup_${version}"
    if [ -f "${nobackup_flag}" ]
    then
        fn_uninstall_osconfig || return 1
        PRINT_LOG "INFO" "The no backup flag file is exist, so no need to rollback osconfig."
        return 0
    fi
    
    backup_file="${backup_path}/osconfig_${version}.tar.gz"
    if [ ! -f "${backup_file}" ]
    then
        PRINT_LOG "FATAL" "The backup file <${backup_file}> is not exist." &>/dev/null
        return 1
    fi

    fn_uninstall_osconfig || return 1
    PRINT_LOG "INFO" "Start to restore osconfig data."
    
    return 0
}


function fn_install_config()
{
    local backup_path="$1"
    local oldinstall_path="$2"
    local newinstall_path="$3"
    local oldsudobin_path="$4"
    local version="$5"
    
    #Note:避免反复回滚报错C00->C10
    get_os_type="${oldinstall_path}/os/bin/getostype.sh"
    if [ -f "${get_os_type}" ]
    then
        get_os_type=$(bash "${get_os_type}")
        if [ ! -z "${get_os_type}" ]
        then
            PRINT_LOG "FATAL" "Exec<getostype.sh> failed."
            return 1
        fi

        PRINT_LOG "WARN" "The osconfig install path is not exist, no need to rollback."
        return 0
    fi

    PRINT_LOG "INFO" "Start to restore osconfig data."
    
    #Note: install sudo to create osconfig install path(/usr/local/osconfig)
    local sudo_pkg="${ABSOLUATELY_PATH}/sudobin_pkg.tar"
    local tmp_sudo_path="/root/osconfig_${RANDOM}"
    sudo "${oldsudobin_path}/execute.sh" "${sudo_pkg}" "${tmp_sudo_path}" "install.sh" "${oldsudobin_path}" &> "${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        PRINT_LOG "FATAL" "Config framework sudo script failure."
        return 1
    fi
    PRINT_LOG "INFO" "Install framework sudo script done."
    
    fn_clean_sudo_script "${oldsudobin_path}" "rollback"
    backup_file="${backup_path}/osconfig_${version}.tar.gz"
    cp "${backup_file}" "${oldinstall_path}" || return $?

    cd "${oldinstall_path}"
    exclude_list="--exclude=logs --exclude=os/bin/getntpinfo.sh --exclude=os/bin/checkntpstatus.sh "
    tar -zxf "${oldinstall_path}/osconfig_${version}.tar.gz" ${exclude_list} &> ${TMP_CONTENT_FILE}
    local retCode=$?
    if [ $retCode -ne 0 -a $retCode -ne 1 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "FATAL" "Exec cmd<tar -zxf \"${oldinstall_path}/osconfig_${version}.tar.gz\">failure, retCode is: $retCode."
        return 1
    fi
    cd - &>/dev/null

    PRINT_LOG "INFO" "Exec cmd<tar -zxf \"${oldinstall_path}/osconfig_${version}.tar.gz\"> success." &>/dev/null
    
    #Note:install osconfig sudo
    local ostype=$(bash /usr/local/osconfig/os/bin/getostype.sh)
    if [ -z "${ostype}" ]
    then
        PRINT_LOG "FATAL" "Execute getostype.sh script failure."
        return 1
    fi
    PRINT_LOG "INFO" "The os type is ${ostype}" &> /dev/null
    
    bash /usr/local/osconfig/os/${ostype}/install.sh "${oldsudobin_path}" &> ${TMP_CONTENT_FILE}
    if [ $? -ne 0 ]
    then
        PRINT_LOG "FATAL" "Execute the install.sh in ${ostype} folder failure."
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        return 1
    fi

    #Note: rollback os_tool
    sudo "${oldsudobin_path}/execute.sh" "${sudo_pkg}" "${tmp_sudo_path}" "upgrade.sh" "rollback" "${version}" &> "${TMP_CONTENT_FILE}"
    local ret_code=$?
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
    if [ $ret_code -ne 0 ]
    then
        PRINT_LOG "FATAL" "Exec cmd <sudo ${oldsudobin_path}/execute.sh ${sudo_pkg} ${tmp_sudo_path} upgrade.sh rollback ${version}> failure, rollback ostool failed."
        return 1
    fi

    cd "${oldinstall_path}"
    chmod 700 "/usr/local/osconfig/os/bin"
    tar -zxf "${oldinstall_path}/osconfig_${version}.tar.gz" os/bin/getntpinfo.sh os/bin/checkntpstatus.sh &> ${TMP_CONTENT_FILE}
    local retCode=$?
    chmod 550 "/usr/local/osconfig/os/bin"
    if [ $retCode -ne 0 -a $retCode -ne 1 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "FATAL" "Exec cmd<tar -zxf \"${oldinstall_path}/osconfig_${version}.tar.gz\">failure, retCode is: $retCode."
        return 1
    fi
    cd - &>/dev/null

    rm -f "${oldinstall_path}/osconfig_${version}.tar.gz" &>/dev/null
    PRINT_LOG "INFO" "Rollback osconfig done."
    
    return 0
}


#*************************************************************#
# Name:         fn_main                                       #
# Description:  main function                                 #
# Parameters:   NULL                                          #
#*************************************************************#
function fn_main()
{
    if [ $# -lt 10 ]
    then
        PRINT_LOG "FATAL" "Usage: osconfig_rollbackup.sh -backuppath <path> -oldinstallpath <path> -newinstallpath <path> -oldsudobin <path> -version <version>"
        return 3
    fi
    
    if [ "$1" != "-backuppath" -o "$3" != "-oldinstallpath" -o "$5" != "-newinstallpath" -o "$7" != "-oldsudobin" -o "$9" != "-version" ]
    then
        PRINT_LOG "FATAL" "Usage: osconfig_rollbackup.sh -backuppath <path> -oldinstallpath <path> -newinstallpath <path> -oldsudobin <path> -version <version>"
        return 3
    fi
    
    backup_path="$2"
    oldinstall_path="$4"
    newinstall_path="$6"
    oldsudobin_path="$8"
    version="${10}"
    
    PRINT_LOG "INFO" "backup_path:<${backup_path}>, oldinstall_path:<${oldinstall_path}>, newinstall_path:<${newinstall_path}>, oldsudobin_path:<${oldsudobin_path}>, version:<${version}>."
    fn_check_env || return $?
    fn_check_params "${backup_path}" "${newinstall_path}" "${oldsudobin_path}" "${version}" || return $?
    [[ "${roll_to_r7}" == False ]] && backup_path="${backup_path}/${version}"

    fn_rollback "${backup_path}" "${oldinstall_path}" "${newinstall_path}" "${oldsudobin_path}" "${version}" || return $?

    fn_install_config "${backup_path}" "${oldinstall_path}" "${newinstall_path}" "${oldsudobin_path}" "${version}" || return $?

    fn_clean_old_register

    # start osconfigserver and register OsconfigService
    if [ -f "/usr/local/osconfig/os/bin/osconfigserver.sh" ]
    then
        bash /usr/local/osconfig/os/bin/osconfigserver.sh restart
        if [[ $? -ne 0 ]]
        then
            PRINT_LOG "FATAL" "start osconfig server failure"
            return 1
        fi
        register_system_service || return $?
        add_cron || return $?
    fi
    return 0
}

function fn_clean_old_register()
{
    rm -rf /usr/local/osconfig/etc/manifest
    rm -rf /usr/local/osconfig/etc/uds
}

fn_main "$@"
RET_MAIN=$?
if [[ "$RET_MAIN" == "0" ]]
then
    logger -t $USER -p local0.info "$FileName;Successful;${OPERIP:-127.0.0.1};Excute execution $FileName to rollback osconfig."
else
    ERROR_CODE=$(echo ${ERROR_CODE} | xargs -n1 | sort -u | xargs)
    [ ! -z "${ERROR_CODE}" ] && PRINT_LOG "INFO" "Error code: ${ERROR_CODE}"
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Excute execution $FileName to rollback osconfig."
fi
[ -f "${TMP_CONTENT_FILE}" ] && rm -f "${TMP_CONTENT_FILE}" &> /dev/null
FUNC_CLEAN_TMP_FILE &> /dev/null
exit ${RET_MAIN}
