#!/bin/bash
SCRIPT_DIR_ENGR=$(cd $(dirname ${BASH_SOURCE}); pwd)
SCRIPT_NAME_ENGR=$(basename ${BASH_SOURCE})

CURRENT_WORK_PATH="${SCRIPT_DIR_ENGR%/*}"

OS_TYPE="$(basename ${CURRENT_WORK_PATH})"
export OS_TYPE

USER=$(id -un)
OPERIP=`who am i | awk '{print $NF}' | sed 's/[()]//g'`

CURRENT_USER="`id -u -n`"

if [ "${CURRENT_USER}" != "root" ]
then
    echo "Only root can run this script"
    logger -t $USER -p local0.info "${SCRIPT_NAME_ENGR};Failed;${OPERIP:-127.0.0.1};Only root can run this script."
    exit 1
fi

#Note: source the error code
. ${SCRIPT_DIR_ENGR}/error_code.inc

function getEnvPath()
{
    local key="$1"
    CUSTOM_FILE="${CURRENT_WORK_PATH}/../conf/custom.cfg"
    if [ -f "${CUSTOM_FILE}" ]
    then
        TMP_PATH_NAME=`cat ${CUSTOM_FILE} | grep "^${key}=.*$" | awk -F"=" '{print $2}' | sed 's/^\s*\|\s*$//g'`
        if [ -n "$TMP_PATH_NAME" ]
        then
            eval "$2=\${TMP_PATH_NAME}"
        fi
    fi
}


#Note: get OS_LOGPATH path from custom.cfg file
getEnvPath "OS_TOOL_LOGPATH" OS_LOGPATH
if [ -z "${OS_LOGPATH}" ]
then
    OS_LOGPATH="/var/log/os_tool"
fi

if [ ! -d "${OS_LOGPATH}" ]
then
    mkdir "${OS_LOGPATH}"
    if [ $? -ne 0 ]
    then
        echo "Make log dir ${OS_LOGPATH} failure."
        logger -t $USER -p local0.info "${SCRIPT_NAME_ENGR};Failed;${OPERIP:-127.0.0.1};Make log dir ${OS_LOGPATH} failure."
        exit 1
    fi
    chmod 700 "${OS_LOGPATH}"
fi
export OS_LOGPATH

#Note: get OS_TMPPATH path from custom.cfg file
getEnvPath "OS_TOOL_TMPPATH" OS_TMPPATH
if [ -z "${OS_TMPPATH}" ]
then
    OS_TMPPATH="/var/log/os_tool/tmp"
fi

if [ ! -d "${OS_TMPPATH}" ]
then
    mkdir "${OS_TMPPATH}"
    if [ $? -ne 0 ]
    then
        echo "Make tmp dir ${OS_TMPPATH} failure."
        logger -t $USER -p local0.info "${SCRIPT_NAME_ENGR};Failed;${OPERIP:-127.0.0.1};Make tmp dir ${OS_TMPPATH} failure."
        exit 1
    fi
    chmod 700 "${OS_TMPPATH}"
fi
export OS_TMPPATH

#Note: get ROOTPATH path from custom.cfg file
getEnvPath "OS_TOOL_INSTALL_ROOT" OS_ROOTPATH
if [ -z "${OS_ROOTPATH}" ]
then
    OS_ROOTPATH="/usr/local/os_tool"
fi
export OS_ROOTPATH

BACKUP_PATH="${OS_ROOTPATH}/backup"
if [ ! -d "${BACKUP_PATH}" ]
then
    mkdir ${BACKUP_PATH}
    chmod 700 "${BACKUP_PATH}"
fi

HARDEN_BACKUP_PATH="${BACKUP_PATH}/hardenos"
if [ ! -d "${HARDEN_BACKUP_PATH}" ]
then
    mkdir ${HARDEN_BACKUP_PATH}
    chmod 700 "${HARDEN_BACKUP_PATH}"
fi
export HARDEN_BACKUP_PATH

CONFIG_BACKUP_PATH="${BACKUP_PATH}/configos"
if [ ! -d "${CONFIG_BACKUP_PATH}" ]
then
    mkdir ${CONFIG_BACKUP_PATH}
    chmod 700 "${CONFIG_BACKUP_PATH}"
fi
export CONFIG_BACKUP_PATH

if [ -z "${MODULE_NAME}" ]
then
    MODULE_NAME="default"
fi

CONSTANT_LOGFILE="${OS_LOGPATH}/scripts/${MODULE_NAME}/${MODULE_NAME}.log"
GLOBAL_BASH_PID=$$
DATE_PLUS_INFO="`date '+%Y%m%d%H%M%S'`"
RANDOM_INFO="${DATE_PLUS_INFO}${RANDOM}"
TMP_CONFIG_PATH="${OS_LOGPATH}/scripts/${MODULE_NAME}/tmp/"
TMP_CONFIG_FILE="${TMP_CONFIG_PATH}/tmp_config_${RANDOM_INFO}.log"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmp_content_${RANDOM_INFO}.log"

if [ ! -d "${TMP_CONFIG_PATH}" ]
then
    mkdir -p "${TMP_CONFIG_PATH}" &> /dev/null
    chmod 750 "${TMP_CONFIG_PATH}" &> /dev/null
fi

#*************************************************************#
# Name        : FUNC_CLEAN_TMP_FILE
# Description : clean tmp file and folder
# Parameters  :
#*************************************************************#
function FUNC_CLEAN_TMP_FILE()
{
    [ -f ${TMP_CONFIG_FILE} ] && rm -f ${TMP_CONFIG_FILE} &> /dev/null
    [ -f ${TMP_CONTENT_FILE} ] && rm -f ${TMP_CONTENT_FILE} &> /dev/null
    [ -d ${TMP_CONFIG_PATH} ] && rm -rf ${TMP_CONFIG_PATH} &> /dev/null
    fn_remove_tmp_file &> /dev/null

    return 0
}

#*************************************************************#
# Name        : PRINT_FILE_TO_LOG
# Description : print file content into log file
# Parameters  : $1 the file to be printed
#*************************************************************#
function PRINT_FILE_TO_LOG()
{
    local file_name=$1

    LOGFILE="${LOGFILE:-${CONSTANT_LOGFILE}}"

    if [ -f "${file_name}" ]
    then
        local check_content=`cat ${file_name}`
        if [ -z "${check_content}" ]
        then
            LOG "This file <${file_name}> is empty."
            return 1
        else
            LOG "Print <${file_name}>"
            LOG "`cat \"${file_name}\" | sed 's/\r//g'`"
        fi
    else
        LOG "This file <${file_name}> does not exist."
        return 1
    fi
}

#*************************************************************#
# Name:        WARN
# Description: record the warning message into the logfile
#*************************************************************#
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************#
# Name:        ERROR
# Description: record the error message into the logfile
#*************************************************************#
function ERROR()
{
    LOG "ERROR: $*"
}

# Name        : PRINT_LOG
# Description : print log
function PRINT_LOG()
{
    local LOG_LEVEL=$1
    local LOG_INFO=$2

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "[Line-${BASH_LINENO}] ${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}


#Note: get single log file size: unit M
function fn_get_shell_log_size()
{
    local log_size=""
    log_size=$(cat /usr/local/os_tool/os/conf/custom.cfg 2>/dev/null  | grep "MAX_SHELL_LOG_SIZE=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_size}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_size}"
    fi
}

function fn_get_shell_log_num()
{
    local log_num=""
    log_num=$(cat /usr/local/os_tool/os/conf/custom.cfg 2>/dev/null | grep "MAX_SHELL_LOG_NUM=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_num}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_num}"
    fi
}


#*************************************************************#
# Name:         LOG
# Description: record the message into the logfile
#*************************************************************#
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSizeM=`fn_get_shell_log_size`
    [ -z "${maxSizeM}" ] && maxSizeM="2"
    local maxSize=$((${maxSizeM} * 1024 * 1024))
    local maxRecord=`fn_get_shell_log_num`
    [ -z "${maxRecord}" ] && maxRecord="5"
    local index=`expr $maxRecord - 1`
    local content="$*"

    if [  -f ${CONSTANT_LOGFILE} ]
    then
        curSize=`ls -lk ${CONSTANT_LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${CONSTANT_LOGFILE}.${index}.gz ]
                then
                    echo y | mv ${CONSTANT_LOGFILE}.${index}.gz ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                    chmod 440 ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                fi
                let index--
            done
            echo y | mv ${CONSTANT_LOGFILE} ${CONSTANT_LOGFILE}.1
            gzip ${CONSTANT_LOGFILE}.1
            chmod 440 ${CONSTANT_LOGFILE}.1.gz
        fi
        if echo "${content}" | grep -w "^WARN:" &> /dev/null
        then
            fn_echo_warn "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        elif echo "${content}" | grep -w "^ERROR:" &> /dev/null
        then
            fn_echo_fail "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        else
            echo -e "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        fi
    else
        mkdir -p `dirname ${CONSTANT_LOGFILE}`
        chmod 750 `dirname ${CONSTANT_LOGFILE}`
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${CONSTANT_LOGFILE}
    fi
}

#*************************************************************#
# Name:        getSysTime
# Description: get the system time
#*************************************************************#
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}


#*************************************************************#
# Name        : fn_echo_succ
# Description : print text with color -- green
# Parameters  : $1 -- text
#*************************************************************#
function fn_echo_succ()
{
    FG=32 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

#*************************************************************#
# Name        : fn_echo_succ
# Description : print text with color -- yellow
# Parameters  : $1 -- text
#*************************************************************#
function fn_echo_warn()
{
    FG=33 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}


#*************************************************************#
# Name        : fn_echo_succ
# Description : print text with color -- red
# Parameters  : $1 -- text
#*************************************************************#
function fn_echo_fail()
{
    FG=31 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}


#*************************************************************#
# Name        : fn_get_value
# Description : get the config
# Parameters  : $1 the absoluately path of config file
#               $2 the key name
#*************************************************************#
function fn_get_value()
{
    local TMP_CONFIG_FILE=$1
    local TMP_KEY_NAME=$2

    cat "${TMP_CONFIG_FILE}" | grep -m1 "^${TMP_KEY_NAME}\s*=" | awk 'BEGIN {FS="="}{print $2}' | xargs
}