#!/bin/bash

#Note:import error code
. /usr/local/osconfig/os/default/bin/error_code.inc &> /dev/null

#---------------------------------------------------#
#-------------- constant variable ------------------#
#---------------------------------------------------#

if [ -z "${MODULE_NAME}" ]
then
    MODULE_NAME="default"
fi
USER=$(id -un)
OPERIP=$(who am i | awk '{print $NF}' | sed 's/[()]//g')

CONSTANT_LOGFILE="/usr/local/osconfig/logs/scripts/${MODULE_NAME}/${MODULE_NAME}.log"
GLOBAL_BASH_PID=$$
DATE_PLUS_INFO="`date '+%Y%m%d%H%M%S'`"
RANDOM_INFO="${DATE_PLUS_INFO}${RANDOM}"
TMP_CONFIG_PATH="/usr/local/osconfig/logs/scripts/${MODULE_NAME}/tmp/tmp_$$"
TMP_CONFIG_FILE="${TMP_CONFIG_PATH}/tmp_config_${RANDOM_INFO}.log"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmp_content_${RANDOM_INFO}.log"

umask 027

if [ ! -d "${TMP_CONFIG_PATH}" ]
then
    mkdir -p "${TMP_CONFIG_PATH}" &> /dev/null
    chmod 750 "${TMP_CONFIG_PATH}" &> /dev/null
fi

#Note:get the sudobin path
SUDOBIN_PATH=$(cat /usr/local/osconfig/os/conf/common.cfg | grep "SUDOBIN_PATH=" | awk -F'=' '{print $2}' | xargs)
if [ -z "${SUDOBIN_PATH}" ]
then
    SUDOBIN_PATH="/usr/local/uniepsudobin"
fi

#Note:export python home and python path
PYTHONHOME="/usr/local/osconfig/python"
export PYTHONHOME
NAMESPACE_ROOT="/opt/oss/manager/"
PYTHONPATH="${PYTHONPATH}:/usr/local/osconfig/os/pyscripts:/usr/local/osconfig/os/default/pyscripts"
export PYTHONPATH
OSADAPTER_ROOT="/usr/local/osconfig/os"
export OSADAPTER_ROOT
OSCONFIG_LOG_DIR="/usr/local/osconfig/logs/pyscripts"
export OSCONFIG_LOG_DIR

#------------------------------------------------------------#
#-----------------------functions----------------------------#
#------------------------------------------------------------#

SSH="ssh -n -o StrictHostKeyChecking=no -o BatchMode=yes -o ConnectTimeout=30"
SCP="scp -rp -o StrictHostKeyChecking=no -o BatchMode=yes -o ConnectTimeout=30"


function remove_repeat_path()
{
    local newitems=""
    IFS_t=$IFS
    IFS=:
    items=$1
    for item in $items
    do
        exist=0
        for ni in $newitems
        do
            if [ "$item" = "$ni" ];
            then
                exist=1
                break
            fi
        done
        if [ $exist -eq 0 ]
        then
            if [ "$newitems" = "" ];
            then
                newitems=$item
            else
                newitems=$newitems:$item
            fi
        fi
    done
    IFS=$IFS_t
    eval $2="\$newitems"
}

PATH="/sbin:/usr/sbin:/usr/local/sbin:/usr/local/bin:/usr/bin:/bin:${PATH}"
remove_repeat_path "${PATH}" tmpPath
export PATH=$tmpPath


#========================================================================#
# Note: define common function
# clean temporary files
function FUNC_CLEAN_TMP_FILE()
{
    [ -f ${TMP_CONFIG_FILE} ] && rm -f ${TMP_CONFIG_FILE} &> /dev/null
    [ -f ${TMP_CONTENT_FILE} ] && rm -f ${TMP_CONTENT_FILE} &> /dev/null
    [ -d ${TMP_CONFIG_PATH} ] && rm -rf ${TMP_CONFIG_PATH} &> /dev/null
    fn_remove_tmp_file &> /dev/null
    
    return 0
}


# Name        : PRINT_FILE_TO_LOG
# Description : print file content into log file
# Parameters  : $1 the file to be printed
function PRINT_FILE_TO_LOG()
{
    local file_name=$1

    LOGFILE="${LOGFILE:-${CONSTANT_LOGFILE}}"
    
    if [ -f "${file_name}" ]
    then
        local check_content=`cat ${file_name}`
        if [ -z "${check_content}" ]
        then
            LOG "This file <${file_name}> is empty."
            return 1
        else
            LOG "Print <${file_name}>"
            LOG "`cat \"${file_name}\" | sed 's/\r//g' | sed '/mac/d' | sed '/MAC/d'`"
        fi
    else
        LOG "This file <${file_name}> does not exist."
        return 1
    fi
}

#*************************************************************#
# Name:        WARN                                           #
# Description: record the warning message into the logfile    #
#*************************************************************#
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************#
# Name:        ERROR                                          #
# Description: record the error message into the logfile      #
#*************************************************************#
function ERROR()
{
    LOG "ERROR: $*"
}

# Name        : PRINT_LOG
# Description : print log
function PRINT_LOG()
{
    local LOG_LEVEL=$1
    local LOG_INFO=$2

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "[Line-${BASH_LINENO}] ${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}

#Note: get single log file size: unit M
function fn_get_shell_log_size()
{
    local log_size=""
    local tmp_path="/usr/local/osconfig/os/conf/common.cfg"
    log_size=$(cat "${tmp_path}" 2>/dev/null | grep "MAX_SHELL_LOG_SIZE=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_size}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_size}"
    fi
}

function fn_get_shell_log_num()
{
    local log_num=""
    local tmp_path="/usr/local/osconfig/os/conf/common.cfg"
    log_num=$(cat "${tmp_path}" 2>/dev/null | grep "MAX_SHELL_LOG_NUM=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_num}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_num}"
    fi
}

#*************************************************************#
# Name:         LOG                                           #
# Description: record the message into the logfile            #
#*************************************************************#
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSizeM=`fn_get_shell_log_size`
    [ -z "${maxSizeM}" ] && maxSizeM="2"
    local maxSize=$((${maxSizeM} * 1024 * 1024))
    local maxRecord=`fn_get_shell_log_num`
    [ -z "${maxRecord}" ] && maxRecord="5"
    local index=`expr $maxRecord - 1`
    local content="$*"

    if [  -f ${CONSTANT_LOGFILE} ]
    then
        curSize=`ls -lk ${CONSTANT_LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${CONSTANT_LOGFILE}.${index}.gz ]
                then
                    echo y | mv ${CONSTANT_LOGFILE}.${index}.gz ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                    chmod 440 ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                fi
                let index--
            done
            echo y | mv ${CONSTANT_LOGFILE} ${CONSTANT_LOGFILE}.1
            gzip ${CONSTANT_LOGFILE}.1
            chmod 440 ${CONSTANT_LOGFILE}.1.gz
        fi
        if echo "${content}" | grep -w "^WARN:" &> /dev/null
        then
            fn_echo_warn "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        elif echo "${content}" | grep -w "^ERROR:" &> /dev/null
        then
            fn_echo_fail "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        else
            echo -e "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        fi
    else
        mkdir -p `dirname ${CONSTANT_LOGFILE}`
        chmod 750 `dirname ${CONSTANT_LOGFILE}`
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${CONSTANT_LOGFILE}
    fi
}

#*************************************************************#
# Name:        getSysTime                                     #
# Description: get the system time                            #
#*************************************************************#
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}


# print text with color -- green
# parameters:   $1 -- text
function fn_echo_succ()
{
    FG=32 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}


# print text with color -- yellow
# parameters:   $1 -- text
function fn_echo_warn()
{
    FG=33 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}


# print text with color -- red
# parameters:   $1 -- text
function fn_echo_fail()
{
    FG=31 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

function fn_check_file_link()
{
    local file="$1"
    if [ -z "${file}"  -o ! -f "${file}" ] 
    then
        PRINT_LOG "FATAL" "The parameter \"${file}\" is invalid." &> /dev/null
        return ${ERR_FILE_NO_EXIST}
    fi

    local softlink=`ls -l "${file}" |cut -c1`
    local hardlink="`stat -c %h ${file}`"
    
    if [[ "${softlink}" == "l" || ${hardlink} -ne 1 ]]
    then
        PRINT_LOG "FATAL" "The file: <${file}> is a softlink file or hardlink file" &> /dev/null
        return ${ERR_LINK_FILE}
    fi
    
    local abs_path=`readlink -m "${file}"`
    local file_path=`echo "${file}" | sed -r -e 's#/+#/#g' -e 's#/$##'`
    if [ "${abs_path}" != "${file_path}" ]
    then
        PRINT_LOG "FATAL"  "The file <${file}> exist link path." &> /dev/null
        return ${ERR_LINK_FILE}
    fi
    
    PRINT_LOG "INFO" "The file: <${file}> is not a softlink file or hardlink file"  &> /dev/null
    return ${SUCCESS_CODE}
}

function fn_check_path_link()
{
    local path="$1"
    if [ -z "${path}" ]
    then
        PRINT_LOG "FATAL" "The path <${path}> parameter is invalid." &> /dev/null
        return ${ERR_PARAMS}
    fi
    
    local abs_path=`readlink -m "${path}"`
    local format_path=`echo "${path}" | sed -r -e 's#/+#/#g' -e 's#/$##'`
    if [ "${abs_path}" != "${format_path}" ]
    then
        PRINT_LOG "FATAL" "The path <${path}> exist link path." &> /dev/null
        return ${ERR_LINK_FOLDER}
    fi
    
    PRINT_LOG "INFO" "The folder: <${path}> is not a softlink path."  &> /dev/null
    return ${SUCCESS_CODE}
}


# Name        : fn_validate_ipv4
# Description : check the ip is validate or not
# Parameters  : $1 ip address
function fn_validate_ipv4()
{
    local IP_ADDR=$1

    echo $IP_ADDR | grep -E "^([0-9]{1,3}\.){3}[0-9]{1,3}$" >/dev/null 2>&1
    if [[ $? -eq 0 ]]
    then
        IFS_TMP=$IFS
        IFS='.'
        IP_ADDR=($IP_ADDR)
        IFS=$IFS_TMP
        if [[ ${IP_ADDR[0]} -ge "0" && ${IP_ADDR[0]} -le "255" \
            && ${IP_ADDR[1]} -ge "0" && ${IP_ADDR[1]} -le "255" \
            && ${IP_ADDR[2]} -ge "0" && ${IP_ADDR[2]} -le "255" \
            && ${IP_ADDR[3]} -ge "0" && ${IP_ADDR[3]} -le "255" ]]
        then
            PRINT_LOG "INFO" "The IP_ADDR<${IP_ADDR[0]}.${IP_ADDR[1]}.${IP_ADDR[2]}.${IP_ADDR[3]}> is OK!"
            return 0;
        fi
    fi

    PRINT_LOG "WARN" "The IP_ADDR<${IP_ADDR[0]}.${IP_ADDR[1]}.${IP_ADDR[2]}.${IP_ADDR[3]}> is invalid!"
    return 102;
}

# Name        : fn_validate_ipv6
# Description : check the ipv6 is validate or not
# Parameters  : $1 ipv6 address
function fn_validate_ipv6()
{
    local IPV6_ADDR="$1"
    
    if echo "${IPV6_ADDR}" | grep "\."
    then
        local ip_last_segment=`echo ${IPV6_ADDR} | awk -F':' '{print $NF}'`
        fn_validate_ip "${ip_last_segment}" >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "The IP_ADDR<${IPV6_ADDR}> is invalid!"
            return ${ERROR_IP_INVALID}
        fi
        
        IFS_TMP=$IFS
        IFS=':'
        IP_ADDR=($IPV6_ADDR)
        IFS=$IFS_TMP

        local ipv6_segment_len=${#IP_ADDR[@]}
        local ipv6_len_exclude_ipv4=`expr ${ipv6_segment_len} - 1`
        local ipv6_exclude_ipv4=`echo "${IPV6_ADDR}" | cut -d : -f 1-${ipv6_len_exclude_ipv4}`
        
        echo "${ipv6_exclude_ipv4}" | egrep -i "^(([A-F0-9]{1,4}:){5}[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){4}:[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){3}(:[A-F0-9]{1,4}){1,2}\
|([A-F0-9]{1,4}:){2}(:[A-F0-9]{1,4}){1,3}\
|[A-F0-9]{1,4}:(:[A-F0-9]{1,4}){1,4}\
|([A-F0-9]{1,4}:){1,5}\
|:(:[A-F0-9]{1,4}){1,5}\
|:)$"
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "The IP_ADDR<${IPV6_ADDR}> is invalid!"
            return ${ERROR_IP_INVALID}
        fi
    else
        echo "${IPV6_ADDR}" | egrep -i "^(([A-F0-9]{1,4}:){7}[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){6}:[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){5}(:[A-F0-9]{1,4}){1,2}\
|([A-F0-9]{1,4}:){4}(:[A-F0-9]{1,4}){1,3}\
|([A-F0-9]{1,4}:){3}(:[A-F0-9]{1,4}){1,4}\
|([A-F0-9]{1,4}:){2}(:[A-F0-9]{1,4}){1,5}\
|[A-F0-9]{1,4}:(:[A-F0-9]{1,4}){1,6}\
|([A-F0-9]{1,4}:){1,7}:\
|:(:[A-F0-9]{1,4}){1,7}\
|::)$"
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "The IP_ADDR<${IPV6_ADDR}> is invalid!"
            return ${ERROR_IP_INVALID}
        fi
    fi
    
    PRINT_LOG "INFO" "The IP_ADDR<${IPV6_ADDR}> is OK!"
    return 0
}

function fn_validate_ip()
{
    local IP_ADDR=$1
    if [ -z "${IP_ADDR}" ]
    then
        PRINT_LOG "FATAL" "The ip address is empty."
        return 1
    fi
    
    echo "${IP_ADDR}" | grep ":" &> /dev/null
    if [[ $? -ne 0 ]]
    then
        fn_validate_ipv4 "${IP_ADDR}" 
        if [ $? -eq 0 ]
        then
            echo "validate ipv4 <${IP_ADDR}> success."
            return 0
        fi
    else
        fn_validate_ipv6 "${IP_ADDR}"
        if [ $? -eq 0 ]
        then
            echo "validate ipv6 <${IP_ADDR}> success."
            return 0
        fi
    fi
    
    PRINT_LOG "FATAL" "validate ip <${IP_ADDR}> failure."
    return 1
}


function record_syslog()
{
    local ret_code="$1"
    local script_name="$2"
    if [ ${ret_code} -eq 0 ]
    then
        logger -t ${USER} -p local0.info "${script_name};Successful;${OPERIP:-127.0.0.1};Execute script ${script_name}"
    else
        logger -t ${USER} -p local0.err "${script_name};Failed;${OPERIP:-127.0.0.1};Execute script ${script_name}"
    fi
}

function fn_invoke_sudo_pkg()
{
    local sudobinpath="$1"
    local tmp_sudo_pkg="$2"
    local tmp_cmd="$3"
    local os_type="$4"
    local sudotmpPath="/root/osconfig_${RANDOM}"
    local tmp_msg="${tmp_cmd}"
    [ "${tmp_cmd}" = "install.sh" ] && tmp_msg="Install"
    [ "${tmp_cmd}" = "uninstall.sh" ] && tmp_msg="Uninstall"
    sudo ${sudobinpath}/execute.sh "${tmp_sudo_pkg}" "${sudotmpPath}" "${tmp_cmd}" "${sudobinpath}" ${os_type} \
    &> "${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        PRINT_LOG "FATAL" "${tmp_msg} sudo script failure."
        return 1
    fi

    return 0
}

function fn_check_argument()
{
    local script_name=$@

    if [ $# -ne 2 -a $# -ne 4 -a $# -ne 6 ]
    then
        echo "Usage: bash ${SCRIPT_NAME} [-input /tmp/input.json] -output /tmp/output.json [-action_type upgrade]"
        return 1
    fi

    hasOuput=false
    for var in $@
    do
       if [ $var == "-output" ]
       then
           hasOuput=true
           break
       fi
    done

    if [ $hasOuput == false ]
    then
        echo "Usage: bash ${SCRIPT_NAME} [-input /tmp/input.json] -output /tmp/output.json [-action_type upgrade]"
        return 1
    fi
    return 0
}

