#!/bin/bash
. ./controller.inc

################################################################################
# Function    : fn_validate_rpm_list
# Description : validate if os contains required components
# parameter list: rpm_file_direction
# Output      :
# Return      : 0 success
#               1 failed
################################################################################
function fn_validate_rpm_list
{
    typeset -i error_comphonent_fail_num=0
    local rpm_file_direction=$1
    local os_rpm_require_list=`cat ${rpm_file_direction}`

    for rpm_component in ${os_rpm_require_list}
    do
        rpm -q ${rpm_component} > /dev/null
        check_result_flag=$?

        if [[ ${check_result_flag} -ne 0 ]];then
            error_comphonent_fail_num=`echo "${error_comphonent_fail_num}+1" | bc`
            print_log "ERROR" "${rpm_component} is not found on the node."
         fi
    done

   if [ "${error_comphonent_fail_num}" -gt 0 ]
   then
   print_log "ERROR" "error_comphonent_fail_num : ${error_comphonent_fail_num} ."
       return 1
   fi

	return 0
}

################################################################################
# Function    : fn_get_rpm_list_location
# Description : get os rpm list file location
# parameter list:os_type, os_subversion
# Output      : global rpm_file_direction
# Return      : 0 success
#               1 failed
################################################################################
function fn_get_rpm_list_location
{
    local os_type=$1
    local os_subversion=$2
    rpm_file_direction=""
    # os_type找不到时的默认路径
    local rpm_location_ostype_common="rpm_list/common"
    # os_subversion找不到时的默认路径
    local rpm_location_ossubversion_common="rpm_list/${os_type}/common"

	# 不同架构版本
	arch=`uname -m`

    # rpm_list正常路径
    local rpm_location="rpm_list/${os_type}/${os_subversion}"
    # 按照rpm_location>rpm_location_ossubversion_common>rpm_location_ostype_common>${rpm_location}/${arch}的优先级获取rpm包路径
    cat ${rpm_location_ostype_common}/rpm_list.txt 1>/dev/null 2>/dev/null && rpm_file_direction=${rpm_location_ostype_common}/rpm_list.txt
    cat ${rpm_location_ossubversion_common}/rpm_list.txt 1>/dev/null 2>/dev/null && rpm_file_direction=${rpm_location_ossubversion_common}/rpm_list.txt
    cat ${rpm_location}/rpm_list.txt  1>/dev/null 2>/dev/null && rpm_file_direction=${rpm_location}/rpm_list.txt
	cat ${rpm_location}/${arch}/rpm_list.txt 1>/dev/null 2>/dev/null && rpm_file_direction=${rpm_location}/${arch}/rpm_list.txt

    if [[ ${rpm_file_direction} == "" ]];then
	    print_log "WARING" "rpm file \"rpm_list.txt\" is not found in localtion ${rpm_location}, ${rpm_location_ossubversion_common} or ${rpm_location_ostype_common}."
        return 1
    fi
    print_log "rpm_file_direction is ${rpm_file_direction}."
    echo "rpm_file_direction is ${rpm_file_direction}" 1>/dev/null
    return 0
}

################################################################################
# Function    : fn_get_os_type_subversion
# Description : get os version and subversion.
# parameter list:global variable os_type os_subversion
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_get_os_type_subversion
{
    local os_release_info=`cat /etc/os-release 2>/dev/null`
    local system_release_info=`cat /etc/system-release 2>/dev/null`
    os_type=`echo ${os_release_info} | egrep -o ' ID=\"[a-zA-Z]+\"' | awk -F '"' '{print $2}'`
    case ${os_type} in
        'sles') os_subversion=`echo ${os_release_info} | egrep -o "VERSION_ID=\"[0-9.]+\"" | egrep -o "[0-9.]+"`
        ;;
        'kylin') os_subversion=`echo ${os_release_info} | egrep -o "VERSION_ID=\"[a-zA-Z0-9.]+\"" | awk -F"\"" '{print $2}'`
        ;;
        'euleros') os_subversion=`echo ${system_release_info} | egrep -o "[0-9.]+ \([a-zA-Z0-9_]+\)" | tr -d ' '`
        ;;
        'rhel') os_subversion=`echo ${system_release_info} | egrep -o "[0-9.]+"`
        ;;
        'centos') os_subversion=`echo ${system_release_info} | egrep -o "[0-9]\.[0-9]"`
        ;;
    esac

	print_log "INFO" "OS version get succesfully. Os type is ${os_type}, os subversion is ${os_subversion}."
    return 0
}

################################################################################
# Function    : check_suse_nscd_version
# Description : Check the version of the nscd component in the SUSE operating system.
# parameter list:null
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function check_suse_nscd_version(){
    [ "${os_type}" != "sles" ] && return 0

    if [ "${os_subversion}" = "12.4" ];then
        # nscd组件最低版本要求：SUSE 12SP4：2.22-114.8.3
        local target_version="nscd-2.22-114.8.3.x86_64"
    elif [ "${os_subversion}" = "12.5" ];then
        # nscd组件最低版本要求：SUSE 12SP5：2.22-109.2
        local target_version="nscd-2.22-109.2.x86_64"
    else
        # 其他版本跳过
        return 0
    fi
    # 环境nscd组件版本
    local current_version="$(rpm -q nscd)"
    # 比较nscd组件版本
    local later_version="$(echo -e "${current_version}\n${target_version}" | sort -Vr | head -1)"
    # 如果当前版本是最两者中较高的,则满足要求
    if [ "${current_version}" = "${later_version}" ];then
        print_log "INFO" "The nscd component version:${current_version} meet the requirements."
        return 0
    fi
    print_log "ERROR" "The nscd component version:${current_version} is lower than ${target_version}"
    print_log "ERROR" "During the upgrade process, there is a low probability that a known bug in nscd will be triggered, which will cause nscd to deny service and cause the upgrade to fail."
    print_log "ERROR" "It is recommended to upgrade nscd to: ${target_version} or above, please contact the SUSE vendor for support."
    return 1
}


################################################################################
# Function    : check_os_components
# Description : check os components to ensure os is suitable for NCE installation.
# parameter list:null
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function check_os_components
{
    os_type=""
    os_subversion=""
    fn_get_os_type_subversion
    fn_get_rpm_list_location ${os_type} ${os_subversion}
    # 如果没有提供rpm列表，不应该报错退出
    if [[ $? -ne 0 ]];then
	    print_log "WARING" "rpm list to be check is not configured,Skip check os components."
        return 0
    fi
    fn_validate_rpm_list ${rpm_file_direction}

    if [[ $? -ne 0 ]];then
	    print_log "ERROR" "Some rpm packages is not installed on the node, please verify."
	    print_log "ERROR" "For details about how to install OS components, see the OS RPM Component Package in<iMaster NCE V100R022C00 Software Installation and Commissioning Guide>."
        return 1
    fi

    # 检查suse操作系统的nscd组件版本
    check_suse_nscd_version || return 1

	print_log "INFO" "OS components check successfully."
    return 0
}