#!/bin/bash
. ./check_disk_space.sh
. ./controller.inc


################################################################################
# Function    : fn_check_dir_mounted
# Description : check whether the target direction is mounted.
# parameter list:null
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_check_dir_mounted
{
    local dir_name=$1
    if [[ ! `df -PTh | sed  '1d' | awk '{print $7}' | egrep "^${dir_name}$"` ]];then
		print_log "ERROR" "dir \"$dir_name\" is not mount dependently"
		((error_fail_num=error_fail_num+1))
        return 1
    fi
    return 0
}




################################################################################
# Function    : fn_judge_machine_type
# Description : judge if target os is physical machine
# parameter list:null
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_judge_machine_type
{
    machineType="vm"
    if [[ ! dmidecode ]];then
    	print_log "WARNING" "The dmidecode command is not supported."
        print_log "INFO" "Deafult is vm."
        machineType="vm"
        return
    fi
    if [[ ! `dmidecode | grep 'Manufacturer' | egrep -i "VMware|OpenStack|QEMU|Standard PC"` ]]; then
        py_check=$(systemd-detect-virt 2>/dev/null | grep -i "none")
        if [ -n "${py_check}" ]; then
            print_log "INFO" "Physical machine!"
            machineType="py"
        fi
    fi
}


################################################################################
# Function    : fn_physicalmachine_modify_target
# Description : if type is physical machine, the target size should be changed.
# parameter list:null
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_physicalmachine_modify_target
{
    #是物理机，区分几块盘，改规格
    _home_size="20"
    _tmp_size="20"
    _usr_size="20"
    _var_size="30"
    _var_log_size="30"
    _var_log_audit_size="10"
    _var_tmp_size="15"

    devInfo=`ls /dev`
    print_log "INFO" "Physical machine!"
    machineType="py"
    #8块盘，包含sda,sdb,sdc
    if [[ "$devInfo" =~ "sda" && "$devInfo" =~ "sdb" && "$devInfo" =~ "sdc" ]]; then
    	print_log "INFO" "Physical machine have 8 pieces of disk."
    else
        if [[ "$devInfo" =~ "sda" && "$devInfo" =~ "sdb" ]]; then
            #7块盘
            print_log "INFO" "Physical machine have 7 pieces of disk."
        else
            if [[ "$devInfo" =~ "sda" ]]; then
                #4块盘
                print_log "INFO" "Physical machine have 4 pieces of disk."
            else
                print_log "INFO" "Physical machine not have this partition plan."
            fi
        fi
    fi
}

################################################################################
# Function    : fn_verify_partition_volume
# Description : check whether the partition volume is enough.
#               if present_dir_size(MB) is larger than target_direction_size(MB), then pass
#
# parameter list:present_dir_size(MB), target_direction_size(MB)
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_verify_partition_volume
{
    local present_dir_size=$1
    local target_dir_size=$2
    local expected_size=`echo "${target_dir_size}" | bc`
    print_log "INFO" "The available space of the install directory is ${present_dir_size}M, expected size is ${target_dir_size}M."
    if [[ `echo "${present_dir_size} < ${target_dir_size}" | bc` -eq 1 ]]; then
    	print_log "ERROR" "The available space of the install directory is less than target directory."
        ((error_fail_num=error_fail_num+1))
        return 1
    fi
    return 0
}

################################################################################
# Function    : fn_verify_opt_partition_volume
# Description : check whether the partition volume is enough.
#               if present_dir_size(MB) is larger than target_direction_size(MB), then pass
#
# parameter list:present_dir_size(MB), target_direction_size(MB)
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_opt_verify_partition_volume
{
    local present_dir_size=$1
    local target_dir_size=$2
    local calc_buffer="0.07"
    local warning_gate_MB=1000
    local expected_size=`echo "${target_dir_size}*(1-${calc_buffer})" | bc`
    print_log "INFO" "This requirement is not mandatory.The total space of the install directory is ${present_dir_size}M, expected size is ${target_dir_size}M."
    if [[ `echo "${present_dir_size} < ${target_dir_size}*(1-${calc_buffer})" | bc` -eq 1 ]]; then
        return 1
    # /opt分区较大的情况
    elif [[ `echo "${present_dir_size} + ${warning_gate_MB} < ${target_dir_size}*(1-0.02)" | bc` -eq 1 ]]; then
    	print_log "WARNING" "The space shortage is `echo "${target_dir_size} - ${present_dir_size}" | bc`MB. Space may not be enough."
    fi
    return 0
}
################################################################################
# Function    : fn_verify_rate_inode
# Description : check whether the rate of inode utilization is enough.
#               if present_inode_size(MB) is larger than target_inode_size(MB), then pass
#
# parameter list:present_inode_size(MB), target_inode_size(MB)
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function fn_verify_rate_inode
{
    local present_inode_size=$1
    local target_inode_size=$2

    print_log "INFO" "The available number of inode in the install directory is ${present_inode_size}, expected number of inode is ${target_inode_size}."
    if [[ `echo "${present_inode_size} < ${target_inode_size}" | bc` -eq 1 ]]; then
    	print_log "ERROR" "The available inode number of the install directory is less than target inode number."
        ((error_fail_num=error_fail_num+1))
        return 1
    fi
    return 0
}

################################################################################
# Function    : check_partition_space
# Description : check the free disk is enough or not.
# parameter list:null
# Output      : None
# Return      : 0 success
#               1 failed
################################################################################
function check_partition_space
{
    # 查看是虚拟机还是物理机
     fn_judge_machine_type
    # 物理机需要调整分区目标
    if [[ ${machineType} == 'py' ]];then
	    physical_opt_exist=`df -h | grep -w "/opt$"`
		if [ -z "${physical_opt_exist}" ]
		then
            print_log "INFO" "Start to set lvm..."
            bash lvm.sh >/dev/null 2>&1 || return 1
            print_log "INFO" "Set lvm successfully."
	    fi
    fi
    #
    warn_root_disk_size=0
    error_root_disk_size=0
    #the variable count the fail time of check step
    typeset -i error_fail_num=0
    typeset -i warn_fail_num=0

    #查看交换分区大小
    print_log "INFO" "--------------------Swap size---------------------"
    present_space1=`swapon -s | grep "partition" | awk '{print $3}' | awk 'BEGIN{sum=0} {sum+=$1} END{print sum}'`
    present_space=`echo "scale=1;${present_space1}/1024" | bc`
    target_space=`echo "scale=1;${swap_size}" | bc`
    fn_opt_verify_partition_volume ${present_space} ${target_space}
    print_log "INFO" "-----------------------------------------------------"

    #查看各个分区大小(除/opt)
    local dir_target_size="0"
    for dir_name in ${target_direction_list[@]}
    do
        if [[ ${dir_name} == '/var/log/audit' ]];then
            dir_target_size=${_var_log_audit_size}
            dir_target_inode_size=${_var_log_audit_inode_size}
        elif [[ ${dir_name} == '/var/log' ]];then
            dir_target_size=${_var_log_size}
            dir_target_inode_size=${_var_log_inode_size}
        elif [[ ${dir_name} == '/var/tmp' ]];then
            dir_target_size=${_var_tmp_size}
            dir_target_inode_size=${_var_tmp_inode_size}
        elif [[ ${dir_name} == '/var' ]];then
            dir_target_size=${_var_size}
            dir_target_inode_size=${_var_inode_size}
        elif [[ ${dir_name} == '/usr' ]];then
            dir_target_size=${_usr_size}
            dir_target_inode_size=${_usr_inode_size}
        elif [[ ${dir_name} == '/tmp' ]];then
            dir_target_size=${_tmp_size}
            dir_target_inode_size=${_tmp_inode_size}
        elif [[ ${dir_name} == '/home' ]];then
            dir_target_size=${_home_size}
            dir_target_inode_size=${_home_inode_size}
        elif [[ ${dir_name} == '/boot' ]];then
            dir_target_size=${_boot_size}
            dir_target_inode_size=${_boot_inode_size}
        elif [[ ${dir_name} == '/' ]];then
            dir_target_size=${_root_size}
            dir_target_inode_size=${_root_inode_size}
        fi
        print_log "INFO" "--------------------${dir_name} size---------------------"
        fn_check_dir_mounted ${dir_name}
        if [[ $? -eq 0 ]];then
            present_space=`df ${dir_name} -m -P | column -t | sed -n "2p" | awk '{print $4}'` >/dev/null 2>&1
            target_space=`echo "scale=1;${dir_target_size}" | bc`
            fn_verify_partition_volume ${present_space} ${target_space}

            if [ $? -eq 0 ]
            then
                echo "[os check successfully] Checking size successfully !"
            fi

            present_space=`df ${dir_name} -i | column -t | sed -n "2p" | awk '{print $4}'` >/dev/null 2>&1
            target_space=`echo "scale=1;${dir_target_inode_size}" | bc`
            fn_verify_rate_inode ${present_space} ${target_space}

            if [ $? -eq 0 ]
            then
                echo "[os check successfully] Checking inode successfully !"
            fi
        fi
        print_log "INFO" "-----------------------------------------------------"
    done

   if [ "${error_fail_num}" -gt 0 ]
   then
       print_log "ERROR" "Failed to check the system partition."
       return 1
    fi

    if [[ "${machineType}" == 'py' ]];then
        print_log "INFO" "--------------------/opt size---------------------"
	    py_opt_exist=$(df -PTh | sed  '1d' | awk '{print $7}' | egrep "/opt$") >/dev/null 2>&1
        if [ -z "${py_opt_exist}" ]
        then
            print_log "ERROR" "dir \"/opt\" is not mount dependently"
            error_fail_num=$(echo "${error_fail_num}+1" | bc)
        else
            present_space=$(df /opt -m -P | column -t | sed -n "2p" | awk '{print $4}') >/dev/null 2>&1
            target_space=`echo "scale=1;${_opt_minimum_size}" | bc`
            fn_verify_partition_volume ${present_space} ${target_space}

            if [ $? -ne 0 ]
            then
                error_fail_num=`echo "${error_fail_num}+1" | bc`
            else
                echo "[os check successfully] Checking OS successfully !"
            fi

            present_space=`df ${dir_name} -i | column -t | sed -n "2p" | awk '{print $4}'` >/dev/null 2>&1
            target_space=`echo "scale=1;${_opt_inode}" | bc`
            fn_verify_rate_inode ${present_space} ${target_space}

            if [ $? -ne 0 ]
            then
                error_fail_num=`echo "${error_fail_num}+1" | bc`
            else
                echo "[os check successfully] Checking OS successfully !"
            fi
            print_log "INFO" "------------------------------------------"
            return 0
        fi
    fi
    #虚拟机环境/opt
    local dir_name='/opt'
    local _opt_size=${_opt_minimum_size}
    local disk_num=`lsblk | awk '{if($6 == "disk")print $1}' | awk '/^[vs]d[a-z]/' | wc -l`
    local opt_mount_flag=0
    tmp_space=`df /tmp -m -P | column -t | sed -n "2p" | awk '{print $2}'` >/dev/null 2>&1
    var_space=`df /var -m -P | column -t | sed -n "2p" | awk '{print $2}'` >/dev/null 2>&1
    if [ "X`lsblk | grep root_bak`" != "X" ]; then
            #suse环境 系统盘增加了31G
            _except_opt_size=`echo "${_except_opt_size}+31*1024" | bc`
    fi
    if [[ $machineType == "vm" ]]; then
    	print_log "INFO" "--------------- dir '/opt' ----------------"
		if [ "X${add_disk}" == "X" ];then
            if [ "X${disk_size}" == "X" ];then
	            print_log "WARNING" "Disk size to be checked is not configured, let /opt size to be minimum."
            else
                _opt_size=`echo "scale=2;${disk_size}*1024-${_except_opt_size}"| bc`
            fi
            if [[ `echo "${_opt_size}<0"|bc` -eq 1 ]];then
                _opt_size=${_opt_minimum_size}
            fi
            opt_exist=`df -PTh | sed  '1d' | awk '{print $7}' | egrep "^${dir_name}$"` >/dev/null 2>&1
            if [ -z "${opt_exist}" ]
            then
                print_log "WARNING" "dir \"$dir_name\" is not mount dependently"
                present_space="0"
            else
                present_space=`df ${dir_name} -m -P | column -t | sed -n "2p" | awk '{print $2}'` >/dev/null 2>&1
            fi
            target_space=`echo "scale=1;${_opt_size} - 1024" | bc`
            fn_opt_verify_partition_volume ${present_space} ${target_space}
            if [[ $? -ne 0 ]];then
                error_fail_num=`echo "${error_fail_num}-1" | bc`
                if [[ "${disk_num}" -gt "1" ]];then
                    disk_1=$(lsblk | awk '{if($6 == "disk")print $1}' | awk '/^[vs]da/')
                    disk_size_b_1=$(lsblk -l -b -o NAME,SIZE,TYPE | grep ${disk_1} | awk '{if($3 == "disk")print $2}')
                    disk_size_gb_1=$(echo "scale=2;${disk_size_b_1}/1024/1024/1024" | bc)
                    partitioned_disk_1_b=0
                    storage_name_capacity_add=$(lsblk -l -b -o NAME,SIZE,TYPE | grep ${disk_1} | awk '{if($3 == "part")print $2}')
                    for storage_capacity in ${storage_name_capacity_add}
                    do
                        partitioned_disk_1_b=$(echo "scale=2;${partitioned_disk_1_b} + ${storage_capacity}" | bc)
                    done
                    partitioned_disk_1_gb=$(echo "scale=2;${partitioned_disk_1_b}/1024/1024/1024" | bc)
                    unpartitioned_size_1=$(echo "scale=2;${disk_size_gb_1}-${partitioned_disk_1_gb}" | bc)
                    need_size=$(echo "scale=1;(${target_space} - ${present_space})/1024" | bc) 2>&1
                    if [[ `echo "${unpartitioned_size_1} > 5" | bc` -eq 1 ]] && [[ `echo "${unpartitioned_size_1} < ${need_size}" | bc` -eq 1 ]];then
                        sh "disk_expand.sh" "${unpartitioned_size_1}" >/dev/null 2>&1
                        if [[ $? -ne 0 ]];then
                            error_fail_num=`echo "${error_fail_num}+2" | bc`
                            print_log "ERROR" "Expand /opt size failed."
                        else
                            opt_mount_flag=1
                        fi
                    fi
                fi
                # /opt partition is not enough. Now try to expand /opt partition
                if [ "${opt_mount_flag}" == "1" ];then
                    sh "disk_expand.sh" `echo "scale=1;(${target_space} - ${present_space})/1024 - ${unpartitioned_size_1}" | bc` 2>&1
                else
                    sh "disk_expand.sh" `echo "scale=2;(${target_space} - ${present_space})/1024" | bc` 2>&1
                fi
                if [[ $? -ne 0 ]];then
                    error_fail_num=`echo "${error_fail_num}+2" | bc`
                    print_log "ERROR" "Expand /opt size failed."
                else
                    present_space=`df ${dir_name} -m -P | column -t | sed -n "2p" | awk '{print $2}'` >/dev/null 2>&1
                    fn_opt_verify_partition_volume ${present_space} ${target_space}
                    if [[ $? -eq 0 ]];then
                        error_fail_num=0
                        print_log "INFO" "Expand /opt size check successfully."
                    else
                        error_fail_num=`echo "${error_fail_num}+2" | bc`
                        print_log "ERROR" "/opt partition size check failed."
                    fi
                fi
            fi
		else
		    total_disk_size=0
			total_space_list=`lsblk -lb -o NAME,SIZE,TYPE | awk '{if($3 == "disk")print $2}'`
			total_space=0
			for disk_space in ${total_space_list}
			do
			    disk_space="${disk_space}B"
				local tmp_size=`fn_get_size_GB ${disk_space}`
				total_space=`echo ${total_space}+${tmp_size} | bc`
			done
			present_disk_size=`echo ${total_space}-${add_disk} | bc`
			if [ "X${disk_size}" == "X" ];then
				print_log "WARNING" "Disk size to be checked is not configured, let /opt size to be minimum."
				_opt_size=${_opt_minimum_size}
			else
				if [[ `echo "${present_disk_size} < ${disk_size}" | bc` -eq 1 ]] && [[ `echo "${present_disk_size} > 100" | bc` -eq 1 ]];then
				    _opt_size=`echo "scale=2;${present_disk_size}*1024-${_except_opt_size}"| bc`

				else
				    _opt_size=`echo "scale=2;${disk_size}*1024-${_except_opt_size}"| bc`
				fi
			fi
            opt_exist=`df -PTh | sed  '1d' | awk '{print $7}' | egrep "^${dir_name}$"` >/dev/null 2>&1
            if [ -z "${opt_exist}" ]
            then
                print_log "WARNING" "dir \"$dir_name\" is not mount dependently"
                present_space="0"
            else
                present_space=`df ${dir_name} -m -P | column -t | sed -n "2p" | awk '{print $2}'` >/dev/null 2>&1
            fi
            target_space=`echo "scale=1;${_opt_size} - 1024" | bc`
            fn_opt_verify_partition_volume ${present_space} ${target_space}
            if [[ $? -ne 0 ]];then
                error_fail_num=`echo "${error_fail_num}-1" | bc`
                # /opt partition is not enough. Now try to expand /opt partition
                sh "disk_expand.sh" `echo "scale=1;(${target_space} - ${present_space})/1024" | bc` 2>&1
                if [[ $? -ne 0 ]];then
                    error_fail_num=`echo "${error_fail_num}+2" | bc`
                    print_log "ERROR" "Expand /opt size failed."
                else
                    present_space=`df ${dir_name} -m -P | column -t | sed -n "2p" | awk '{print $2}'` >/dev/null 2>&1
                    fn_opt_verify_partition_volume ${present_space} ${target_space}
                    if [[ $? -eq 0 ]];then
                        error_fail_num=0
                        print_log "INFO" "Expand /opt size check successfully."
                    else
                        error_fail_num=`echo "${error_fail_num}+2" | bc`
                        print_log "ERROR" "/opt partition size check failed."
                    fi
                fi
            fi
		fi
	    print_log "INFO" "-----------------------------------------------------"
	fi

    if [ "${error_fail_num}" -gt 0 ]
    then
        return 1
        #failed
    else
        check_fi_disk ${add_disk}
        if [ "$?" != "0" ]; then
            ((error_fail_num=error_fail_num+1))
            return 1
        fi
        return 0
    fi

}

function check_fi_disk()
{
    # Analyzer_Calc_Storage和Analyzer_FI_MgmtCtrl 节点才会去检查
    calc_storage=$(hostname | egrep "Analyzer_Calc_Storage|Analyzer-Calc-Storage|AnalyzerCalcStorage")
    fi_mgmrctrl=$(hostname | egrep "Analyzer_FI_MgmtCtrl|Analyzer-FI-MgmtCtrl|AnalyzerFIMgmtCtrl")
    if [ -n "${calc_storage}" ];then
        local node_name="${calc_storage}"
    elif [ -n "${fi_mgmrctrl}" ];then
        local node_name="${fi_mgmrctrl}"
    fi
    if [ -n "${calc_storage}" -o -n "${fi_mgmrctrl}" ];then
        print_log "INFO" "--------------check FI disk-------------------"
        fi_part_exits=$(lsblk | grep "/srv")
        if [ -n "${fi_part_exits}" ];then
            print_log "INFO" "The FI partition already exists."
            print_log "INFO" "Checking FI partition space successfully."
            return 0
        fi
        local add_disk=$1
        if [ "X${add_disk}" == "X" ];then
            # add_disk 参数为空的时候
            print_log "WARNING" "The FI disk size is not configured, skip check."
            return 0
        fi
        # Get all disks with name vdx and sdx.
        local fi_disk_list=$(lsblk | awk '{if($6 == "disk")print $1}' | awk '/^[vs]d[a-z]/')
        for fi_disk in ${fi_disk_list}
        do
            # Get total disk volume
            local fi_disk_total_cap_b=$(lsblk -l -b -o NAME,SIZE,TYPE | grep ${fi_disk} | awk '{if($3 == "disk")print $2}')
            fi_disk_total_cap=$(echo "scale=2;${fi_disk_total_cap_b}/1024/1024/1024" | bc)
            # if the disk is not partitioned, the disk is available
            local disk_partition_list=$(lsblk -l -o NAME,SIZE,TYPE | grep ${fi_disk} | awk '{if($3 == "part")print $1}')
            if [ -n "${disk_partition_list}" ];then
                print_log "INFO" "The ${fi_disk} is partitioned."
                continue
            fi

            #如果检查出磁盘的空间不够 打印节点空点不足
            if [[ $(echo "${add_disk} > ${fi_disk_total_cap}" | bc) -eq 1 ]];then
                #否则打印不足日志continue
                print_log "INFO" "The ${fi_disk} is not partitioned."
                print_log "ERROR" "The available space of the FI is ${fi_disk_total_cap}G, less than ${add_disk}G."
                return 1
            else
                #ok则打印检查ok日志 return 0
                print_log "INFO" "The ${fi_disk} is not partitioned."
                print_log "INFO" "Checking FI partition space successfully."
                return 0
            fi
        done
        # 循环中没有发现可以用的磁盘 就报错
        print_log "ERROR" "No available disk is found for FI, Checking FI partition space failed."
        return 1
    fi
}
