#!/bin/bash
#==============================================================
# Copyright  Huawei Technologies Co., Ltd. 1998-2019. All rights reserved.
# File Name             : disk_expand_new.sh
# Version:Data          : 2019/06/04  
# Description           : Expand /opt volume using extra disk or existing disk volume
# Others                : 
# History               : 
#==============================================================


#==============================================================
## @Usage func_name target_disk_name target_disk_size target_disk_partition(1-5)
## @Return ${SUCCESS_CODE}
## @Description expand/grow the target disk partition
#==============================================================
function fn_expand_partition()
{
    echo "-----------fn_expand_partition-----------"
    local target_disk_name=$1
    local target_disk_size=$2
    local target_disk_partition=$3
    local vg_name=`lvdisplay | grep opt | grep Path | awk -F '/' '{print $3}'`
    local target_disk_size_MB=`echo ${target_disk_size} | awk '{ printf "%d", $1*1024-20}'`
    if [ -z "${vg_name}" ]
    then
        echo "start to create physical volume"
        sleep 5
        pvcreate /dev/"${target_disk_name}${target_disk_partition}"
        if [[ $? -ne 0 ]]
        then 
            echo "physical volume create failed"
            return 10
        fi
        echo "physical volume ${target_disk_name}${target_disk_partition} create successfully"
        pvdisplay >/dev/null 2>&1
        # create volume group "vg_root"
        vgextend vg_root /dev/"${target_disk_name}${target_disk_partition}"
        if [[ $? -ne 0 ]]
        then 
            echo "volume group vg_root extend failed"
            return 11
        fi
        echo "volume group vg_root extend successfully"
        vgdisplay >/dev/null 2>&1
        lvcreate -l 100%FREE -n opt vg_root -y
        if [[ $? -ne 0 ]]
        then 
            echo "logical volume /dev/vg_root/opt create failed"
            return 12
        fi
        echo "logical volume /dev/vg_root/opt create successfully"
        lvdisplay >/dev/null 2>&1
        mkfs.ext4 /dev/vg_root/opt >/dev/null 2>&1
        if [[ $? -ne 0 ]]
        then 
            echo "volume /dev/vg_root/opt init failed"
            return 13
        fi
        echo "volume /dev/vg_root/opt init successfully"
        sleep 2
        mount /dev/vg_root/opt /opt
        if [[ $? -ne 0 ]]
        then 
            echo "mount /dev/vg_root/opt to /opt failed"
            return 14
        fi
        echo "mount /dev/vg_root/opt to /opt successfully"
        # set fstab
        cat /etc/fstab | grep "/opt" | grep "/dev/mapper/vg_root-opt" >/dev/null 2>&1
        if [[ $? -ne 0 ]]
        then
            echo "/dev/mapper/vg_root-opt /opt ext4 defaults,iversion 1 2" >> /etc/fstab
        fi
    else
        local target_partition=`lvdisplay | grep Name | grep opt | awk '{print $3}'`
        # build physical volume
        echo "start to create physical volume"
        sleep 5
        pvcreate /dev/"${target_disk_name}${target_disk_partition}"
        if [[ $? -ne 0 ]]
        then 
            echo "physical volume create failed"
            return 10
        fi
        echo "physical volume ${target_disk_name}${target_disk_partition} create successfully"
        pvdisplay >/dev/null 2>&1
        # extend existing volume group "vg_root"
        vgextend ${vg_name} /dev/"${target_disk_name}${target_disk_partition}"
        if [[ $? -ne 0 ]]
        then 
            echo "volume group vg_root extend failed"
            return 11
        fi
        echo "volume group vg_root extend successfully"
        vgdisplay >/dev/null 2>&1
        # extend existing logical volume "opt"
        lvextend -L +${target_disk_size_MB}M /dev/${vg_name}/${target_partition}
        if [[ $? -ne 0 ]]
        then 
            echo "logical volume ${target_partition} extend failed"
            return 12
        fi
        echo "logical volume ${target_partition} extend successfully"
        local filesystem=`df -HT | grep opt | awk '{print$2}'`
        if [[ ${filesystem} = 'xfs' ]]
        then
            xfs_growfs /opt >/dev/null 2>&1
        else
            resize2fs /dev/${vg_name}/${target_partition} >/dev/null 2>&1
        fi
        if [[ $? -ne 0 ]]
        then 
            echo "logical volume ${target_partition} resize to filesystem failed"
            return 13
        fi
        echo "logical volume ${target_partition} resize to filesystem successfully"
        lvdisplay /dev/${vg_name}/${target_partition}
    fi
    return 0
}


#==============================================================
## @Usage func_name disk_name_list disks_size_list disks_part_num_list
## @Return ${SUCCESS_CODE}
## @Description judge partition If is not partitioned, then selected. Else the first partitioned.
#==============================================================
function fn_judge_partition()
{
    local disk_name_list=$1
    local disks_size_list=$2
    local disks_part_num_list=$3
    local avai_disk_num=${#disk_name_list[@]}
    if [[ avai_disk_num -lt 1 ]]
    then
        return 5
    fi
    local avai_disk_index=0
    for ((i=0;i<${avai_disk_num};i++));
    do
        if [[ ${disks_part_num_list[i]} =~ 1 ]]
        then
            avai_disk_index=${i}
            break
        fi
    done
    echo "${disk_name_list[${avai_disk_index}]} ${disks_size_list[${avai_disk_index}]} ${disks_part_num_list[${avai_disk_index}]}"
    return 0
}

#==============================================================
## @Usage func_name 
## @Return ${SUCCESS_CODE}
## @Description rescan storage adapter
#==============================================================
function fn_rescan_storage_adapter()
{
    echo "-----------fn_rescan_storage_adapter-----------"
    scsi_host_list=`ls /sys/class/scsi_host`
    echo "Begin to rescan storage adater. Existing hosts are ${scsi_host_list}"
    # resan all the scsi adapters
    for scsi_host in ${scsi_host_list}
    do
        echo "- - -" > /sys/class/scsi_host/${scsi_host}/scan
    done
    # wait several seconds to ensure all the disk been found
    echo "Rescan cmds have been sent. Wait serveral seconds for scanning result"
    sleep 5
    return 0
}

#==============================================================
## @Usage func_name  size(P,T,G,M,K,B)
## @Return ${SUCCESS_CODE}
## @Description transfer the disk size in GB format.
## input is $1(with unit), $2(GB format)
#==============================================================
function fn_get_size_GB()
{
    local quant=`echo $1 | grep -E '[0-9.]+' -o`
    local unit=`echo $1 | grep -E '[PTGMKB]' -o`
    case ${unit} in
    "P")
        echo `echo "${quant}*1024*1024" | bc`
        ;;
    "T")
        echo `echo "${quant}*1024" | bc`
        ;;
    "G")
        echo `echo "${quant}" | bc`
        ;;
    "M")
        echo `echo "scale=2;${quant}/1024" | bc`
        ;;
    "K")
        echo `echo "scale=2;${quant}/1024/1024" | bc`
        ;;
    "B")
        echo `echo "scale=2;${quant}/1024/1024/1024" | bc`
        ;;
    esac
    return 0
}

#==============================================================
## @Usage func_name 
## @Return ${SUCCESS_CODE}
## @Description get the available disk. If disk has not enough capacity, return False
#==============================================================
function fn_get_avai_disk()
{
    echo "-----------fn_get_avai_disk-----------"
    # Get all disks with name vdx and sdx.
    local opt_disk_list=`lsblk | awk '{if($6 == "disk")print $1}' | awk '/[vs]d[a-z]/'`
    echo "Exsiting disks are ${opt_disk_list}"
    for opt_disk in ${opt_disk_list}
    do
        # Get total disk volume
        local opt_disk_total_cap_b=`lsblk -l -b -o NAME,SIZE,TYPE | grep ${opt_disk} | awk '{if($3 == "disk")print $2}'`
        opt_disk_total_cap=`echo "scale=2;${opt_disk_total_cap_b}/1024/1024/1024" | bc`
        echo "${opt_disk} total disk volume is  ${opt_disk_total_cap}GB"
        # Get partitioned disk volume
        local partitioned_disk_volume_GB=0
        local storage_name_capacity_add=`lsblk -l -o NAME,SIZE,TYPE | grep ${opt_disk} | awk '{if($3 == "part")print $2}'`  
        for storage_capacity in ${storage_name_capacity_add}
        do
            storage_capacity=`fn_get_size_GB "${storage_capacity}"`
            partitioned_disk_volume_GB=`echo "scale=2;${partitioned_disk_volume_GB} + ${storage_capacity}" | bc`
        done
        local unpartitioned_size=`echo "scale=2;${opt_disk_total_cap}-${partitioned_disk_volume_GB}" | bc`
        echo "${opt_disk} unpartitioned disk volume is  ${unpartitioned_size}GB"
        # if disk unpartitioned size is greater than size needed, continue.
        if [[ `echo "${unpartitioned_size} < ${disk_size}" | bc` -eq 1 ]] ||[[ `echo "${unpartitioned_size} <= 1" | bc` -eq 1 ]] 
        then
            echo "${opt_disk} unpartitioned disk volume is not enough for disk expanding."
            echo "Unpartitioned disk volume is ${unpartitioned_size} GB, volume required is ${disk_size}GB."
            continue
        fi
        # reserved 0.01GB storage for buffering
        local partition_size=`echo "scale=2;${unpartitioned_size}-0.01" | bc`
        if [[ `echo "${disk_size} > 0" | bc` -eq 1 ]]
        then
            partition_size=${disk_size}
        fi
        # get the partition number.
        local disk_partition_list=`lsblk -l -o NAME,SIZE,TYPE | grep ${opt_disk} | awk '{if($3 == "part")print $1}'`
        partition_index_temp=1
        # if the disk is not partitioned, the disk is available
        if [[ -z ${disk_partition_list} ]]
        then
            avai_disks_name_list="${avai_disks_name_list} ${opt_disk}" 
            avai_disks_size_list="${avai_disks_size_list} ${partition_size}" 
            avai_disks_part_num_list="${avai_disks_part_num_list} 1"
            echo "${opt_disk} is not partitioned. An optional partition is ${opt_disk}1, target disk size is ${partition_size}GB"
        fi
        # if partition number in range(1-5) is occupied, then the disk is not available to use
        for partition_index_temp in {1..4}
        do
            if [[ ! ${disk_partition_list} =~ ${partition_index_temp} ]]
            then
                avai_disks_name_list="${avai_disks_name_list} ${opt_disk}" 
                avai_disks_size_list="${avai_disks_size_list} ${partition_size}" 
                avai_disks_part_num_list="${avai_disks_part_num_list} ${partition_index_temp}"
                echo "An optional partition is ${opt_disk}${partition_index_temp}, target disk size is ${partition_size}GB"
            break 
            fi
        done
    done
    avai_disks_name_list=(${avai_disks_name_list})
    avai_disks_size_list=(${avai_disks_size_list})
    avai_disks_part_num_list=(${avai_disks_part_num_list})
    if [[ -z ${avai_disks_name_list} ]]
    then
        opt_lv_exists=`lvdisplay | grep opt | grep Path | awk -F '/' '{print $3}'`
        vgs_free=`vgs | grep -v "VFree" | awk '{print $NF}' | tr "[ptgmkb]" "[PTGMKB]"`
        vgs_free_GB=`fn_get_size_GB "${vgs_free}"`
        if [ ! -z ${vgs_free_GB} ] && [[ `echo "${vgs_free_GB} > 50.0" | bc` -eq 1 ]] && [ -z "${opt_lv_exists}" ]
        then
            opt_vg_name="vg_root"
            lvcreate -y -l 100%VG -n opt ${opt_vg_name}
            if [[ $? -ne 0 ]]
            then 
                echo "logical volume /dev/mapper/${opt_vg_name}-opt create failed"
                return 12
            fi
            echo "logical volume /dev/mapper/${opt_vg_name}-opt create successfully"
            lvdisplay
            mkfs.ext4 /dev/mapper/${opt_vg_name}-opt >/dev/null 2>&1
            if [[ $? -ne 0 ]]
            then 
                echo "volume /dev/mapper/${opt_vg_name}-opt init failed"
                return 13
            fi
            echo "volume /dev/mapper/${opt_vg_name}-opt init successfully"
            mount /dev/mapper/${opt_vg_name}-opt /opt
            if [[ $? -ne 0 ]]
            then 
                echo "mount /dev/mapper/${opt_vg_name}-opt to /opt failed"
                return 14
            fi
            echo "mount /dev/mapper/${opt_vg_name}-opt to /opt successfully"
            # set fstab
            cat /etc/fstab | grep "/opt" | grep "/dev/mapper/${opt_vg_name}-opt" >/dev/null 2>&1
            if [[ $? -ne 0 ]]
            then
                echo "/dev/mapper/${opt_vg_name}-opt /opt ext4 defaults,iversion 1 2" >> /etc/fstab
            fi
            exit 0
        else
            echo "No suitable disk is found, task ends"
            return 6
        fi
    fi
    echo "avai_disks_name_list is ${avai_disks_name_list},\
        avai_disks_size_list is ${avai_disks_size_list},\
        avai_disks_part_num_list is ${avai_disks_part_num_list}"
    return 0
}

function fn_build_grater_partition()
{
    local disk_name=$1
    local disk_partition=$2
    local disk_size=$3
    local os_subversion=""
    local partition_mode="parted"
    
    # use parted tool for vdb
    if [ ${partition_mode} = "parted" ];then
        expect<<EOF
        set timeout 30
        spawn ${partition_mode} /dev/${disk_name}
        expect "*parted*"
        send "mklabel gpt\r"
        expect {
        "*parted*" {send "mkpart ${disk_name}${disk_partition}\r"}
        "*Yes/No*" {send "Y\r"}
        }
        expect "*parted*" 
        send "mkpart ${disk_name}${disk_partition}\r"
        expect "*type*"
        send "ext4\r"
        expect "*Start*"
        send "0\r"
        expect "*End*"
        send "100%\r"
        expect "*Ignore/Cancel*"
        send "I\r"
        expect "*parted*"
        send "toggle ${disk_partition} lvm\r"
        expect "*parted*"
        send "q\r"
        expect eof
        expect timeout 
        send "target_disk_partition build failed, timeout";return 127
EOF
    fi
    partprobe
    sleep 2
    num=1
    while [ "${num}" -le 60 ]
    do
        part_check=$(ls /dev/"${disk_name}${disk_partition}" 2>/dev/null)
        if [ -z "${part_check}" ] 
        then
            num=$(expr ${num} + 1)
            sleep 3
        else
            echo "target_disk_partition build successfully"
            num=100
        fi
    done

    return 0
}

#==============================================================
## @Usage func_name 
## @Return ${SUCCESS_CODE}
## @Description build target disk partition(Using all the remaining volume)
## @ErrorCode:
#==============================================================
function fn_build_partition()
{
    echo "-----------fn_build_partition-----------"
    # reserved. Should also supports gpt
    local partition_mode="fdisk"
    echo "The input parameter in fn_build_partition function: target_disk_partition is ${target_disk_partition}, target disk size is ${target_disk_size}, the target disk name is ${target_disk_name}"
    local target_disk_size_MB=`echo ${target_disk_size} | awk '{ printf "%d", $1*1024-10}'`
    echo "target_disk_partition is ${target_disk_partition}, target disk expanding size is ${target_disk_size_MB}MB"
    LANG="en_US"
    # check if 
    if [[ `echo "${target_disk_size} < 2000" | bc` -eq 1 ]];then
        if [ ${partition_mode} = "fdisk" ];then
            expect<<EOF
            set timeout 30
            spawn fdisk /dev/${target_disk_name}
            expect "*help):"
            send "n\r"
            expect "*primary*"
            send "p\r"
            expect {
            "*Partition number*" {send "${target_disk_partition}\r\r"}
            "*First sector*" {send "\r"}
            }
            expect "*Last sector*"
            send "+${target_disk_size_MB}M\r"
            expect "*help*"
            send "t\r"
            expect {
            "*Partition number*" {send "${target_disk_partition}\r8e\r"}
            "*Hex*" {send "8e\r"}
            "*type*" {send "8e\r"}
            }
            expect "*help*"
            send "w\r"
            expect eof
            expect timeout 
            send "target_disk_partition build failed, timeout";return 127
EOF
        fi
        partprobe
        sleep 2
        fdisk_check=$(lsblk | grep "${target_disk_name}${target_disk_partition}")
        if [ -z "${fdisk_check}" ];then
            [[ -f fdisk.txt ]] && rm -f fdisk.txt
            touch fdisk.txt
            if [ "${target_disk_partition}" == "1" ];then
                echo "n" >> fdisk.txt
                echo "p" >> fdisk.txt
                echo "${target_disk_partition}" >> fdisk.txt
                echo "" >> fdisk.txt
                echo "+${target_disk_size_MB}M" >> fdisk.txt
                echo "t" >> fdisk.txt
                echo "8e" >> fdisk.txt
                echo "w" >> fdisk.txt
            else
                echo "n" >> fdisk.txt
                echo "p" >> fdisk.txt
                echo "${target_disk_partition}" >> fdisk.txt
                echo "" >> fdisk.txt
                echo "+${target_disk_size_MB}M" >> fdisk.txt
                echo "t" >> fdisk.txt
                echo "${target_disk_partition}" >> fdisk.txt
                echo "8e" >> fdisk.txt
                echo "w" >> fdisk.txt
            fi
            fdisk /dev/${target_disk_name} < fdisk.txt >/dev/null 2>&1
            partprobe
            sleep 2
        fi
        num=1
        while [ "${num}" -le 60 ]
        do
            part_check=$(ls /dev/"${target_disk_name}${target_disk_partition}" 2>/dev/null)
            if [ -z "${part_check}" ] 
            then
                num=$(expr ${num} + 1)
                sleep 3
            else
                echo "target_disk_partition build successfully"
                num=100
                return 0
            fi
        done
        return 0
    else
        fn_build_grater_partition  ${target_disk_name} ${target_disk_partition} ${target_disk_size}
    fi
}


#==============================================================
## @Usage func_name 
## @Return ${SUCCESS_CODE}
## @Description main function
## @ErrorCode: 3:rescan storage failed; 4:get available disk failed 5:No enough available disk
#==============================================================
function fn_main()
{
    disk_opt="sdb"
    partition_opt="1"
    avai_disks_name_list=""
    avai_disks_size_list=""
    avai_disks_part_num_list=""
    disk_size=-1
    if [[ $# -eq 1 ]]
    then
    #0.01GB for buffering
        disk_size=`echo "scale=2;$1-0.01" | bc` 
    fi
    # rescan adapter in case storage not found
    fn_rescan_storage_adapter || return $?
    # get the available disk. If disk has not enough capacity, return False
    fn_get_avai_disk || return $?
    # judge whether the available disk meets the need. return (target_disk disk_size partition_number)
    echo "-----------fn_judge_partition-----------"
    result=`fn_judge_partition ${avai_disks_name_list} ${avai_disks_size_list} ${avai_disks_part_num_list}` || return $?
    result=(${result})
    target_disk_name=${result[0]}
    target_disk_size=${result[1]}
    target_disk_partition=${result[2]}
    echo "target_disk_name is ${target_disk_name}, target_disk_size is ${target_disk_size}GB, target_disk_partition is ${target_disk_partition}GB"
    # build the available partition
    fn_build_partition || return $?
    # expand/grow the opt partition 
    fn_expand_partition ${target_disk_name} ${target_disk_size} ${target_disk_partition} || return $?
    return 0
}


#==============================================================
# main entrance
#checking if the parameter of this script is not null , and provide the usage of this script   
if [[ $# -gt 1 ]]
then
    echo "FATAL" "Usage: $0 or $0 expand_disk_size(GB)"
    exit 1
fi
#checking if disk size is digits. To ensure no injection attack
if [[ $# -eq 1 ]] && [[ `echo $1 | grep "[^0-9.]"` ]]
then
    echo "FATAL" "Expand disk size should be in digits format"
    exit 2
fi
if [[ $# -eq 1 ]] && [[ `echo "${1} < 1" | bc` -eq 1 ]]
then
    echo "FATAL" "Expand disk size should be greater than 1GB"
    exit 3
fi
if [[ $# -eq 1 ]]
then
    echo "Expand disk size is $1 GB"
else
    echo "Expand disk size is not specified. All the remaining space of the suitable disk will be used to expand target partition"
fi
fn_main $@
MAIN_RET=$?
echo "Expand disk script exited, result is ${MAIN_RET}"
if [[ "${MAIN_RET}" == "0" ]]
then
    echo "`date` `hostname` `whoami` disk_expand.sh;success;Expand disk success." >> /var/log/localmessages
else
    echo "`date` `hostname` `whoami` disk_expand.sh;Failed;Expand disk Failed." >> /var/log/localmessages
fi
exit ${MAIN_RET}
#==============================================================