#!/bin/bash

#
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
#

SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

INSTALL_PATH="/tmp"

ES_OM_PKG_NAME="${1}"

# EasySuite包存放路径
ES_PATH="/opt/oss/easysuite"

# Easysutie软链接路径
ES_LINK_PATH="/opt/oss/easysuite/easysuite"

NODE_ID_FILE="/opt/oss/manager/var/agent/mcagentid.conf"

MANAGER_IP_FILE="/opt/oss/manager/var/agent/managerip.conf"

LOG_PATH="/opt/oss/log/manager/easysuite_upgrade"

LOG_FILE="${LOG_PATH}/deploy_easysuite_om.log"

RUN_AS_USER="ossadm"

##############################################################
# 初始化日志(处理日志绕接)
# @param
#   level --日志级别
#   msg   --日志信息
# @env
#   LOG_PATH --日志文件存放路径
#   LOG_FILE --日志文件名
##############################################################
function logger()
{
    msg="${2}"
    if [ "${1}" == "ERROR" ]
    then
        level="ERROR"
    elif [ "${1}" == "WARN" ]
    then
        level="WARN"
    else
        level="INFO"
    fi
    if [ ! -d "${LOG_PATH}" ]
    then
        mkdir -p ${LOG_PATH}
    fi
    if [ -f "${LOG_FILE}" ];then
        log_size=$(ls -l "${LOG_FILE}" | awk '{print $5}')
        if [ "${log_size}" -gt 5000000 ]
        then
            echo "The log_file is :${LOG_FILE}">>${LOG_FILE}
            echo "The log_size is :${log_size}">>${LOG_FILE}
            date_str=$(date '+%Y-%m-%d %H:%M:%S')
            echo "${date_str}|Zip the log." >>${LOG_FILE}
            LOG_FILEbak="${LOG_FILE}.1"
            cp -p ${LOG_FILE} ${LOG_FILEbak}
            echo 'Zip the log' >${LOG_FILE}
        fi
    fi
    # ERROR 打印回显
    if [ "${level}" != "INFO" ]
    then
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${level}] $2" | tee -a ${LOG_FILE}
    else
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${level}] $2" >> ${LOG_FILE}
    fi
}

##############################################################
# 判断是否为0号节点
# @env
#   NODE_ID_FILE --平台存放节点id文件
##############################################################
function is_node_is_0()
{
    local error_msg="EasySuite_OM cannot be deployed on the current node. The script can be executed only on node 0 of NCE."
    if [ ! -f "${NODE_ID_FILE}" ]
    then
        # 无法在当前节点部署EasySuite_OM工具,请切换到NCE的0号节点执行脚本
        logger "ERROR" "${error_msg}"
        return 1
    fi
    cat ${NODE_ID_FILE} | grep -qw "localnodeid=0"
    if [ $? -eq 0 ]
    then
        return 0
    fi
    logger "ERROR" "${error_msg}"
    return 1
}

##################################################
# 校验执行用户
# 脚本要求使用oss用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')] [ERROR] The script must be run as the ossadm or root user."
        return 1
    fi
}

##############################################################
# 标准输出
##############################################################
function fn_usage()
{
    cat <<EOF
Usage:
    bash deploy_easysuite_om.sh EasySuite_V100R021C10_OM_linux-x86-64.zip
Options:
    <easysuite_om pkg name>
Example:
    bash deploy_easysuite_om.sh <easysuite_om pkg name>
EOF
    return 0
}

##############################################################
# 检查是否已有EasySuite在运行
##############################################################
function check_es_process()
{
    ps -ef | grep 19090 | grep -v grep | grep -q easysuite
    if [ $? -ne 0 ]
    then
        # 进程不存在，正常退出
        return 0
    fi
    echo ---------------------------------------------------------------------------------------
    logger 'WARN' "Easysuite is running already.Check. If you need to continue the operation, run the stop.sh command to stop the EasySuite and continue."
    es_pid=$(netstat -anp 2>/dev/null | grep "tcp.*19090.*LISTEN" | awk '{print $NF}' | awk -F/ '{print $1}')
    if [ -n "${es_pid}" -a "${es_pid}" != "-" ]
    then
        process_data=$(ps -ef | fgrep ${es_pid} | grep -v grep | xargs)
        logger 'WARN' "Process run user: $(echo ${process_data} | awk '{print $1}')"
        logger 'WARN' "Process pid: $(echo ${process_data} | awk '{print $2}')"
        logger 'WARN' "Process start time: $(echo ${process_data} | awk '{print $5}')"
        logger 'WARN' "Easysuite: $(echo ${process_data} | awk '{print $8}')"
    fi
    echo ---------------------------------------------------------------------------------------
    return 1
}

##############################################################
# 判断配置文件中指定的软件包是否存在
# @env
#   ES_OM_PKG_NAME --easysuite_om包名
#   ES_OM_PKG_CONFIG_FILE --存放easysuite_om包名配置文件
##############################################################
function is_es_om_exited()
{
    if [ -z "${ES_OM_PKG_NAME}" ]
    then
        logger "ERROR" "Easysuite_om package(${ES_OM_PKG_NAME}) is not exited."
        fn_usage
        exit 1
    fi

    # easysuite_om包存放绝对路径
    local es_pkg_path="${INSTALL_PATH}/${ES_OM_PKG_NAME}"
    if [ ! -f "${es_pkg_path}" ]
    then
        logger "ERROR" "File(${es_pkg_path}) is not exited."
        exit 1
    fi
    return 0
}

##############################################################
# 判断节点上是否已经部署相同版本的Easysuite目录
# @env
#   ES_OM_PKG_NAME --easysuite_om包名
#   ES_OM_PKG_CONFIG_FILE --存放easysuite_om包名配置文件
##############################################################
function is_es_om_deploy()
{
    local es_unzip_path="/opt/oss/easysuite/${ES_OM_PKG_NAME//.zip/}"

    if [ -d ${es_unzip_path} ]; then
        logger "ERROR" "EasySuite OM of the same version has been deployed.Terminate current operation."
        echo "Deploy path:$(ls -ld ${es_unzip_path})"
        return 1
    fi
    return 0
}

##############################################################
# 获取1号管理节点IP
# @env
#   MANAGER_IP_FILE --存放Managerip的配置文件
##############################################################
function get_remote_ip()
{
    # 获取远端Managerip
    # 遍历所有节点ssh检查nodeid是否为1
    all_node_ip=$(cat ${MANAGER_IP_FILE} | grep ^managerip | awk -F= '{print $NF}')
    for node_ip in ${all_node_ip//,/ }
    do
        ssh ${RUN_AS_USER}@${node_ip} -o stricthostkeychecking=no "cat ${NODE_ID_FILE} | grep -qw localnodeid=1" 2>/dev/null
        if [ $? -eq 0 ]
        then
            echo "${node_ip}"
            return 0
        fi
    done
    return 1
}

##############################################################
# 在OMP主备/opt/oss/easysuite目录下分别传包，并且解压部署EasySuite
# 5节点场景也只部署0,1号节点
##############################################################
function deploy_om_on_other_node()
{
    # OMP无主备跳过
    cat ${MANAGER_IP_FILE} | grep ^managerip | grep -q ,
    if [ $? -ne 0 ]
    then
        return 0
    fi

    remote_manager_ip=$(get_remote_ip)
    if [ -z "${remote_manager_ip}" ]
    then
        logger "ERROR" "Failed to get the other manager ip."
        thread_ret_code=1
        return 1
    fi
    # 初始化远端Easysuite工作路径
    ssh -o stricthostkeychecking=no ${RUN_AS_USER}@${remote_manager_ip} "mkdir -p ${ES_PATH};rm -rf ${ES_LINK_PATH}" 2>/dev/null

    # 处理ipv6场景
    scp_remote_ip="${remote_manager_ip}"
    if [[ "${remote_manager_ip}" =~ ":" ]]
    then
        scp_remote_ip=["${remote_manager_ip}"]
    fi

    # scp传输软件包到对端
    local es_pkg_path="${INSTALL_PATH}/${ES_OM_PKG_NAME}"
    scp -o stricthostkeychecking=no "${es_pkg_path}" ${RUN_AS_USER}@${scp_remote_ip}:/opt/oss/easysuite 2>/dev/null

    # 解压并创建软链接
    local unzip_es_pkg_path="/opt/oss/easysuite/${ES_OM_PKG_NAME}"
    local es_unzip_path="/opt/oss/easysuite/${ES_OM_PKG_NAME//.zip/}"
    # 清理目录
    ssh -o stricthostkeychecking=no ${RUN_AS_USER}@${remote_manager_ip} "chmod -R u+w ${es_unzip_path} 2>/dev/null;rm -rf ${es_unzip_path}" 2>/dev/null

    # 解压Easysuite包
    ssh -o stricthostkeychecking=no ${RUN_AS_USER}@${remote_manager_ip} "unzip -q ${unzip_es_pkg_path} -d ${es_unzip_path}" 2>/dev/null
    if [ $? -ne 0 ]
    then
        logger "ERROR" "Failed to unzip the easysuite package(${unzip_es_pkg_path})."
        thread_ret_code=1
        return 1
    fi

    # 创建Easysuite软链接
    ssh -o stricthostkeychecking=no ${RUN_AS_USER}@${remote_manager_ip} "cd /opt/oss/easysuite && ln -sn ${ES_OM_PKG_NAME//.zip/} easysuite" 2>/dev/null
}

##############################################################
# 在本节点部署Easysuite_OM
##############################################################
function deploy_om_on_local_node()
{
    local es_pkg_path="${INSTALL_PATH}/${ES_OM_PKG_NAME}"
    local es_unzip_path="/opt/oss/easysuite/${ES_OM_PKG_NAME//.zip/}"
    # 创建目录并清理软链接
    mkdir -p ${ES_PATH}
    rm -rf ${ES_LINK_PATH}
    # 清理目录
    # 递归赋予写权限,避免无法删除
    chmod -R u+w ${es_unzip_path} 2>/dev/null
    rm -rf ${es_unzip_path}

    # 解压Easysuite包
    unzip -q ${es_pkg_path} -d ${es_unzip_path}
    if [ $? -ne 0 ]
    then
        logger "ERROR" "Failed to unzip the easysuite package(${es_pkg_path})."
        thread_ret_code=1
        return 1
    fi

    # 创建Easysuite软链接
    cd /opt/oss/easysuite && ln -sn ${ES_OM_PKG_NAME//.zip/} easysuite
    bash /opt/oss/easysuite/easysuite/start.sh
}

##############################################################
# 在0和1号管理节点分别部署Easysuite_OM
##############################################################
function deploy_pkg()
{
    # 检查EasySuite进程
    check_es_process || return 0

    thread_ret_code=0
    deploy_om_on_local_node &
    deploy_om_on_other_node &
    wait
    return ${thread_ret_code}
}

##############################################################
# 脚本总入口
##############################################################
function main()
{
    # 检查脚本执行用户
    check_user || return 1

    # 非0号节点退出
    is_node_is_0 || return 0

    # 安装目录下包不存在退出
    is_es_om_exited || return 1

    # 是否已经部署相同版本的Easysuite目录
    is_es_om_deploy || return 1

    # 部署工具包到0和1号节点
    deploy_pkg
}

run_user=$(whoami)
if [ "${run_user}" == "root" ]
then
    script="${SCRIPT_PATH}/${SELF_FILE}"
    chown -h root:ossgroup "${script}"
    su - ossadm -c "bash ${script} ${ES_OM_PKG_NAME}"
else
    main "$*"
fi
exit $?
