#!/usr/bin/python
# -*- coding: utf-8 -*-
import re
import traceback

from getDBConnection import get_zenith_session
from common_tasks.base_task import BaseTask

ONE_PACKAGE = 500


class UpgradeTNeamNeinfo(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradeTNeamNeinfo, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeTNeamNeinfo init product_name is %s" % product_name)
        self.src_db_session = get_zenith_session('ISDB', 'ISDB', product_name)
        if self.src_db_session is None:
            self.error("ISDB is None")
            return
        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('ISDB', 'ISDB', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "t_neam_neinfo"
        self.src_table_cols = ("c_neid", "c_uniqueid")
        self.src_table_cols_index = {y:x for x, y in enumerate(self.src_table_cols)}
        self.dst_talble = "t_neam_neinfo"
        self.dst_table_cols = ("c_neid", "c_uniqueid")

    def convert_data(self, paras):
        def covert_ip_addr(prefix,c_uniqueId ):
            if re.compile(r"%s([\d]{1,3}\.[\d]{1,3}\.[\d]{1,3}\.[\d]{1,3})" % prefix).match(c_uniqueId):
                # 能匹配上，就是IPv4
                return c_uniqueId.replace(prefix, "ipv4://").replace("_", "/")
            else:
                # 不是IPv4就是IPv6,因为IPv6的格式比较灵活，所以只要不是IPv4的就当做IPv6处理
                return c_uniqueId.replace(prefix, "ipv6://").replace("_", "/")

        c_neid = paras[self.src_table_cols_index.get("c_neid")]
        c_uniqueId = paras[self.src_table_cols_index.get("c_uniqueid")]
        if c_uniqueId is None:
            self.warning("c_uniqueId is None")
        else:
            # 这里确保c_uniqueId是字符串类型
            c_uniqueId = str(c_uniqueId)
            if c_uniqueId.startswith("access_"):
                #  FAN的地址格式：
                # 1）access_10.1.1.1
                # 2）access_10.1.1.1_0.2.1.14.12（ONU无IP地址场景）
                # 需要转化成
                # 1）ipv4://10.1.1.1; ipv4://10.1.1.1/0.2.1.14.12（ONU无IP地址场景）
                # 2）ipv6://xxxx.xxxx.xxxx.xxxx.xxxx.xxxx.xxxx.xxxx
                c_uniqueId = covert_ip_addr("access_", c_uniqueId)
            elif c_uniqueId.startswith("bits_"):
                # BITS的地址格式的处理过程一样
                c_uniqueId = covert_ip_addr("bits_", c_uniqueId)
            elif c_uniqueId.startswith("ip_"):
                # IP的地址格式跟FAN的处理过程一样
                c_uniqueId = covert_ip_addr("ip_", c_uniqueId)
            elif c_uniqueId.startswith("3rd_"):
                # 3rd的地址格式跟FAN的处理过程一样
                c_uniqueId = covert_ip_addr("3rd_", c_uniqueId)
            elif c_uniqueId.startswith("em3rd_ip_"):
                # 3rd的地址格式跟FAN的处理过程一样
                c_uniqueId = covert_ip_addr("em3rd_ip_", c_uniqueId)
            elif c_uniqueId.endswith("_transqx") or c_uniqueId.endswith("_ptnv8"):
                # 物理ID：8978553_transqx，12345678_ptnv8
                # 转化成：phyid://8978553
                # Done： 物理ID：phyid://24-166（计算方法：1573030/65536-1573030%65536）
                # 不管后缀是_transqx还是_ptnv8，全都去除掉
                s_uniqueId = c_uniqueId.replace("_transqx", "").replace("_ptnv8", "")
                if s_uniqueId.isdigit():
                    i_uniqueId = int(s_uniqueId)
                    c_uniqueId = "phyid://%s-%s" % (int(i_uniqueId / 65536), i_uniqueId % 65536)
                else:
                    self.warning("c_uniqueid invalid: %s" % c_uniqueId)
                    c_uniqueId = c_uniqueId
            elif c_uniqueId.endswith("_transtl1"):
                # 物理ID：8978553_transtl1
                # 转化成：tl1://8978553
                # 不需要计算
                # 后缀是_transtl1，去掉后缀，改为tl1://xxxx的格式
                c_uniqueId = "tl1://%s" % c_uniqueId.replace("_transtl1", "")
            elif c_uniqueId.endswith("_transone"):
                # 物理ID：global_8978553_transone
                # 转化成：one://global_8978553_transone
                # 不需要计算
                # _transone，改为one://xxxx的格式
                c_uniqueId = "one://%s" % c_uniqueId

            else:
                # 如果既不是IP也不是物理ID，就不需要执行转化
                pass

        # 注意，这里的数据要反过来，以便update语句可以使用
        return tuple(None if x is None else str(x) for x in (c_uniqueId, c_neid))

    def to_UpgradePara(self, datas):
        insert_stmt = "UPDATE %s SET %s=? WHERE %s=?" % \
                      (self.src_table, self.src_table_cols[1], self.src_table_cols[0])
        self.debug("insert sql stmt: %s" % insert_stmt)
        list_datas = []
        for data in datas:
            self.debug("original data is: %s, length is:%s" % (data, len(data)))
            data = self.convert_data(data)
            self.debug("coverted date is: %s, length is:%s"%(data, len(data)))
            if len(data) == len(self.dst_table_cols):
                tuple_data = tuple(data)
                list_datas.append(tuple_data)
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def do(self):
        try:
            self.info('UpgradeTNeamNeinfo::do start!')
            select_stmt = "select %s from %s"\
                          % (",".join(self.src_table_cols), self.src_table)
            datas = self.exec_query_sql(self.src_db_cursor, select_stmt)
            self.info('get para data success count:%d'%len(datas))

            self.to_UpgradePara(datas)
            self.info('to_UpgradePara done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeTNeamNeinfo got exception')
            self.error(traceback.format_exc())
            return -1
        self.info('UpgradeTNeamNeinfo::do done')
        return 0


if __name__ == '__main__':

    tool = UpgradeTNeamNeinfo()
    print('[INFO] UpgradeTNeamNeinfo start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeTNeamNeinfo finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
