'''
use this to upgrade web privilege
'''
#!/usr/bin/python
# -*- coding: utf-8 -*-
import json
import traceback

from getDBConnection import get_zenith_session
from common_tasks.base_task import BaseTask

TEMPLATE_PATH = "common_privilege_upgrade_template.json"
DO_NOT_DELETE_PRIVILEGE_SET = ["EMS.APP.TopoAPP.Create_Ne",
                               "EMS.APP.TopoAPP.Delete_Ne",
                               "EMS.APP.TopoAPP.Modify_Ne",
                               "EMS.APP.TopoAPP.Auto_Discovery",
                               "EMS.APP.TopoAPP.Period_Discovery"]

class UpgradePrivilege(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradePrivilege, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradePrivilege init product_name is %s" % product_name)
        self.undefault_roles = []
        self.old_to_new_oper, self.old_to_new_opergroup = self.read_data_from_json(TEMPLATE_PATH)

        self.userdb_sess = get_zenith_session('userdb', 'userdb', product_name)
        if self.userdb_sess is None:
            self.error("invbasedb_sess is None")
            return
        self.userdb_sess.autocommit(True)
        self.userdb_cursor = self.userdb_sess.cursor()

        self.privilegedb_sess = get_zenith_session('privilegedb', 'privilegedb', product_name)
        if self.privilegedb_sess is None:
            self.error("get privilegedb session fail")
            return
        self.privilegedb_sess.autocommit(True)
        self.dst_db_cursor = self.privilegedb_sess.cursor()

    def get_undefault_role(self):
        self.debug("start")
        self.userdb_cursor.execute('SELECT ID FROM ROLE WHERE ISDEFAULT=0')
        undefault_roles = self.userdb_cursor.fetchall()
        if undefault_roles:
            self.undefault_roles = list(undefault_roles)

        self.info("the undefault roles size: %d" % (len(self.undefault_roles)))

    def get_all_opid_by_role(self, curr_role):
        select_stmt = 'SELECT OPID FROM ROLE_OP_RELATION WHERE ROLEID=%d'\
                      %(int(curr_role))
        raw_operid_list = []
        self.debug("select_stmt sql:%s"%select_stmt)
        self.dst_db_cursor.execute(select_stmt)
        raw_operids = self.dst_db_cursor.fetchall()
        for raw_operid in raw_operids:
            raw_operid_list.append(list(raw_operid)[0])

        self.debug("the role's oper is: %s " % raw_operid_list)
        return raw_operid_list

    def insert_new_operid(self, curr_role, new_operid):
        # 重复插入会失败，抛异常即可
        insert_stmt = "INSERT INTO ROLE_OP_RELATION (ROLEID, OPID, ORGID) VALUES (?, ?, ?)"
        list_datas = [(str(curr_role), new_operid, "default-organization-id")]
        self.exec_sql(insert_stmt, list_datas)

    def delete_old_operid(self, curr_role, old_operid):
        """
        删除老的权限
        :param curr_role: 与权限关联的角色
        :param old_operid: 老的待删除的权限ID
        :return: None
        """
        list_datas = [(str(curr_role), old_operid)]
        delete_stmt = "DELETE FROM ROLE_OP_RELATION WHERE ROLEID=? AND OPID=?"
        self.exec_sql(delete_stmt, list_datas)

    def update_new_oper_impl(self, curr_role):
        self.debug("update_new_oper_impl start")
        raw_operids = self.get_all_opid_by_role(curr_role)
        self.info("roleid: %s, raw_operids counts: %s" % (curr_role, len(raw_operids)))
        # Done: 当出现重复new_operid的时候，要去重；
        new_operids_has_been_inserted = []
        new_opergroupids_has_been_inserted = []
        old_operids_to_be_deleted = []
        for raw_operid in raw_operids:
            new_operid = self.old_to_new_oper.get(raw_operid)
            # 用raw_operid获取独立权限ID
            if new_operid:
                # 如果是独立权限ID找到对应的映射值；
                if raw_operid not in DO_NOT_DELETE_PRIVILEGE_SET:
                    old_operids_to_be_deleted.append(raw_operid) # 插入新ID时，对应的老ID后面要删除掉
                for one_operid in new_operid:
                    if one_operid not in new_operids_has_been_inserted:
                        # 如果还没有插入到数据表中，则执行数据插入；
                        new_operids_has_been_inserted.append(one_operid)  # 记录已经插入的独立权限ID，避免重复插入
                        self.insert_new_operid(curr_role, one_operid)

            # 用raw_operid获取完整权限ID
            new_opergroupid = self.old_to_new_opergroup.get(raw_operid)
            if new_opergroupid:
                # 如果是完整权限ID找到对应的映射值
                for one_operid in new_opergroupid:
                    if  one_operid not in new_opergroupids_has_been_inserted:
                        # 如果还没有插入到数据表中，则执行数据插入；
                        new_opergroupids_has_been_inserted.append(one_operid)
                        self.insert_new_operid(curr_role, one_operid)

        # Done: 同时，对于独立权限，要删除老权限；对于完整权限组，要保留老权限；
        for old_operid in old_operids_to_be_deleted:
            self.delete_old_operid(curr_role, old_operid)

        self.debug("update_new_oper_impl end")

    def batch_update_new_oper(self):
        self.debug("batch_update_new_oper start")
        for undefault_role in self.undefault_roles:
            curr_role = list(undefault_role)[0] # undefault_role: (12,)
            self.update_new_oper_impl(curr_role)

        self.debug("batch_update_new_oper end")

    def update_operset_op(self):
        self.debug("update_operset_op start")
        # Done:这里要先过滤一下，避免对大量的操作集执行无效操作
        select_stmt = "SELECT OPSETID, OPID FROM T_OPSET_OP_RELATION WHERE OPSETID IN " \
                      "(SELECT ID FROM T_OPERATION_SET WHERE OPTYPE = 4 OR OPTYPE = 2)"
        self.dst_db_cursor.execute(select_stmt)
        raw_operids = list(self.dst_db_cursor.fetchall())
        self.info("raw_operids counts: %s" % len(raw_operids))
        # Done: 当出现重复new_operid的时候，要去重；
        new_operids_has_been_inserted = []
        new_opergroupids_has_been_inserted = []
        old_operids_to_be_deleted = []
        for opersetid, raw_operid in raw_operids:
            opersetid = str(opersetid)
            new_operid = self.old_to_new_oper.get(raw_operid)
            # 用raw_operid获取独立权限ID
            if new_operid:
                # 如果是独立权限ID找到对应的映射值；
                if raw_operid not in DO_NOT_DELETE_PRIVILEGE_SET:
                    old_operids_to_be_deleted.append((opersetid, raw_operid))  # 插入新ID时，对应的老ID后面要删除掉
                for one_operid in new_operid:
                    if (opersetid, one_operid) not in new_operids_has_been_inserted:
                        # 如果还没有插入到数据表中，则执行数据插入；
                        new_operids_has_been_inserted.append((opersetid, one_operid))  # 记录已经插入的独立权限ID，避免重复插入
                        self.insert_opset_op_relation(opersetid, one_operid)

            # 用raw_operid获取完整权限ID
            new_opergroupid = self.old_to_new_opergroup.get(raw_operid)
            if new_opergroupid:
                # 如果是完整权限ID找到对应的映射值
                for one_operid in new_opergroupid:
                    if (opersetid, one_operid) not in new_opergroupids_has_been_inserted:
                        # 如果还没有插入到数据表中，则执行数据插入；
                        new_opergroupids_has_been_inserted.append((opersetid, one_operid))
                        self.insert_opset_op_relation(opersetid, one_operid)

        # Done: 同时，对于独立权限，要删除老权限；对于完整权限组，要保留老权限；
        self.delete_opset_op_relation(old_operids_to_be_deleted)

        self.debug("update_operset_op end")

    def insert_opset_op_relation(self, opsetid, opid):
        insert_sql = "INSERT into T_OPSET_OP_RELATION(OPSETID, OPID) " \
                     "values(?, ?)"
        self.debug("%s" % insert_sql)
        paras = (opsetid, opid)
        self.exec_sql(insert_sql, [paras])

    def delete_opset_op_relation(self, old_operid):
        del_sql = "DELETE FROM T_OPSET_OP_RELATION WHERE OPSETID=? AND OPID=?"
        list_datas = old_operid
        self.exec_sql(del_sql, list_datas)

    def close_session(self):
        self.userdb_cursor.close()
        self.userdb_sess.close()
        self.dst_db_cursor.close()
        self.privilegedb_sess.close()

    def do(self):
        try:
            self.info("UpgradePrivilege::do start!")
            self.info("get default role start.")
            self.get_undefault_role()
            self.info("get default role end.")
            self.info("batch update rule oper start.")
            self.batch_update_new_oper()
            self.info("batch update rule oper end.")
            self.info("batch operset oper start.")
            self.update_operset_op()
            self.info("batch operset oper end.")
            self.close_session()
            self.info("UpgradePrivilege::do done.")
            return 0
        except Exception as e:
            self.close_session()
            self.error(e)
            self.error('UpgradePrivilege got exception')
            self.error(traceback.format_exc())
            return -1

    def read_data_from_json(self, tamplate_path):
        raw_to_new_operid = {}
        raw_to_new_opergroupid = {}
        with open(tamplate_path, 'r') as my_file:
            config_json = json.load(my_file)
        if isinstance(config_json, dict):
            for cfg in config_json:
                if 'raw_oper' in cfg and 'new_oper' in cfg:
                    raw_to_new_operid[cfg['raw_oper']] = cfg['new_oper']
                if 'raw_group' in cfg and 'new_group' in cfg:
                    raw_to_new_opergroupid[cfg['raw_group']] = cfg['new_group']
        elif isinstance(config_json, list):
            for cfg in config_json:
                if 'raw_oper' in cfg and 'new_oper' in cfg:
                    raw_to_new_operid[cfg['raw_oper']] = cfg['new_oper']
                if 'raw_group' in cfg and 'new_group' in cfg:
                    raw_to_new_opergroupid[cfg['raw_group']] = cfg['new_group']

        self.debug("result:%s" % raw_to_new_operid)
        return raw_to_new_operid, raw_to_new_opergroupid

