# -*- coding:utf-8 -*-
"""检查客户自备SUSE系统的组件"""

import os
import logging
import json
from utils.log_util import EasysuiteLogger
from utils.sshUtil import CsshCmd
from utils.ssh_util import SSHCmd
from utils.thread_util import MultiThreadPool
from func.upgrade.common.upgrade_local_ssh import UpgradeLocalSsh
from func.upgrade.upgrade_operation_mgr.common.upgrade_util import UpgradeUtil

local_logger = logging.getLogger(__name__)


def execute_cmd(cmd, ssh_node, logger=None):
    """
    功能描述: 执行shell命令，返回结果值
    :return: 执行结果
    """
    if UpgradeLocalSsh.is_es_node(ssh_node.get("login_ip")):
        bool_value, stdout = UpgradeLocalSsh.send_admin_local_cmd(ssh_node.get("su_pwd"), cmd,
                                                                  node_ip=ssh_node.get("login_ip"))
        if not bool_value and logger:
            logger.easysuite_error(f"Failed to execute the command: {cmd} on the node {ssh_node.get('login_ip')}")
        return bool_value, stdout
    stdcode, stdout, stderr = SSHCmd.run_su_cmd(cmd, ssh_node.get("login_ip"), ssh_node.get("login_user"),
                                                ssh_node.get("login_pwd"), ssh_node.get("ssh_port"),
                                                ssh_node.get("su_user"),
                                                ssh_node.get("su_pwd"), path=None, watchers=None, timeout=None,
                                                disown=False, is_single_quote=True)
    if stdcode and logger:
        logger.easysuite_error(f"Failed to execute the command: {cmd} on the node {ssh_node.get('login_ip')}")
        logger.easysuite_error(f"Possible causes: Node {ssh_node.get('login_ip')} fails to be connected.")
        return False, stdout
    return True, stdout


def execute_check_cmd(ip, ssh_node, logger, default_rpm_set):
    """
    功能描述: 检查节点组件是否齐全
    :return: 执行结果
    """
    if not ip:
        return True
    ssh_para = "-o ConnectTimeout=30 -o stricthostkeychecking=no -o ConnectionAttempts=5 -o " \
               "ServerAliveInterval=3 -o ServerAliveCountMax=20"
    cmd = "rpm -qa --qf='%{name},'"
    query_cmd = f"result=$(su - ossadm -c \"ssh {ssh_para} {ip} {cmd}\" 2>>/dev/null);echo $result"
    result, out_put = execute_cmd(query_cmd, ssh_node, logger)
    if not result:
        logger.easysuite_error("OS components check failed in {}".format(ip))
        return False
    env_rpm_set = set(out_put.strip().split(","))
    not_installed_set = default_rpm_set - env_rpm_set
    if not_installed_set:
        logger.easysuite_error("Some rpm packages is not installed in {}.".format(ip))
        logger.easysuite_error("The not installed rpm packages list:{}".format(not_installed_set))
        return False
    logger.easysuite_info("OS components check successfully in {}".format(ip))
    return True


def execute_task_check(kvs):
    """
    功能描述: 执行检查任务
    :return: 执行结果
    """
    try:
        # 登录信息
        ssh_nodes_param = get_node_connect_info(kvs)
        logger = EasysuiteLogger.get_logger(kvs, "task_check_os_components")
        logger.easysuite_start("start to check os components")
        execute_thread_pool = MultiThreadPool()

        cur_dir = os.path.split(os.path.realpath(__file__))[0]
        path_map = {"suse-12sp4": "rpm_list/sles/12.4/rpm_list.txt", "suse-12sp5": "rpm_list/sles/12.5/rpm_list.txt"}
        os_type = get_os_type(kvs)
        if os_type not in path_map:
            logger.easysuite_error(f"The os type is {os_type}, unsupported os type")
            return False
        rpm_list_file = os.path.join(cur_dir, path_map.get(os_type))
        logger.easysuite_info(f"Path of the component check list file: {rpm_list_file}")

        default_rpm_set = set()
        if not os.path.isfile(rpm_list_file):
            logger.easysuite_error("the {} not found.".format(rpm_list_file))
            return False
        with open(rpm_list_file, mode='r') as f:
            for line in f.read().splitlines():
                default_rpm_set.add(line)

        # 执行任务
        for site, ssh_node in ssh_nodes_param.items():
            if not ssh_node.get("login_ip") or site == "arbit":
                continue
            result, ip_list = get_ip_list(site, ssh_node, logger)
            if not result:
                logger.easysuite_error("get remote ip failed.")
                return False
            for ip in ip_list:
                execute_thread_pool.add_thread(execute_check_cmd, ip, ssh_node, logger, default_rpm_set)
        # 检查结果信息
        results = execute_thread_pool.execute_with_result()
        if all(results):
            logger.easysuite_finish("OS components check successfully in all nodes.")
            return True
        logger.easysuite_error("OS components check failed.")
        return False
    except Exception as e:
        logger.easysuite_error("Execute check task exception.{}".format(e))
        return False


def get_ip_list(site, ssh_node, logger):
    if site == "arbit":
        return True, [ssh_node.get("login_ip")]
    cmd = "cat /opt/oss/manager/etc/sysconf/nodelists.json"
    result, output = execute_cmd(cmd, ssh_node, logger)
    if not result:
        return False, []
    return True, _get_all_node_ips(output)


def _get_all_node_ips(nodelist):
    node_ips = []
    data = json.loads(nodelist)
    for k, v in data.get('nodeList').items():
        for item in v.get('IPAddresses'):
            if 'maintenance' in item.get('usage'):
                node_ips.append(item.get('IP'))
    return node_ips


def get_node(kvs, node_name, is_arbit=None):
    """
    封装节点信息
    """
    if is_arbit:
        login_ip = UpgradeUtil.get_value_from_main(kvs.get("easysuite.task_id"), "arbitrate_ip")
        login_user = kvs.get("{}_login_user".format(node_name))
        login_pwd = kvs.get("{}_login_pwd".format(node_name))
    else:
        login_ip = kvs.get('{}_ip'.format(node_name))
        login_user = kvs.get("standby_login_user")
        login_pwd = kvs.get('{}_{}_pwd'.format(node_name, login_user))
    return {
        'login_ip': login_ip,
        'login_user': login_user,
        'login_pwd': login_pwd,
        'su_user': "root",
        'su_pwd': kvs.get('{}_root_pwd'.format(node_name)),
        'ssh_port': kvs.get('{}_ssh_port'.format(node_name)),
        'sftp_port': kvs.get('{}_sftp_port'.format(node_name)),
        'node_name': node_name
    }


def get_node_connect_info(kvs):
    """
    获取节点登录信息
    """
    ssh_node_dict = {}
    # 非单管场景
    if kvs.get("single_mgr_domain") == "no":
        master_node = get_node(kvs, "node_omp01")
        ssh_node_dict.update({"master": master_node})
        # 异地容灾场景
        if kvs.get("protection") == 'id_protection_hot':
            standby_node = get_node(kvs, "node_standby_omp01")
            ssh_node_dict.update({"standby": standby_node})
    # 单管场景
    else:
        master_node = get_node(kvs, "node_nmsserver")
        ssh_node_dict.update({"master": master_node})
        # 异地容灾场景
        if kvs.get("protection") == 'id_protection_hot':
            standby_node = get_node(kvs, "node_standby_nmsserver")
            ssh_node_dict.update({"standby": standby_node})
    if kvs.get("is_arbitration_exist") == "yes":
        arbit_node = get_node(kvs, "node_arbitrate", True)
        ssh_node_dict.update({"arbit": arbit_node})
    return ssh_node_dict


def get_os_type(kvs):
    """
    功能描述：判断此任务是否在操作系统上是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    ssh_nodes_param = get_node_connect_info(kvs)
    for site, ssh_node in ssh_nodes_param.items():
        # 某个节点来判断系统类型
        ip = ssh_node['login_ip']
        if not ip:
            continue
        local_logger.info("check os type in {}".format(ip))

        check_suse_cmd = "[ -f /etc/SuSE-release ] && echo suse"
        flag, stdout = execute_cmd(check_suse_cmd, ssh_node)
        local_logger.info("check suse:{}".format(stdout))
        if not "suse" in stdout:
            return "NO-INVOlVED-OS"

        check_suse_type_cmd = "lsblk | grep root_bak &>>/dev/null || echo suse_customer"
        flag, stdout = execute_cmd(check_suse_type_cmd, ssh_node)
        local_logger.info("check suse type:{}".format(stdout))
        if not "suse_customer" in stdout:
            return "NO-INVOlVED-OS"

        check_suse_version_cmd = "grep PATCHLEVEL /etc/SuSE-release"
        flag, stdout = execute_cmd(check_suse_version_cmd, ssh_node)
        local_logger.info("check suse version:{}".format(stdout))
        suse_version = stdout.split("=")[1].strip()
        if suse_version.find("4") or suse_version.find("5"):
            return "suse-12sp{}".format(suse_version)
        else:
            return "NO-INVOLVED-SUSE-VERSION"


def check(kvs):
    """
    功能描述：判断此任务是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    os_type = get_os_type(kvs)
    local_logger.info("success to get os: {}".format(os_type))
    if not os_type:
        return False
    return "suse" in os_type


def exe(kvs):
    return [[('', '', execute_task_check(kvs))]]
