#!/bin/bash
TASK_MGR_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/"
# 配置工作目录
WORK_PATH="/opt/upgrade/easysuite_upgrade/workpath/"
CLOUDSOP_PROFILE="/opt/oss/manager/bin/engr_profile.sh"

##########################################
# 初始化日志(处理日志绕接)
# @param
#   level --日志级别
#   msg   --日志信息
# @env
#   LOG_PATH --日志文件存放路径
#   LOG_FILE --日志文件名
##########################################
function LOG()
{
    msg="${2}"
    if [ "${1}" == "ERROR" ]
    then
        level="ERROR"
    elif [ "${1}" == "WARN" ]
    then
        level="WARN"
    else
        level="INFO"
    fi
    if [ ! -d "${LOG_PATH}" ]
    then
        mkdir -p ${LOG_PATH}
        # 目录可读,文件可写
        id sopuser 2>/dev/null | find ${LOG_PATH} -type d | xargs setfacl -m u:sopuser:r-x
        id sopuser 2>/dev/null | find ${LOG_PATH} -type f | xargs setfacl -m u:sopuser:r--
    fi
    if [ -f "${LOG_FILE}" ];then
        log_size=$(ls -l "${LOG_FILE}" | awk '{print $5}')
        if [ "${log_size}" -gt 5000000 ]
        then
            echo "The log_file is :${LOG_FILE}">>${LOG_FILE}
            echo "The log_size is :${log_size}">>${LOG_FILE}
            Date=$(date '+%Y-%m-%d %H:%M:%S')
            echo "${Date}|Zip the log." >>${LOG_FILE}
            LOG_FILEbak="${LOG_FILE}.1"
            cp -p ${LOG_FILE} ${LOG_FILEbak}
            echo 'Zip the log' >${LOG_FILE}
        fi
    fi
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${level}] $2" >> ${LOG_FILE}
}


function TASK_LOG()
{
    log_level="${1}"
    log_info="${2}"
    LOG "${log_level}" "${log_info}"
    [ -z "${TASKID}" ] && TASKID="${TASK_ID}"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [$$] ${log_info}">>/opt/upgrade/easysuite_upgrade/taskmgr/${TASKID}/task.log
}

##########################################
# 初始化日志(处理日志绕接) 附带回显
# @param
#   level --日志级别
#   msg   --日志信息
# @env
#   LOG_PATH --日志文件存放路径
#   LOG_FILE --日志文件名
##########################################
function LOG_PRINT()
{
    LOG "${1}" "${2}"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [$$] | ${level} | ${2}"
}

##########################################
# 初始化任务管理目录
# @param
#   taskid --任务ID
##########################################
function init_taskmgr()
{
    task_id="${1}"
    task_mgr_path="/opt/upgrade/easysuite_upgrade/taskmgr/"
    if [ ! -d "${task_mgr_path}" ]
    then
        mkdir -p ${task_mgr_path}
    fi
    GROUP="ossgroup"
    OSS_USER=$(id -nu 3001)
    chown :${GROUP} /opt/upgrade/easysuite_upgrade
    chown ${OSS_USER}:${GROUP} /opt/upgrade/easysuite_upgrade/taskmgr/
    if [ -f "${task_mgr_path}/${task_id}/task.log" ]
    then
        rm -f ${task_mgr_path}/${task_id}/task.progress ${task_mgr_path}/${task_id}/task.status 2>/dev/null
    else
        rm -rf ${task_mgr_path}/${task_id} 2>/dev/null
        mkdir -p ${task_mgr_path}/${task_id}
        touch ${task_mgr_path}/${task_id}/task.log
    fi
    echo Progress=0 >${task_mgr_path}/${task_id}/task.progress
    echo Status=running >${task_mgr_path}/${task_id}/task.status
    # 目录r-x,文件r--sopuser 权限
    id sopuser 2>/dev/null | find ${task_mgr_path} -type d | xargs setfacl -m u:sopuser:r-x
    id sopuser 2>/dev/null | find ${task_mgr_path} -type f | xargs setfacl -m u:sopuser:r--
    chown -R ${OSS_USER}:${GROUP} ${task_mgr_path}
}

##################################################
# 刷Easysuite任务状态
##################################################
function fresh_es_status()
{
    local task_id="${1}"
    local status="${2}"
    local file_path="${TASK_MGR_PATH}/${task_id}/task.status"
    echo Status=${status} >${file_path}
}

##################################################
# 刷Easysuite任务进度
##################################################
function fresh_es_progress()
{
    local task_id="${1}"
    local progress="${2}"
    local file_path="${TASK_MGR_PATH}/${task_id}/task.progress"
    echo Progress=${progress} >${file_path}
}

##################################################
# 异常退出
##################################################
function do_exit()
{
    local do_task_id="${1}"
    echo Status=fail >${TASK_MGR_PATH}/${do_task_id}/task.status
    echo Progress=100 >${TASK_MGR_PATH}/${do_task_id}/task.progress
}

##################################################
# 检查脚本执行用户
##################################################
function check_ossadm_user()
{
    # 执行用户
    user=$(id -un)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ]
    then
       echo "The user(${user}) does not have the execute permission."
       return 1
    fi
    return 0
}

##########################################
#  初始化升级配置参数
##########################################
function initial_params()
{
    plan_data_file="${1}"
    # 文件是否存在
    if [ ! -f "${plan_data_file}" ]
    then
        LOG_PRINT "ERROR" "The file(plan_data_file:${plan_data_file}) does not exist"
        return 1
    fi
    # 产品名 plandata.json productnam
    PRODUCT_NAME=$(grep -Po 'productname[" :]+\K[^"]+'  "${plan_data_file}")
    src_version=$(grep -Po 'src_version[" :]+\K[^"]+'  "${plan_data_file}")
    des_version=$(grep -Po 'des_version[" :]+\K[^"]+'  "${plan_data_file}")
    if [ -z "${PRODUCT_NAME}" -o -z "${src_version}" -o -z "${des_version}" ]
    then
        LOG_PRINT "ERROR" "The param (PRODUCT_NAME:${PRODUCT_NAME},src_version:${src_version},des_version:${des_version}) does not exist"
        return 1
    fi
    UPGRADE_PATH="${src_version}-${des_version}"
}

##########################################
#  查询 product_NCE.json
##########################################
function query_product()
{
    output_path="${1}"
    product="${2}"
    LOG_PRINT "INFO" "Start to run {OMP_PATH}/tools/resmgr/queryproduct.sh"
    # 生成product_NCE.json
    ret_msg=$(bash /opt/oss/manager/tools/resmgr/queryproduct.sh -pn "${product}" -output ${output_path})
    ret_code=$?
    if [ ${ret_code} -ne 0 ]
    then
        LOG_PRINT "ERROR" "Failed to run {OMP_PATH}/tools/resmgr/queryproduct.sh;ret_code:${ret_code}"
        LOG_PRINT "ERROR" "Failed to run {OMP_PATH}/tools/resmgr/queryproduct.sh;ret_msg:${ret_msg}"
        return 1
    fi
    product_file="${output_path}/product_${product}.json"

    # 判断文件是否存在
    if [ ! -f "${product_file}" ]
    then
        LOG_PRINT "ERROR" "The file(product_file:${product_file}) does not exist"
        LOG_PRINT "ERROR" "Failed to run {OMP_PATH}/tools/resmgr/queryproduct.sh;ret_code:${ret_code}"
        return 1
    fi
    LOG_PRINT "INFO" "Finished to run {OMP_PATH}/tools/resmgr/queryproduct.sh;ret_code:${ret_code}"
    return 0
}

##########################################
#  更新 product_NCE.json
##########################################
function modify_product()
{
    input_file="${1}"

    LOG_PRINT "INFO" "Start to run {OMP_PATH}/tools/resmgr/modifyproductinfo.sh"
    # 更新 product_NCE.json
    ret_msg=$(bash /opt/oss/manager/tools/resmgr/modifyproductinfo.sh -input "${input_file}")
    ret_code=$?
    if [ ${ret_code} -ne 0 ]
    then
        LOG_PRINT "ERROR" "Failed to run {OMP_PATH}/tools/resmgr/modifyproductinfo.sh;ret_code:${ret_code}"
        LOG_PRINT "ERROR" "Failed to run {OMP_PATH}/tools/resmgr/modifyproductinfo.sh;ret_msg:${ret_msg}"
        return 1
    fi
    LOG_PRINT "INFO" "Finished to run {OMP_PATH}/tools/resmgr/modifyproductinfo.sh;ret_code:${ret_code}"
}

##########################################
#  任务并发
##########################################
function fresh_task_progress()
{
    local task_id="${1}"
    local expect_time="${2}"
    local flag_file="${3}"
    # 开始进度
    [ -z "${4}" ] && local start_progress=0 || local start_progress="${4}"
    # 结束进度
    [ -z "${5}" ] && local finish_progress=100 || local finish_progress="${5}"
    local count=0
    local time_count=$(expr ${expect_time} \* 60)
    while true
    do
        # 文件存在退出
        if [ -f ${flag_file} ]
        then
            ret_code=$(cat ${flag_file})
            rm -f "${flag_file}" &>/dev/null
            # 只有为0时正常退出
            if [ "${ret_code}" == "0" ]
            then
                return 0
            fi

            return 1
        fi
        # 文件不存在继续模拟刷新进度
        sleep 1
        # 小于最大值,继续计算,不产生无穷大数
        [ ${count} -lt ${time_count} ] && let count++
        progress_num=$(expr 100 \* ${count} / ${time_count})
        # 加上起始进度
        progress_num=$(expr ${progress_num} + ${start_progress})

        # 超过最大值置为max-1
        if [ ${progress_num} -ge ${finish_progress} ]
        then
            progress_num=$(expr ${progress_num} - 1)
        fi
        fresh_es_progress "${task_id}" "${progress_num}"
    done
}

##########################################
#  控制敏感文件权限
#  queryproduct导出内容文件默认权限不可控,统一处理一把
##########################################
function set_file_permit()
{
    local sensitive_key=""000""000""010""000""000" "AAA""AHg""AAA""AAA""AA""
    file_path="/opt/upgrade/backup/migrate /opt/upgrade/easysuite_upgrade/scripts/ /opt/upgrade/easysuite_upgrade/workpath/"
    for one_key in ${sensitive_key}
    do
        for find_path in ${file_path}
        do
            # 满足rw*r****权限文件, *为统配含义
            find ${find_path} -type f -name "*.json" -perm -640 2>/dev/null | xargs grep -l ${one_key} | sort -u | xargs chmod 600 &>/dev/null
        done
    done
    return 0
}

##########################################
#  删除历史冗余文件
##########################################
function clear_upgrade_history()
{
    local plan_data_file="${1}/plandata.json"
    product_name=$(grep -Po 'productname[" :]+\K[^"]+'  "${plan_data_file}")
    src_version=$(grep -Po 'src_version[" :]+\K[^"]+'  "${plan_data_file}")
    des_version=$(grep -Po 'des_version[" :]+\K[^"]+'  "${plan_data_file}")
    # 清理历史运维面升级前备份的old_default_value.json文件
    CONFIG_PATH="${WORK_PATH}/"${src_version}-${des_version}"/workpath-${product_name}/old_default_value.json"
    [ -f "${CONFIG_PATH}" ] && rm -f "${CONFIG_PATH}"
    return 0
}

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    # 3001为ossadm用户ID
    if [ "$(whoami)" != "$(id -nu 3001)" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##########################################
#  调用py文件入口函数
##########################################
function execute_python_script()
{
    # 校验入参
    TASKID="${1}"
    # 初始化脚本日志
    LOG "INFO" "Start to init ${LOG_FILE}."

    # 初始化任务
    init_taskmgr "${TASKID}"

    # py文件不存在时使用pyc
    [ -f "${PYTHON_SCRIPT}" ] || PYTHON_SCRIPT="${PYTHON_SCRIPT}c"

    # 目标py文件不存在调用报错退出
    if [ ! -f "${PYTHON_SCRIPT}" ]
    then
        TASK_LOG "ERROR" "The file $(basename ${PYTHON_SCRIPT}) is not exist"
        fresh_es_status "${TASKID}" "fail"
        fresh_es_progress "${TASKID}" "100"
        return 0
    fi

    # 加载平台环境变量,用于调用py脚本
    source "${CLOUDSOP_PROFILE}"

    # 执行python脚本,过程信息记录脚本日志,脚本异常退出返回码非零,刷新任务为失败状态,兜底措施
    python "${PYTHON_SCRIPT}" "$@" &>>${LOG_FILE} || (fresh_es_status "${TASKID}" "fail";fresh_es_progress "${TASKID}" "100")
}
