#!/bin/bash

##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

##################################################
# 初始化日志路径和日志文件名
##################################################
LOG_PATH="/opt/upgrade/easysuite_upgrade/logs"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
UPGRADE_PATH="/opt/upgrade"

##################################################
# 校验执行用户
# 脚本要求使用oss用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##################################################
# 定义相关全局变量
##################################################
function init_variable()
{
    # Easysuite任务相关日志/进度/状态文件
    TASK_ID_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/${TASKID}"
    TASK_LOG_FILE="${TASK_ID_PATH}/task.log"
    TASK_PROGRESS_FILE="${TASK_ID_PATH}/task.progress"
    TASK_STATUS_FILE="${TASK_ID_PATH}/task.status"

    # OMP目录
    OMP_PATH="/opt/upgrade/${PKGNAME//.zip/}"
    UPGRADE_SCRIPT="${OMP_PATH}/Upgrade_manager.sh"

    # 是否是新任务标志位
    NEW_TASK="TRUE"
    [ -d "${TASK_ID_PATH}" ] && NEW_TASK="FALSE"
}

##################################################
# 刷Easysuite任务状态
##################################################
function fresh_status() {
    local status=$1
    echo Status=${status} >${TASK_STATUS_FILE}
}

##################################################
# 刷Easysuite任务进度
##################################################
function fresh_progress() {
    local progress=$1
    echo Progress=${progress} >${TASK_PROGRESS_FILE}
}

##################################################
# 刷Easysuite结果为成功
##################################################
function fresh_finish()
{
    fresh_progress "100"
    fresh_status "finish"
}

##################################################
# 刷Easysuite结果为失败
##################################################
function fresh_fail()
{
    fresh_progress "100"
    fresh_status "fail"
}


function get_exec_param()
{
    upgrade_params=""
    args_num=$#
    for i in $(seq ${args_num})
    do
        # 检验入参是否合法,防止shell注入
        if [[ "$2" =~ "&" || "$2" =~ ";" || "$2" =~ "$" || "$2" =~ "|" || "$2" =~ "；" ]]
        then
           LOG "ERROR" "illegal params for $1 $2"
           exit 1
        fi
        case "${1}" in
            # action,pkgname,taskid,upgradepath 必选参数
            -action) ACTION=$2;shift 2;;
            -pkgname) PKGNAME=$2;shift 2;;
            -taskid) TASKID=$2;shift 2;;
            -upgradepath) upgradepath=$2;shift 2;;
            -pwfile) pwfile=$2;shift 2;;
            *) upgrade_params="${upgrade_params} ${1} ${2}";shift 2;;
        esac
    done

    # 拓展校验
    echo "${ACTION}" | grep -qE 'step1|step2|default_upgrade|post_upgrade|kernel_upgrade|app_upgrade'
    if [ $? -ne 0 ]
    then
        echo "unsupported params for ACTION(${ACTION})";
        exit 1
    fi
    echo "parse input params finished"
}

##################################################
# 校验omp包是否存在
# 同时拷贝omp包到/opt/upgrade工作路径
# 前台已做完整性校验，不再重复校验
##################################################
function copy_omp_workpath()
{
    if [ ${NEW_TASK} != "TRUE" -a -d "${OMP_PATH}" ]
    then
        return 0
    fi
    # 新建升级任务重新解压
    rm -rf ${OMP_PATH}

    unzip_omp_package || return $?
    return 0
}

##################################################
# 解压OMP软件包
##################################################
function unzip_omp_package()
{
    if [ -d "${OMP_PATH}" ]
    then
        return 0
    fi
    if [ -f "/opt/pub/software/mgr-installdisk/${PKGNAME}" ]
    then
        unzip /opt/pub/software/mgr-installdisk/${PKGNAME} -d ${OMP_PATH} >>${LOG_FILE} 2>&1
        if [ $? -ne 0 ]
        then
            TASK_LOG "ERROR" "Failed to unzip (${PKGNAME})."
            fresh_fail
            return 1
        fi
    else
        TASK_LOG "ERROR" "The software package (${PKGNAME}) is not uploaded. "
        fresh_fail
        return 1
    fi
}

##################################################
# 校验OMP版本
# 跳过OMP已成功安装场景
##################################################
function check_omp_version()
{
    local omppatch_version="$(bash ${UPGRADE_SCRIPT} -action query_version)"
    local omp_version="$(cat /opt/oss/manager/var/share/software_define.yaml | grep "^softwareVersion" | awk '{print $2}')"
    UPGRADE_FLAG="$(bash ${UPGRADE_SCRIPT} -action query_flag)"
    if [ "${omppatch_version}" == "${omp_version}" ];then
        # 版本为相同版本，并且安装状态为已安装完OMP，直接跳过安装
        grep "UpgradeOMP=success" "${UPGRADE_FLAG}" >>${LOG_FILE} 2>&1
        if [ $? -eq 0 -o ! -f "${UPGRADE_FLAG}" ]
        then
            TASK_LOG "WARN" "The software version is the same as the patch package version, skip the patch installation. "
            post_upgrade
            fresh_finish
            return 1
        fi
    fi
    return 0
}

##################################################
# 备份OMP升级日志
##################################################
function pack_omp_log()
{
    # 回滚完成删除OMP包、日志打包
    local omp_log_path="${OMP_PATH}/logs"
    local easysuite_log_path="/opt/oss/log/manager/easysuite_upgrade/"
    local task_id_log_path="${easysuite_log_path}/${TASKID}.zip"
    if [ -d "${omp_log_path}" -a ! -f "${task_id_log_path}" ]
    then
        zip -j ${task_id_log_path} ${omp_log_path}/*
        chmod 440 ${task_id_log_path}
    fi
    rm -rf ${OMP_PATH} >/dev/null 2>&1
}

##################################################
# 升级后置处理，修复/opt/oss/log目录下可能存在
# root权限OMP目录的问题
##################################################
function post_upgrade()
{
    ls -l /opt/oss/log/ 2>/dev/null | grep 'OMP' | grep -q 'root'
    if [ $? -eq 0 ]
    then
        TASK_LOG "INFO" "Start to modify directory permissions."
        pre_file="/opt/oss/manager/agent/DeployAgent/tools/shscript/upgradedb/preupgrade/01_open_dbuser_ssh.sh"
        post_file="/opt/oss/manager/agent/DeployAgent/tools/shscript/upgradedb/postupgrade/01_close_dbuser_ssh.sh"
        if [ -f "${pre_file}" -a -f "${post_file}" ]
        then
            bash ${pre_file} &>>${LOG_FILE}
            bash ${post_file} &>>${LOG_FILE}
        fi
        TASK_LOG "INFO" "Finished to modify directory permissions."
    fi
}

##################################################
# 防止文件句柄被大量占用，touch失败
##################################################
function create_result_flag()
{
    local flag_file=$1
    TASK_LOG "INFO" "Start to create the ${flag_file} file."
    for i in $(seq 10)
    do
        [ ! -f "${flag_file}" ] && touch "${flag_file}"
        if [ -f "${flag_file}" ]
        then
            TASK_LOG "INFO" "The ${flag_file} file is created successfully."
            return 0
        fi
        sleep 1
    done
    TASK_LOG "ERROR" "Failed to create the ${flag_file} file."
    return 1
}

##################################################
# 刷新第二阶段Easysuite进度信息
##################################################
function fresh_step2()
{
    if [ -n "${UPGRADE_FLAG}" -a -f "${UPGRADE_FLAG}" ]
    then
        sed -i '/failure/d' "${UPGRADE_FLAG}"
    fi
    local success_flag=$1
    local failed_flag=$2
    CHGRP_FLAG="TRUE"
    while true
    do
        change_group
        # 1、失败或成功标志文件存在，进入判断逻辑
        if [ -f "${success_flag}" -o -f "${failed_flag}" ]
        then
            break
        fi
        if [ ! -f "${UPGRADE_FLAG}" ]
        then
            sleep 1
            continue
        fi
        # 考虑模拟进度
        sleep 1
    done
}

##################################################
# 如果升级前无sopgroup用户组
# 升级完ICMR之后，切换升级工作目录属组为sopgroup
##################################################
function change_group()
{
    if [ ${CHGRP_FLAG} == 'FALSE' ]
    then
        return 0
    fi
    id -gn sopuser 2>/dev/null
    if [ $? -eq 0 ]
    then
        # 保证taskmgr及其子目录sopuser有写权限
        setfacl -m u:sopuser:r-x ${UPGRADE_PATH}
        setfacl -m u:sopuser:r-x ${UPGRADE_PATH}/easysuite_upgrade
        setfacl -m u:sopuser:r-x ${UPGRADE_PATH}/easysuite_upgrade/scripts
        setfacl -m u:sopuser:r-x ${UPGRADE_PATH}/easysuite_upgrade/databasemigrate
        setfacl -m u:sopuser:r-x ${UPGRADE_PATH}/easysuite_upgrade/upgrade_frame_scripts
        find ${UPGRADE_PATH}/easysuite_upgrade/upgrade_frame_scripts -type f -user ossadm | xargs setfacl -m u:sopuser:r-x
        find ${UPGRADE_PATH}/easysuite_upgrade/taskmgr -type f -user ossadm | xargs setfacl -m u:sopuser:r--
        find ${UPGRADE_PATH}/easysuite_upgrade/taskmgr -type d -user ossadm | xargs setfacl -m u:sopuser:r-x
        CHGRP_FLAG="FALSE"
    fi
}

##################################################
# 刷新第一阶段Easysuite进度信息
##################################################
function fresh_step1()
{
    if [ -n "${UPGRADE_FLAG}" -a -f "${UPGRADE_FLAG}" ]
    then
        sed -i '/failure/d' "${UPGRADE_FLAG}"
    fi
    local success_flag=$1
    local failed_flag=$2
    while true
    do
        # 1、失败或成功标志文件存在，进入判断逻辑
        if [ -f "${success_flag}" -o -f "${failed_flag}" ]
        then
            if [ -f "${failed_flag}" ]
            then
                fresh_status "fail"
            elif [ -f "${success_flag}" ]
            then
                pack_omp_log
                fresh_status "finish"
            fi
            break
        fi
        if [ ! -f "${UPGRADE_FLAG}" ]
        then
            sleep 1
            continue
        fi
        # 考虑模拟进度
        sleep 1
    done
}

##################################################
# 升级sudoconfig-osconfig-icmr
##################################################
function upgrade_step1()
{
    copy_omp_workpath || return $?
    UPGRADE_FLAG="$(bash ${UPGRADE_SCRIPT} -action query_flag)"
    grep -qi 'UpgradeICMR=success' ${UPGRADE_FLAG}
    if [ $? -eq 0 ]
    then
        TASK_LOG "INFO" "Upgrade sudoconfig/osconfig/icmr Successfully."
        fresh_finish
        return 0
    fi

    upgrade_omp_paras="-action step1"

    local TAIL=$(date +%Y%m%d%H%M%S)
    local success_flag="${OMP_PATH}/step1_success_${TAIL}.flag"
    local fail_flag="${OMP_PATH}/step1_failed_${TAIL}.flag"

    bash ${UPGRADE_SCRIPT} ${upgrade_omp_paras} >>${TASK_LOG_FILE} 2>&1 && create_result_flag ${success_flag} || create_result_flag ${fail_flag} &

    fresh_step1 "${success_flag}" "${fail_flag}" &
    wait
    # 只刷新结束状态
    fresh_progress "100"
}

##################################################
# 配置升级OMP时默认密码
##################################################
function get_user_config_value()
{
    # 目录下文件存在不再二次生成
    [ -f "${OMP_PATH}/nce_upgrade_params.json" ] && return 0
    #替换user_config_value
    . ${CLOUDSOP_PROFILE}
    (echo ${omp_default_key}; echo ${omp_new_sopuser};) | python -c "import sys;sys.path.append('${SCRIPT_PATH}');from create_pwfile import create_pw_file;create_pw_file('${OMP_PATH}')" >>${LOG_FILE} 2>&1
}

##################################################
# 升级uniep-配置用户隔离-配置端口监听
##################################################
function upgrade_step2()
{
    # Case:step2、post_upgrade、kernel_upgrade、app_upgrade
    local step_action="${1}"

    # 前置
    if [ "${step_action}" == "step2" -o "${step_action}" == "kernel_upgrade" ]
    then
        # 解压omp
        copy_omp_workpath || return $?
        check_omp_version || return 0
    fi
    # 做第二遍检查避免软件包不存在
    unzip_omp_package || return $?
    get_user_config_value
    upgrade_omp_paras="-action ${step_action}"

    # 管理面升级认证信息参数
    [ -z ${pwfile} ] && pwfile="${OMP_PATH}/nce_upgrade_params.json"
    upgrade_omp_paras="${upgrade_omp_paras} ${upgrade_params} -pwfile ${pwfile}"
    TASK_LOG "INFO" "Upgrade omp params:${upgrade_omp_paras}"

    local TAIL=$(date +%Y%m%d%H%M%S)
    local success_flag="${OMP_PATH}/${step_action}_success_${TAIL}.flag"
    local fail_flag="${OMP_PATH}/${step_action}_failed_${TAIL}.flag"
    # 如果存在标志文件,删除旧的标志文件
    [ -f "${success_flag}" ] && rm -f "${success_flag}"
    [ -f "${fail_flag}" ] && rm -f "${fail_flag}"

    bash ${UPGRADE_SCRIPT} ${upgrade_omp_paras} >>${TASK_LOG_FILE} 2>&1 && create_result_flag ${success_flag} || create_result_flag ${fail_flag} &
    sleep 3
    # 再启一个线程进程进度刷新
    fresh_step2 "${success_flag}" "${fail_flag}" &
    wait

    # 结束时
    step_status="fail"
    if [ -f "${success_flag}" ]
    then
        step_status="success"
    fi

    if [ "${step_status}" == "success" ]
    then
        if [ "${step_action}" == "step2" -o "${step_action}" == "default_upgrade" -o "${step_action}" == "app_upgrade" ]
        then
            # 管理面升级核心操作成功升级后置
            post_upgrade
        fi
        if [ "${step_action}" == "step2" -o "${step_action}" == "post_upgrade" ]
        then
            # 所有步骤的最后的后置操作OK,打包日志文件
            pack_omp_log
        fi
        fresh_status "finish"
    elif [ "${step_status}" == "fail" ]
    then
        fresh_status "fail"
    fi
    # 只刷新结束状态
    fresh_progress "100"
}

##################################################
# 脚本主入口
##################################################
function main()
{
    # 加载公共方法
    . ${SCRIPT_PATH}/../common.sh
    LOG "INFO" "Start to init log."
    # 校验入参，初始化参数

    get_exec_param $@ || return 1
    if [ -n "${upgradepath}" ];then
        pre_custom_upgrade_config_file="/opt/upgrade/backup/workpath/upgrade_params_config_manager_${upgradepath}.json"
        if [ -f "${pre_custom_upgrade_config_file}" ];then
            upgrade_omp_params=$(cat ${pre_custom_upgrade_config_file} 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(\" \".join(obj['upgrade_omp_params'])); " 2>/dev/null)
        fi
        if [ -n "${upgrade_omp_params}" ];then
             get_exec_param ${upgrade_omp_params} || return 1
        fi
    fi

    init_variable

    LOG "INFO" "Parse input params succeed. "

    init_taskmgr ${TASKID}

    # 根据入参执行各自方法
    case ${ACTION} in
    "step1")
        upgrade_step1
        return $?
    ;;
    *)
        upgrade_step2 "${ACTION}"
        return $?
    ;;
    esac
}
