# encoding=utf-8
"""
功 能：备份产品数据任务下发,获取任务url
版权信息：华为技术有限公司，版本所有(C) 2020-2029
修改记录：2020-07-07 12:00 创建
"""
import json
import os
import sys
import time

from util import httpclient
from commonlog import Logger

logger = Logger().getinstance(sys.argv[0])
INSTANCE = ["mysql", "gauss"]
NO_NEED_BACKUP_DB_TYPE = ["redis"]
LOGIC_BACKUP = "text"
RUNNING_STATE = ["INIT", "RUNNING", "WAITING"]
FAILURE_STATE = ["FAILURE", "PARTIAL_SUCCESS", "TERMINATING", "TERMINATED"]


class BackupProductData:
    """
    功能描述：备份产品数据任务下发,获取任务url
    接口：
    """
    @staticmethod
    def gauss_db_bct(product_name, action):
        """
        开启或关闭高斯数据库BCT
        :param product_name: 产品名
        :param action: 操作类型
        :return:
        """
        cmd = "bash /opt/oss/manager/apps/UniEPAgent/bin/uniep_tool backup bct -project %s -operate %s" \
              % (product_name, action)
        for _ in range(0, 2):
            ret_code = os.system(cmd)
            if ret_code == 0:
                logger.info("[gauss_db_bct] action: %s, retcode: %s" % (action, ret_code))
                break
            else:
                time.sleep(1)
        if ret_code != 0:
            logger.error("[gauss_db_bct] action: %s, retcode: %s" % (action, ret_code))
            return False
        return True

    @staticmethod
    def get_full_db_instace(ssh_node, product_name, uniep_version, backup_type):
        """
        获取所有数据库实例
        :param:  ssh_node ssh登录信息（ip，username，port等）
        :return: 执行结果（True：成功，Fasle：失败）
        :return: db_instances 数据库实例结构体
        """
        db_instance_url = "/rest/plat/dbmgr/v1/main/instances/action?action-id=export-containerlist"
        user_list = ["adminUser", "dbUser", "readUser", "rplUser"]

        params = dict(
            tenant=product_name,
            userList=user_list
        )
        if "R019" in uniep_version:
            params.update({
                "userName": ssh_node.get("username"),
                "clientIP": ssh_node.get("ip")
            })
        if "R020" in uniep_version:
            params.update({"excludeType": ["gauss"]})
            NO_NEED_BACKUP_DB_TYPE.append("gauss")
        try:
            db_infos_list = []
            status, response = httpclient.IRHttpClient().post(db_instance_url, params)
            db_dict = json.loads(response)
            if status != 200:
                return False, None
            for key, value in db_dict.items():
                if not value.get("dbList"):
                    continue
                db_infos = BackupProductData.get_db_infos(value)
                if db_infos:
                    db_infos_list.append(db_infos)
            db_instances = BackupParameter.get_database_body(db_infos_list)
            if backup_type:
                db_instances.update(globalDBBackupMode=backup_type)
            return True, db_instances
        except Exception:
            return False, None

    @staticmethod
    def get_db_infos(sub_dict):
        """
        获取单个数据库实例
        :param:  sub_dict 单个的数据库实例结构体
        :return: db infos 单个数据库实例信息
        """
        # 构造获取数据库实例的url和body体
        db_infos = {}
        instance_id = sub_dict.get("dbInstanceID")
        instance_type = sub_dict.get("containerType")
        if instance_type not in NO_NEED_BACKUP_DB_TYPE:
            db_instance_id = instance_id
            db_name_list = list(sub_dict.get("dbList").keys())
            if instance_type.lower() in INSTANCE:
                mode = LOGIC_BACKUP
            else:
                mode = "binary"
            db_infos = BackupParameter.get_db_infos_body(db_instance_id, mode, db_name_list)
        return db_infos

    @staticmethod
    def get_backup_product_url(backup_product_para):
        """
        :param backup_product_para:
        :return:
        """
        backup_product_url = "/rest/plat/brmgr/v1/backupforspec/backupdataaction?action=backup"
        status, response = httpclient.IRHttpClient().post(backup_product_url, backup_product_para)
        backup_result = json.loads(response)
        return status, backup_result.get("url")


class BackupParameter:
    """
    生成备份操作相关接口所需要的参数
    """

    @staticmethod
    def get_db_infos_body(instance_id, mode, db_name):
        """
        构造数据库body体信息
        :param:  instanceID 数据库实例ID
        :param:  mode 模式
        :param:  db_name 数据库实例名称
        :return: 数据库body体
        """
        body = dict(
            dbInstanceID=instance_id,
            mode=mode,
            dbCompressedParam="-parallel=4 -level=1",
            dbCompressedMode="zstd",
            dbName=db_name
        )
        return body

    @staticmethod
    def get_database_body(db_infos_list):
        """
        构造数据库列表body体信息
        :param:  db_infos_list 数据库信息列表
        :return: 数据库列表结构体
        """
        body = dict(
            dbinfo=db_infos_list,
            concurrency=3,
            compressionRatio=-1,
            dbBackupConcurrentParam=4
        )
        return body

    @staticmethod
    def get_service_list_body():
        """
        构造微服务列表body体
        :param:  None
        :return: 服务列表结构体
        """
        body = dict(
            serviceList=[
                {
                    "serviceName": "ALL"
                }
            ],
            concurrency=3,
            compressionRatio=5
        )
        return body

    @staticmethod
    def config_backup_product_body(product_name, db_instance, other, remote_backup):
        """
        构造备份产品数据的body体
        :param: productName 产品名称
        :param: db_instance 数据库实例结构体
        :param: service_list 服务列表
        :param: remote_backup 是否上传备份服务器
        :return: 备份产品数据结构体
        """
        body = dict(
            action="backup",
            mode="upgrade",
            database=db_instance,
            other=other,
            needToUploadToBackupServer=remote_backup,
            productName=product_name,
            mutexLevel="none",
            backupPath="/opt/oss/backuptmp",
            resource={
                "mode": {
                    "en_US": "Upgrade",
                    "zh_CN": "升级"
                }
            }
        )
        return body


def backup_product_service(ssh_node, product_name, uniep_version, upgrade_path, remote_backup, backup_type):
    """
    备份产品数据模块入口
    :param ssh_node:
    :param product_name:
    :param uniep_version:
    :param upgrade_path:
    :param remote_backup:
    :param backup_type:
    :return:
    """
    result = False
    backup_product_result = {"status": 0, "url": "", "msg": ""}
    for _ in range(0, 2):
        result, db_instance = BackupProductData.get_full_db_instace(ssh_node, product_name, uniep_version, backup_type)
        if result:
            break
        time.sleep(3)
    if not result:
        backup_product_result["status"] = 400
        backup_product_result["msg"] = "Failed to get full db instace."
        stdout = json.dumps(backup_product_result, ensure_ascii=False)
        print(stdout, end="")
        return None
    other = BackupParameter.get_service_list_body()
    # ------------------------------------------------------
    db_instance = get_backup_db_instance(backup_product_result, db_instance, product_name, upgrade_path)

    if os.path.exists("other.json"):
        with open("other.json") as file:
            other = json.load(file)

    backup_product_para = BackupParameter.config_backup_product_body(product_name, db_instance, other, remote_backup)
    status, backup_result = BackupProductData.get_backup_product_url(backup_product_para)

    backup_product_result["status"] = status
    backup_product_result["url"] = backup_result
    backup_product_result["db_instance"] = db_instance
    stdout = json.dumps(backup_product_result, ensure_ascii=False)
    print(stdout, end="")
    return backup_product_result


def get_backup_db_instance(backup_product_result, db_instance, product_name, upgrade_path):
    """
    读取备份数据库实例配置数据
    :param backup_product_result:
    :param db_instance:
    :param product_name:
    :param upgrade_path:
    :return:
    """
    file_dest = "/opt/upgrade/backup/workpath/upgrade_params_config_{}_{}.json".format(product_name,
                                                                                       upgrade_path)
    if not os.path.exists(file_dest):
        return db_instance
    #
    with open(file_dest) as file:
        upgrade_params_config = json.load(file)
        if upgrade_params_config:
            database = upgrade_params_config.get("database")
            if database == {}:
                backup_product_result["status"] = 200
                backup_product_result["url"] = "not to need backup Product data"
                stdout = json.dumps(backup_product_result, ensure_ascii=False)
                print(stdout, end="")
            if database:
                db_instance = database
    return db_instance


def main(argv):
    """
    :param argv:
    :return:
    """
    ssh_node = {
        "ip": argv[1],
        "username": argv[2]
    }
    product_name = argv[3]
    uniep_version = argv[4]
    upgrade_path = argv[5]
    remote_backup = argv[6]
    backup_product_with_diff = argv[7]
    backup_type = None
    try:
        if backup_product_with_diff == "true":
            # BCT打开前先关闭再打开
            BackupProductData.gauss_db_bct(product_name, "off")
            BackupProductData.gauss_db_bct(product_name, "on")
            backup_type = "FULL_BACKUP"
        if not backup_product_service(ssh_node, product_name, uniep_version, upgrade_path, remote_backup, backup_type):
            return False
    except Exception as exception:
        stdout = json.dumps({"status": 400, "url": "", "msg": str(exception)}, ensure_ascii=False)
        print(stdout, end="")


if __name__ == '__main__':
    main(sys.argv)
