# -*- coding: utf-8 -*-

import os
import ast
import sys
import json
import logging

LOG = logging.getLogger(sys.argv[0])
CONTAIN_LIST = "/opt/upgrade/backup/migrate/new/containerlist.json"
OUT_CONTAIN_LIST = "/opt/upgrade/backup/migrate/new/containerlist_new.json"


def init_log():
    # 初始化日志
    LOG.setLevel(logging.DEBUG)
    log_path = os.path.join(os.getcwd(), "logs")
    if not os.path.exists(log_path):
        os.mkdir(log_path)
    log_file = os.path.join(log_path, "modify_containlist.log")
    fh = logging.FileHandler(log_file, mode="w")
    formatter = logging.Formatter("%(asctime)s - %(filename)s[line:%(lineno)d] - %(levelname)s: %(message)s")
    fh.setFormatter(formatter)
    fh.setLevel(logging.DEBUG)
    ch = logging.StreamHandler()
    ch.setLevel(logging.WARNING)
    ch.setFormatter(formatter)
    LOG.addHandler(fh)
    LOG.addHandler(ch)


class ModifyContainlist:

    def __init__(self):
        self.input_dict = {}

    def pre_check(self, input_params):
        """
        功能描述：1、将输入参数转换为字典2、判断containerlist.json
        返回值：true、false
        """
        try:
            if not os.path.isfile(CONTAIN_LIST):
                LOG.error(
                    "The database configuration file does not exist.path:%s" % CONTAIN_LIST)
                return False
            self.input_dict = ast.literal_eval(input_params)
            LOG.debug("Input dict:%s" % self.input_dict)
            return True
        except Exception as e:
            LOG.error(
                "Incorrect input format.Example:python %s \"{'ODRSZenith':['cmdbcoresvrdb','eamdb']}\"" %
                sys.argv[0])
            return False

    def rename_contain_list(self):
        """
        功能描述：根据入参的字典，将数据库连接文件对应的实例下的数据库名修改
        返回值：true、false
        """
        try:
            with open(CONTAIN_LIST, "r") as file_obj:
                file_data = json.load(file_obj)
        except Exception as e:
            LOG.error("Failed to import the configuration.MSG:%s" % e)
            return False
        tmp_ins = file_data.keys()
        # 仅当最外层存在实例名时才去进行循环
        if tmp_ins:
            self.rename_db(file_data, tmp_ins)
        return self.save_contain_file(file_data)

    def rename_db(self, file_data, tmp_ins):
        for ins, re_dblist in self.input_dict.items():
            for dbins in tmp_ins:
                if ins not in dbins:
                    continue
                dblist = file_data.get(dbins, {}).get('dbList', {})
                if not dblist:
                    continue
                tmp_dblist = []
                for db in dblist.keys():
                    tmp_dblist.append(db)
                ModifyContainlist.del_old_name_db(dbins, dblist, file_data, re_dblist, tmp_dblist)

    @staticmethod
    def del_old_name_db(dbins, dblist, file_data, re_dblist, tmp_dblist):
        for dbname in tmp_dblist:
            if dbname not in re_dblist:
                continue
            tmp_dbname = "%s_%s" % (dbname, "new")
            file_data[dbins]['dbList'].update({tmp_dbname: dblist.get(dbname)})
            del file_data[dbins]['dbList'][dbname]

    @staticmethod
    def save_contain_file(file_data):
        """
        功能说明:保存contain_file
        :param file_data:
        :return:
        """
        try:
            with os.fdopen(os.open(OUT_CONTAIN_LIST,
                                   os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                                   mode=0o600), "w") as f:
                json.dump(file_data, f)
        except Exception as e:
            LOG.error("Failed to export the configuration.MSG:%s" % e)
            return False
        try:
            os.remove(CONTAIN_LIST)
            os.rename(OUT_CONTAIN_LIST, CONTAIN_LIST)
            return True
        except Exception as e:
            LOG.error("Failed to modify the file.MSG:%s" % e)
            return False


def main():
    modify = ModifyContainlist()
    if len(sys.argv) != 2:
        return False
    arg = str(sys.argv[1])
    if modify.pre_check(arg):
        return modify.rename_contain_list()
    else:
        return False


if __name__ == '__main__':
    init_log()
    if main():
        sys.exit(0)
    else:
        sys.exit(1)
