# encoding=utf-8
"""
功 能：异地容灾操作
版权信息：华为技术有限公司，版本所有(C) 2020-2029
修改记录：2020-04-23 12:00 创建
"""

import json
import os
import shutil
import subprocess
import sys
import time
from datetime import datetime

from commonlog import Logger
from taskmgr_util import Taskmgrutil
from uniep_taskmgr import Unieptaskmgr
from util import httpclient

ADD_DR_COST_TIME = 60 * 180  # 等待容灾数据同步完成的超时时间，3小时

LOG = Logger().getinstance(sys.argv[0])

DR_ARB_URL = "/rest/drservice/v1/main/drmgr/custom/queryParam"
DR_MGR_URL = "/rest/drmgrservice/v1/main/drmgr"
EDIT_DR_URL = "/rest/drmgrservice/v1/main/drmgr/editdr"
BASE_DR_DATA = {
    "drinfo": {
        "customParamExtensionJSON": {
            "NCEProductCheckParamsTab1": {
                "autoMigrateSwitch": "",
                "arbiterType": "",
                "arbiterIp": "",
                "hbRetryInterval": "",
                "hbRetryTimes": "",
                "switchDelay": "",
                "isFastSwitchOverEnable": "",
                "isAppArbitrationOn": "",
                "isNetworkArbitrationOn": "",
                "isDBArbitrationOn": "",
                "isRedisArbitrationOn": "",
                "isAppArbiDelayTimeOn": "",
                "isNetWorkFrontEndSwitchHide": ""
            }
        },
        "id": "",
        "primarySiteName": "",
        "secondarySiteName": ""
    },
    "clientIP": "",
    "healthCheckConfiguration": {
        "startTime": "",
        "timedTaskSwitch": ""
    },
    "userName": ""
}


def set_task_data(e_task_path, status='', progress='', msg=''):
    """
    功能说明:更新升级任务配置数据
    :param e_task_path:
    :param status:
    :param progress:
    :param msg:
    :return:
    """
    if status:
        Taskmgrutil.set_e_taskstatus(e_task_path, status)
    if progress:
        Taskmgrutil.set_e_taskprogress(e_task_path, progress)
    if msg:
        Taskmgrutil.set_msg_append(e_task_path, msg)


def create_dr_site(e_task_path, dr_info_file, action_type, dr_consistent, product_name, rep_direction):
    """
    功能说明:汇总,统一处理内部异常,避免异常界面不报错
    :param e_task_path:
    :param dr_info_file:
    :param action_type:
    :param dr_consistent:
    :param product_name:
    :param rep_direction:
    :return:
    """
    try:
        result = DROperationService.add_dr_operation_service(e_task_path, dr_info_file,
                                                             action_type, dr_consistent,
                                                             product_name, rep_direction)
    except Exception as e:
        error_msg = f"Failed to create dr site.Exception:{e}"
        LOG.error(error_msg)
        set_task_data(e_task_path, status='error', progress=100, msg=error_msg)
        return False
    if not result:
        set_task_data(e_task_path, status='error', progress=100)
        return False
    set_task_data(e_task_path, status='success', progress=100)
    return True


def read_file_data(file_path):
    """
    功能说明:读取json文件数据
    :param file_path:
    :return:
    """
    # 文件不存在退出
    if not os.path.isfile(file_path):
        return False
    with open(file_path, 'r') as r_stream:
        data = json.load(r_stream)
    return data


def get_drservice_data():
    """
    功能说明:查询Common容灾服务保存数据
    :return:
    """
    params = {'isEdit': 'false'}
    for _ in range(10):
        status, response = httpclient.IRHttpClient().post(DR_ARB_URL, params)
        if status == 200 and response:
            break
        time.sleep(3)
    else:
        return False
    return json.loads(response)


def get_drmgrserice_data():
    """
    功能说明:获取平台容灾服务保存数据
    :return:
    """
    for _ in range(10):
        status, response = httpclient.IRHttpClient().get(DR_MGR_URL)
        if status == 200 and response:
            break
        time.sleep(3)
    else:
        return False
    return json.loads(response)


def build_extend_arb_data(base_data):
    """
    功能说明:构造仲裁页面拓展信息
    :param base_data:
    :return:
    """
    drservice_data = get_drservice_data()
    lower_arb_keys = {"hbRetryInterval": "hb_retry_interval",
                      "hbRetryTimes": "hb_retry_times",
                      "switchDelay": "autoswitch_delaytime"}
    arb_keys = ['autoMigrateSwitch', 'arbiterType', 'arbiterIp',
                'isFastSwitchOverEnable', 'isAppArbitrationOn', 'isNetworkArbitrationOn',
                'isDBArbitrationOn', 'isRedisArbitrationOn', 'isAppArbiDelayTimeOn',
                'isNetWorkFrontEndSwitchHide']
    arb_tab = base_data.get('drinfo').get('customParamExtensionJSON')\
        .get('NCEProductCheckParamsTab1')
    for one_key in arb_keys:
        if one_key in drservice_data:
            arb_tab.update({one_key: drservice_data.get(one_key)})
        elif one_key in arb_tab:
            arb_tab.pop(one_key)
    for one_key, one_value in lower_arb_keys.items():
        if one_value in drservice_data:
            arb_tab.update({one_key: drservice_data.get(one_value)})
        elif one_key in arb_tab:
            arb_tab.pop(one_key)


def build_edit_dr_data(dr_health_time_file):
    """
    功能说明:构造编辑容灾所需数据
    :param dr_health_time_file:
    :return:
    """
    # id startTime timedTaskSwitch
    dr_health_data = read_file_data(dr_health_time_file)
    dr_mgr_data = get_drmgrserice_data().get('data')[0]
    base_data = BASE_DR_DATA
    base_data.update({"clientIP": '127.0.0.1'})
    base_data.update({"userName": 'admin'})
    base_data.get('healthCheckConfiguration').update({"startTime":
                                                          dr_health_data.get('startTime')})
    base_data.get('healthCheckConfiguration').update({"timedTaskSwitch":
                                                          dr_health_data.get('timedTaskSwitch')})
    base_data.get('drinfo').update({"id": dr_mgr_data.get('id')})
    base_data.get('drinfo').update({"primarySiteName": dr_mgr_data.get('primarySiteName')})
    base_data.get('drinfo').update({"secondarySiteName": dr_mgr_data.get('secondarySiteName')})
    build_extend_arb_data(base_data)
    return base_data


def edit_health_time(dr_health_time_file):
    """
    功能说明:刷新容灾健康检查时间
    :param dr_health_time_file:
    :return:
    """
    params = build_edit_dr_data(dr_health_time_file)
    for _ in range(10):
        status, response = httpclient.IRHttpClient().post(EDIT_DR_URL, params)
        if status == 200 and response:
            break
        time.sleep(3)
    else:
        return False
    return True


def get_rep_direction(rep_direction, dr_consistent):
    """
    功能说明:获取转换后容灾复制方向
    :param rep_direction:
    :param dr_consistent:
    :return:
    """
    all_rep_direction = ["PRIMARY_TO_SECONDARY", "SECONDARY_TO_PRIMARY"]
    if dr_consistent == 'true':
        return rep_direction
    all_rep_direction.remove(rep_direction)
    return all_rep_direction[0]


class DROperationService:
    """
    功能描述：异地容灾操作功能，包括删除容灾、创建容灾和容灾倒换
    接口：
    """
    add_dr_start_time = 0

    @staticmethod
    def get_install_time():
        """
        功能描述：获取安装时间
        返回： 安装时间
        修改记录：新增方法
        """
        url = DR_ARB_URL
        params = {'isEdit': 'false'}
        status, response = httpclient.IRHttpClient().post(url, params)
        if status != 200:
            LOG.warning(
                "[get_install_time] Failed to invoke interface. retCode: %s." % str(status))
            return False
        custom_info = json.loads(response)
        installed_time = custom_info.get("installedTime")
        return installed_time

    @staticmethod
    def update_dr_info_file(dr_info_file):
        """
        功能描述：获取当前异地容灾环境信息并刷新容灾配置文件
        参数：容灾配置文件
        返回： 成功/失败
        修改记录：新增方法
        """
        url = DR_MGR_URL
        status, response = httpclient.IRHttpClient().get(url)
        if status != 200:
            LOG.warning(
                "[update_dr_info_file] Failed to invoke interface. retCode: %s." % str(status))
            return False
        if not json.loads(response).get("data"):
            LOG.warning("[update_dr_info_file] Response is null.")
            return False
        drs_info = json.loads(response)["data"][0]
        if not drs_info:
            return False

        data = DROperationService.get_dr_config_info(dr_info_file, drs_info)
        LOG.info("[update_dr_info_file] drinfo.json queryed python: %s" % str(data))

        with os.fdopen(os.open(dr_info_file,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o660), 'w') as json_file:
            json_file.write(json.dumps(data))
        return True

    @staticmethod
    def get_dr_config_info(dr_info_file, drs_info):
        """
        功能说明:获取容灾配置数据
        :param dr_info_file:
        :param drs_info:
        :return:
        """
        dr_rep_direction = []
        primary_products_name = []
        secondary_products_name = []
        primary_site_name = drs_info.get("primarySiteName")
        secondary_site_name = drs_info.get("secondarySiteName")
        primary_site_ip = drs_info.get("primaryHBIPs")
        secondary_site_ip = drs_info.get("secondaryHBIPs")
        master_site_ip = ";".join(primary_site_ip)
        dr_site_ip = ";".join(secondary_site_ip)
        products = drs_info.get("products")
        for product in products:
            primary_product_name = product.get("primaryProductName")
            primary_products_name.append(primary_product_name)
            secondary_product_name = product.get("secondaryProductName")
            secondary_products_name.append(secondary_product_name)
            primary_product_status = product.get("primaryProductStatus")
            if primary_product_status == "standby":
                rep_direction = "PRIMARY_TO_SECONDARY"
            else:
                rep_direction = "SECONDARY_TO_PRIMARY"
            dr_rep_direction.append(rep_direction)
        installed_time = DROperationService.get_install_time()
        shutil.copyfile(dr_info_file, "%s.bak" % dr_info_file)
        with open("%s.bak" % dr_info_file, 'r', encoding='utf-8') as file_dr:
            data = json.load(file_dr)
        data["drinfo"]["primarySiteProductName"] = primary_products_name
        data["drinfo"]["secondarySiteProductName"] = secondary_products_name
        data["drinfo"]["primarySiteName"] = primary_site_name
        data["drinfo"]["primarySiteIP"] = primary_site_ip
        data["drinfo"]["secondarySiteName"] = secondary_site_name
        data["drinfo"]["secondarySiteIP"] = secondary_site_ip
        data["drinfo"]["repDirection"] = dr_rep_direction
        ext_info = data["drinfo"]["extInfo"][0]["info"]
        dr_ext_config = {"masterSiteIP": master_site_ip,
                         "drSiteIP": dr_site_ip,
                         "primarySiteName": primary_site_name,
                         "secondarySiteName": secondary_site_name,
                         "installedTime": installed_time}
        for k_key in dr_ext_config:
            for info in ext_info:
                if info["key"] == k_key:
                    info["value"] = dr_ext_config[k_key]
        return data

    @staticmethod
    def query_dr_product_name():
        """
        功能描述：查询异地容灾环境产品名称
        参数：无
        返回： 产品名称列表
        修改记录：新增方法
        """
        url = DR_MGR_URL
        status, response = httpclient.IRHttpClient().get(url)
        if status != 200:
            LOG.warning(
                "[query_dr_product_name] Failed to invoke interface. retCode: %s." % str(status))
            return False
        info = json.loads(response)["data"]
        if len(info) <= 0:
            return []
        drs_info = json.loads(response)["data"][0]
        primary_products_name = []
        secondary_products_name = []
        if len(drs_info) > 0:
            products = drs_info.get("products")
            for product in products:
                primary_product_name = product.get("primaryProductName")
                primary_products_name.append(primary_product_name)
                secondary_product_name = product.get("secondaryProductName")
                secondary_products_name.append(secondary_product_name)

        return primary_products_name

    @staticmethod
    def query_dr_data():
        """
        功能描述：查询容灾环境容灾关系
        参数：无
        返回：True or False
        修改记录：新增方法
        """
        url = DR_MGR_URL
        status, response = httpclient.IRHttpClient().get(url)
        if status != 200:
            LOG.warning("[query_dr_data] Failed to invoke interface.\
             retCode: %s." % str(status))
            return False
        if json.loads(response)["data"]:
            return True
        return False

    @staticmethod
    def query_operation_result(operation_result, e_task_path, operation):
        """
        功能描述：查询异地容灾操作结果
        参数：任务查询URL，任务路径
        返回： 任务进度及详情
        修改记录：新增方法
        """
        start_time = DROperationService.get_date_time()
        Taskmgrutil.set_msg_append(e_task_path, f"[{start_time}]"
                                                f" Start to query the task result.")
        site_name = "standby" if operation != "migrate" else "active"
        error_tips = f"For more details, log in to the management plane of the {site_name} site and " \
                     "view the task list. If the task has not been created, choose System > Log Management " \
                     "> System Logs from the main menu to view the failure information."
        try:
            result_info = json.loads(operation_result)[0]
            result_url = result_info.get("resultUri")
            task_progress = 0
            while task_progress < 100:
                result, response = DROperationService.uniep_get_request(result_url)
                if not result:
                    return False, "Query operation result failed."
                task_result = json.loads(response)
                LOG.info("Query task result：%s" % task_result)
                task_status = task_result.get("retcode")
                LOG.info("[query_operation_result],status:{},response:{}".format('200', response))
                if task_status == 1:
                    detail = task_result.get("detail")
                    if detail:
                        detail = detail.get("en_US")
                    detail += error_tips
                    DROperationService.set_task_error(detail, e_task_path)
                    return False, "Query result failed."

                Taskmgrutil.set_e_taskstatus(e_task_path, "running")
                Taskmgrutil.set_e_taskprogress(e_task_path, "%s" % task_progress)

                task_progress = int(task_result.get("progress"))
                if task_progress == 100:
                    detail = task_result.get("detail")
                    if detail:
                        detail = detail.get("en_US")
                    if task_status == 0 and operation != "add":
                        # 任务状态为成功(删除和倒换）
                        return True, detail
                    elif task_status == 0 and operation == "add":
                        # 任务状态为成功(关联)
                        Taskmgrutil.set_e_taskstatus(e_task_path, "running")
                        Taskmgrutil.set_e_taskprogress(e_task_path, "99")
                        Taskmgrutil.set_msg_append(e_task_path, "[%s]" % DROperationService.get_date_time() + " " + detail)
                        Taskmgrutil.set_msg_append(e_task_path, "[%s]" % DROperationService.get_date_time() +
                                                   " Waiting for data synchronization of the DR system...")
                        LOG.info("Waiting for data synchronization of DR system...")
                        sync_flag = DROperationService.post_build_dr(e_task_path)
                        if sync_flag:
                            return True, "Data synchronization of the DR system is complete."
                        else:
                            tips = "The DR association is successful, but data replication is not complete. Log in to " \
                                   "the management plane to confirm that the replication is complete, and then perform " \
                                   "the next step Switching the primary and secondary sites."
                            LOG.info(
                                "The DR association is successful, but data replication is not complete.")
                            return True, tips
                    # 任务状态为失败
                    detail += error_tips
                    return False, detail
                time.sleep(5)
            return True, ''
        except Exception as e_msg:
            LOG.error("[query_operation_result] Exception:{}".format(e_msg))
            return False, "Query failed."

    @staticmethod
    def query_dr_task_status(operation, products, response, e_task_path):
        """
        功能说明:封装容灾任务下发后查询动作
        :param action
        :param products
        :param response
        :param e_task_path
        :return True、 False
        """
        result, detail = DROperationService.query_operation_result(response, e_task_path, operation)
        if operation  == 'migrate':
            DROperationService.check_status_after_switch(e_task_path, products)
        if result:
            if operation == 'migrate':
                if not DROperationService.wait_dr_migrate_tasks_finish(e_task_path):
                    return False
            DROperationService.set_task_success(detail, e_task_path)
        else:
            DROperationService.set_task_error(detail, e_task_path)
        return result

    @staticmethod
    def wait_dr_migrate_tasks_finish(e_task_path):
        # 监控容灾倒换关联任务是否完成
        timeout = 3600
        interval = 10
        max_time = int(time.monotonic()) + timeout
        dr_migrate_task_url = "/rest/drmgrservice/v1/main/drmgr/connection/running/products"
        while int(time.monotonic()) < max_time:
            status, response = DROperationService.uniep_get_request(dr_migrate_task_url)
            if not status:
                tips = "Failed to query dr migrate tasks."
                DROperationService.set_task_error(tips, e_task_path)
                return False
            dr_task_list = json.loads(response)
            if not dr_task_list:
                return True
            tips = "Dr migrate tasks are running"
            DROperationService.set_task_msg(tips, e_task_path)
            time.sleep(interval)
        tips = "Query dr migrate tasks time out"
        DROperationService.set_task_error(tips, e_task_path)
        return False

    @staticmethod
    def set_task_error(detail, e_task_path):
        Taskmgrutil.set_e_taskstatus(e_task_path, "error")
        Taskmgrutil.set_e_taskprogress(e_task_path, "100")
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + detail)

    @staticmethod
    def set_task_success(detail, e_task_path):
        Taskmgrutil.set_e_taskstatus(e_task_path, "success")
        Taskmgrutil.set_e_taskprogress(e_task_path, "100")
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + detail)

    @staticmethod
    def set_task_msg(detail, e_task_path):
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + detail)
    @staticmethod
    def post_build_dr(e_task_path):
        # 关联后检查数据的同步状态，超时3小时
        time_counter = 0
        product_name = DROperationService.query_dr_product_name()
        while True:
            sync_status = DROperationService.get_rep_status(product_name)
            LOG.info("sync_status is %s" % sync_status)
            if sync_status:
                time.sleep(30)
                time_counter += 1
                confirm_status = DROperationService.get_rep_status(product_name)
                LOG.info("confirm_status is %s" % sync_status)
                if confirm_status:
                    return True  # 连续两次查到“同步中”、“已同步”的状态，认为数据同步完成
            # 关联步骤超过3小时并且至少查过一次数据复制状态未同步完成,则不再查询,提示"关联成功但数据同步未完成"
            if time.time() >= DROperationService.add_dr_start_time + ADD_DR_COST_TIME:
                break
            # 同步未完成，则每隔5分钟打印一个提示日志
            if time_counter >= 10:
                DROperationService.set_task_msg("Waiting for data synchronization of the DR system...", e_task_path)
                time_counter -= 10
            time.sleep(30)
            time_counter += 1
        return False  # 超过3小时未同步，认为数据同步异常

    @staticmethod
    def operate_delete_dr(products_name):
        """
        功能描述：异地容灾主备站点分离
        参数：产品名称列表
        返回： response
        修改记录：新增方法
        """
        # 60秒检查，检查OK直接下发，检查不OK持续60秒后往下执行
        DROperationService.check_omp_task_status(60)

        url = "/rest/plat/drmgrservice/v1/main/operation?action=deletedr"
        params = {"productName": products_name}
        status, response = httpclient.IRHttpClient().post(url, params)
        if status != 200:
            LOG.warning(
                "[operate_delete_dr] Failed to invoke interface. retCode: %s." % str(status))
            return False

        return response

    @staticmethod
    def operate_migrate_dr(products_name):
        """
        功能描述：异地容灾产品倒换
        参数：产品名称列表
        返回： response
        修改记录：新增方法
        """
        # 主备倒换前检查当前节点是否是主用站点
        site = ""
        url = "/rest/plat/drmgrservice/v1/main/sitestatus"
        parameter = {'productname': []}
        for _ in range(5):
            status, response = httpclient.IRHttpClient().post(url, parameter)
            if status == 200:
                site = json.loads(response)[0].get('local')
                break
        if site == "active":
            LOG.warning("[operate_migrate_dr] The DR system switchover task is complete.")
            return "skip_step"

        # 创建倒换任务
        url = "/rest/plat/drmgrservice/v1/main/operation?action=migrateproduct"
        params = {"productName": products_name}
        status, response = httpclient.IRHttpClient().post(url, params)
        if status != 200:
            LOG.warning(
                "[operate_migrate_dr] Failed to invoke interface. retCode: %s." % str(status))
            return False

        return response

    @staticmethod
    def operate_build_dr(drinfo):
        """
        功能描述：创建异地容灾
        参数：productname,
        返回： progress,status,msg
        修改记录：新增方法
        """
        url = "/rest/plat/drmgrservice/v1/main/operation?action=builddr"
        params = drinfo
        status, response = httpclient.IRHttpClient().post(url, params)
        if status != 200:
            LOG.warning(
                "[operate_build_dr] Failed to invoke interface. retCode: %s." % str(status))
            return False

        return response

    @staticmethod
    def get_rep_status(product_name):
        """
        功能描述：查询产品的数据同步状态
        参数：产品名称列表
        返回：复制状态为“已同步”或“同步中”则为True，其余为False
        """
        url = "/rest/plat/drmgrservice/v1/main/repstatus"
        client = httpclient.IRHttpClient()
        params = {"productName": product_name}
        response = client.post(url, params)
        if response[0] != 200:
            LOG.warning(
                '[get_rep_status]Failed to invoke the repstatus interface. retCode: %s.' % str(
                    response[0]))
            return False
        # 根据状态显示对应的具体状态
        result_list = json.loads(response[1])
        for result in result_list:
            rep_msg = result["repstatus"]
            if "Abnormal" in str(rep_msg) or "Delay" in str(rep_msg) or "Repairing" in str(rep_msg) \
                    or "Switching" in str(rep_msg) or "FullSyncing" in str(rep_msg):
                LOG.info("[get_rep_status] The data synchronization of the DR system is %s" % rep_msg)
                return False
        return True

    @staticmethod
    def uniep_get_request(rest_url):
        """
        功能说明:发送get请求
        :param rest_url
        :return query_progress, query_status
        """
        local_uniep_task_mgr = Unieptaskmgr()
        return local_uniep_task_mgr.send_get_request(rest_url)

    @staticmethod
    def get_product_status(product_name):
        """
        功能说明:检查产品整体运行状态
        :param product_name
        :return query_progress, query_status
        """
        default_progress = 50
        query_progress = default_progress
        query_status = 'NA'
        __query_url = '/rest/plat/omp/v1/main/monitor/products/%s/status' % product_name
        for _ in range(0, 3):
            # 连续下发三次, 三次之后等待3秒
            result, response = DROperationService.uniep_get_request(__query_url)
            if result and response:
                product_status = json.loads(response)
                query_progress = product_status.get('progress', default_progress)
                query_status = product_status.get('status', 'NA')
        return query_progress, query_status

    @staticmethod
    def check_service_status(product_name, time_out=60):
        """
        功能说明:检查产品服务状态
        :param product_name
        :param time_out
        :return True、False
        """
        is_running = False
        product_status = 'UNKNOWN'
        for _ in range(0, time_out):
            progress, product_status = DROperationService.get_product_status(product_name)
            log_msg = "check_service_status.product_name:%s " \
                      "progress:%s product_status:%s" % (product_name, progress, product_status)
            if progress == 100 and product_status == 'RUNNING':
                is_running = True
                LOG.info("END:%s" % log_msg)
                break
            LOG.info("QUERY:%s" % log_msg)
            time.sleep(1)
        if not is_running:
            LOG.info("END.is_running:%s" % is_running)
            return False, product_status
        return True, product_status

    @staticmethod
    def check_products_service(products):
        """
        功能说明:检查所有产品服务状态
        :param products
        :return True、False
        """
        is_all_running = True
        not_running_products = []
        for one_product in products:
            __query_url = '/rest/plat/omp/v1/main/monitor/products/%s/status' % one_product
            result, status = DROperationService.check_service_status(one_product, 300)
            if not result:
                LOG.info("check_service_status:product:%s is not running.url:%s" % (one_product,
                                                                                    __query_url))
                is_all_running = False
                not_running_products.append({'product_name': one_product, 'product_status': status})
            LOG.info("check_service_status:product:%s is running.url:%s" % (one_product,
                                                                            __query_url))
        return is_all_running, not_running_products

    @staticmethod
    def get_date_time():
        """
        功能说明:获取当前时间
        :param 无
        :return ‘2020-12-10 10:14:01’
        """
        return datetime.strptime(datetime.now().strftime('%Y-%m-%d %H:%M:%S'), '%Y-%m-%d %H:%M:%S')

    @staticmethod
    def check_status_after_switch(task_path, products):
        """
        功能说明:倒换后检查产品状态
        :param task_path
        :param products
        :return True、False
        """
        # 检查产品服务运行状态
        if not products:
            return
        start_check_msg = "Start to check product service status"
        end_check_msg = "Finish to check product service status"
        LOG.info(start_check_msg)
        result, products_info = DROperationService.check_products_service(products)
        if not result:
            products_info_msg = "The service is not running.porducts: %s" % products_info
            LOG.info(products_info_msg)
        LOG.info(end_check_msg)

    @staticmethod
    def check_task_is_running(tasks_list):
        """
        功能说明:过滤任务列表，检查是否有运行中任务
        :params tasks_list
        :return True、False
        """
        for one_task in tasks_list:
            currentState = one_task.get('currentState')
            isEnabled = one_task.get('isEnabled')
            isScheduleJob = one_task.get('isScheduleJob')
            if currentState in ['RUNNING', 'TERMINATING']:
                LOG.warning("task (%s) is running" % one_task)
                return True
            if currentState == "WAITING" and ((isScheduleJob == "yes" and isEnabled == "yes") or isScheduleJob == "no"):
                LOG.warning("task (%s) is running" % one_task)
                return True
        return False

    @staticmethod
    def check_omp_task_status(time_out=60):
        """
        功能说明:检查管理面任务列表是否有正在运行的任务
        :params time_out
        :return True、False True:无其他任务运行 False:有其他任务运行
        """
        is_no_task_running = False
        LOG.info("Start to check omp task status")
        for _ in range(0, time_out):
            query_omp_task_url = '/rest/plat/omp/v1/main/common/taskmgr/tasks'
            result, response = DROperationService.uniep_get_request(query_omp_task_url)
            if not result or not response:
                LOG.warning("url:%s result:%s response:%s" % (query_omp_task_url,
                                                              result, response))
                time.sleep(1)
                continue
            # 检查是否已有任务正在运行
            omp_tasks = json.loads(response)
            if DROperationService.check_task_is_running(omp_tasks):
                time.sleep(1)
            else:
                is_no_task_running = True
                break
        LOG.info("Finished to check omp task status")
        return is_no_task_running

    @staticmethod
    def start_products(e_task_path, products):
        """
        功能说明:删除容灾关系后，启动备站点服务
        :param e_task_path 任务目录
        :param products 产品列表,支持多产品
        :return True、False True:产品启动成功 False:产品启动失败
        """
        query_nodes_url = '/rest/productconfiguration/v1/nodes'
        response_list = []
        for _ in range(0, 60):
            status, response = DROperationService.uniep_get_request(query_nodes_url)
            if status:
                response_list = json.loads(response)
                break
            time.sleep(1)
        products_node = {}
        for product in products:
            for one_node in response_list:
                if one_node.get("productName") == product:
                    products_node.update({product: one_node.get("manageIP")})
                    break
        if not products_node:
            DROperationService.set_task_error("Failed to query product ip.", e_task_path)
            return False
        for product, ip in products_node.items():
            start_time = DROperationService.get_date_time()
            Taskmgrutil.set_msg_append(e_task_path,
                                       "[%s]" % start_time + " " + "Start to start %s." % product)
            ssh_cmd = ". /opt/oss/manager/bin/engr_profile.sh;" \
                      "ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o " \
                      "ConnectionAttempts=3 -o ServerAliveInterval=10 %s" % ip
            run_cmd = "%s bash ${OSS_ROOT}/agent/bin/ipmc_adm -cmd startnodes -tenant %s -type app" % (ssh_cmd, product)
            LOG.info("Run cmd:%s" % run_cmd)
            exitcode, output = subprocess.getstatusoutput(run_cmd)
            if exitcode != 0:
                DROperationService.set_task_error("Failed to start %s." % product, e_task_path)
                LOG.error("Failed to start %s. Msg:%s" % (product, output))
                return False
            start_time = DROperationService.get_date_time()
            Taskmgrutil.set_msg_append(e_task_path,
                                       "[%s]" % start_time + " " + "Finished to start %s." % product)
        Taskmgrutil.set_e_taskstatus(e_task_path, "success")
        Taskmgrutil.set_e_taskprogress(e_task_path, "100")
        return True

    @staticmethod
    def delete_dr(e_task_path, product_name):
        """
        删除容灾关系
        :param e_task_path:任务路径
        :param product_name: 产品名称
        :return:
        """
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + "Start to separate the existing remote DR relationship.")
        response = DROperationService.operate_delete_dr(product_name)
        if not response:
            Taskmgrutil.set_msg_append(e_task_path,
                                       "[%s]" % DROperationService.get_date_time() + " " + "Failed to deliver the task.")
            return False
        try:
            result, detail = DROperationService.query_operation_result(response, e_task_path, "delete")
        except Exception as e_msg:
            LOG.error("[query_operation_result] Exception:{}".format(e_msg))
            return False
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + detail)
        return result

    @staticmethod
    def check_dr_sitestatus(e_task_path):
        """
        查询容灾关系是否完全解除
        :param e_task_path:任务路径
        :return: True 完全解除 False未完全解除
        """
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + "Querying the DR relationship.")
        url = "/rest/plat/drmgrservice/v1/main/sitestatus"
        parameter = {'productname': []}
        for _ in range(5):
            status, response = httpclient.IRHttpClient().post(url, parameter)
            if status == 200 and not json.loads(response):
                Taskmgrutil.set_msg_append(e_task_path,
                                           "[%s]" % DROperationService.get_date_time() + " " + "The DR relationship has been completely deleted.")
                return True
        Taskmgrutil.set_msg_append(e_task_path,
                                   "[%s]" % DROperationService.get_date_time() + " " + "The DR relationship is not completely deleted.")
        return False

    @staticmethod
    def delete_dr_pro(e_task_path, action_type, product_name=None, set_task=True):
        """
        删除容灾，第一次删除成功后检查容灾是否完全删除，没有重新删除容灾一次
        :param e_task_path:任务路径
        :param action_type: 是否是回滚时解除容灾，回滚时解除容灾需要拉起产品服务
        :param product_name:产品列表，不传入就直接查询
        :param set_task:退出时是否刷新Easysuite状态
        :return:
        """
        if not product_name:
            products = DROperationService.query_dr_product_name()
        else:
            products = product_name
        if not products and DROperationService.query_dr_data() is False:
            DROperationService.set_task_success(
                "Remote DR system has been deleted before this step.", e_task_path)
            return True
        if not DROperationService.get_rep_status(product_name):
            DROperationService.set_task_error("The data synchronization status of the DR system is abnormal. "
                                              "Rectify the data synchronization fault of the DR system and try again",
                                              e_task_path)
            return False
        result = DROperationService.delete_dr(e_task_path, products)
        if not result:
            Taskmgrutil.set_e_taskstatus(e_task_path, "error")
            Taskmgrutil.set_e_taskprogress(e_task_path, "100")
            return False
        if not DROperationService.check_dr_sitestatus(e_task_path):
            result = DROperationService.delete_dr(e_task_path, products)
            if not result:
                Taskmgrutil.set_e_taskstatus(e_task_path, "error")
                Taskmgrutil.set_e_taskprogress(e_task_path, "100")
                return False
        if not set_task:
            return True
        if action_type != "rollback":
            Taskmgrutil.set_e_taskstatus(e_task_path, "success")
            Taskmgrutil.set_e_taskprogress(e_task_path, "100")
            return True
        return DROperationService.start_products(e_task_path, products)

    @staticmethod
    def add_dr_operation_service(e_task_path, dr_info_file,
                                 action_type, dr_consistent, product_name, rep_direction):
        """
        功能说明:创建异地容灾系统入口
        :param e_task_path:
        :param dr_info_file:
        :param action_type:
        :param dr_consistent:
        :param product_name:
        :param rep_direction:
        :return:
        """
        DROperationService.add_dr_start_time = time.time()
        tip_detail = '[Description] The association task between the primary and secondary sites ' \
                     'includes DR creation and first full synchronization. The task duration ' \
                     'depends on the management scale, number of equivalent NEs, and network ' \
                     'bandwidth. The maximum duration of an association task between the active ' \
                     'and standby nodes is 2 hours. The estimated task execution time is the ' \
                     'average value. The actual time is subject to the actual running.'
        Taskmgrutil.set_msg_append(e_task_path, tip_detail)

        # 判断drinfo.json文件是否存在
        if not os.path.isfile(dr_info_file):
            missing_drinfo_tip = "Failed to associate the primary and secondary sites because the DR configuration" \
                                 " does not exist. Possible cause: The primary and secondary sites are not separated" \
                                 "using EasySuite. Solution: Manually configure the DR system on the HA page of the" \
                                 " NCE management plane."
            Taskmgrutil.set_msg_append(e_task_path, missing_drinfo_tip)
            Taskmgrutil.set_e_taskstatus(e_task_path, "fail")
            Taskmgrutil.set_e_taskprogress(e_task_path, "100")
            return False

        with open(dr_info_file, 'r') as info_dr:
            dr_info = json.load(info_dr)
        Taskmgrutil.set_e_taskstatus(e_task_path, "running")
        Taskmgrutil.set_e_taskprogress(e_task_path, "0")
        if len(product_name) > 0 or DROperationService.query_dr_data():
            Taskmgrutil.set_msg_append(e_task_path,
                                       "The remote DR - Product relationship already exists.")
            del_result = DROperationService.delete_dr_pro(e_task_path, action_type, product_name,
                                                          False)
            if del_result is False:
                return False
        if action_type == 'rollback':
            rep_direction = 'PRIMARY_TO_SECONDARY'
        all_rep_direction = []
        for _ in dr_info.get('drinfo').get('repDirection'):
            all_rep_direction.append(get_rep_direction(rep_direction, dr_consistent))
        dr_info.get('drinfo').update({'repDirection': all_rep_direction})
        response = DROperationService.operate_build_dr(dr_info)
        if response is False:
            return False
        result, detail = DROperationService.query_operation_result(response, e_task_path, 'add')
        set_task_data(e_task_path, msg=detail)
        if not result:
            return False
        edit_health_time(os.path.join(os.path.dirname(dr_info_file), 'dr_health_time.json'))
        return True

    @staticmethod
    def migrate_dr_operation_service(e_task_path, product_name):
        """
        功能说明:异地容灾倒换模块入口
        :param e_task_path:
        :param product_name:
        :return:
        """
        if not product_name:
            set_task_data(e_task_path, "error", "100", "The DR product name list is empty.")
        # 查询当前容灾的数据同步状态
        start_time = time.time()
        cost_time = 0
        flag = False
        LOG.info("[Migrate] Start to query data_rep_status.")
        while cost_time <= 180:
            flag = DROperationService.get_rep_status(product_name)
            if flag:
                break
            time.sleep(1)
            cost_time = time.time() - start_time
        LOG.info("[Migrate] data_rep_status flag is %s" % flag)
        # 查询180s若数据同步状态一直异常则报错并退出
        if not flag:
            set_task_data(e_task_path, "error", "100", "Abnormal data synchronization status.")
            return False
        # 数据同步状态正常则调倒换接口
        response = DROperationService.operate_migrate_dr(product_name)
        if not response:
            set_task_data(e_task_path, "error", "100", "Failed to switchover.Possible cause:"
                                                       "EasySuite does not support the "
                                                       "switchover of NCE of the this version.")
            return False
        if response == "skip_step":
            detail = "The DR system switchover task is complete."

            DROperationService.set_task_success(detail, e_task_path)
            return True
        elif not DROperationService.query_dr_task_status("migrate", product_name, response, e_task_path):
            return False
        return True


def main(argv):
    """
    容灾主函数
    :param argv:必要入参
    :return:
    """
    action_type = ''
    dr_consistent = 'true'
    rep_direction = "SECONDARY_TO_PRIMARY"
    operation = argv[1]
    # 创建容灾add,script_id,action_type(upgrade/rollback),dr_consistent(true/false),path,rep_direction
    if len(argv) == 7:
        action_type = argv[3]
        dr_consistent = argv[4]
        dr_info_file = argv[5]
        rep_direction = argv[6]
    # 删除容灾delete,script_id,action_type(upgrade/rollback)
    if len(argv) == 4:
        action_type = argv[3]
    # 倒换或更新容灾信息migrate,script_id/update,path
    if len(argv) == 3 and operation == "update":
        dr_info_file = argv[2]
        result = DROperationService.update_dr_info_file(dr_info_file)
        return result

    e_task_id = argv[2]
    e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
    e_task_path = os.path.join(e_taskmgr_path, e_task_id)

    # 初始化方法
    taskmgr_function = Taskmgrutil()
    # 初始化任务
    taskmgr_function.init_e_taskmgr(e_task_path)
    # 查询容灾产品配置数据
    product_name = DROperationService.query_dr_product_name()

    if operation == "delete":
        return DROperationService.delete_dr_pro(e_task_path, action_type)
    if operation == "migrate":
        return DROperationService.migrate_dr_operation_service(e_task_path, product_name)
    elif operation == "add":
        return create_dr_site(e_task_path, dr_info_file, action_type, dr_consistent, product_name, rep_direction)
    else:
        return False



if __name__ == '__main__':
    main(sys.argv)
