# encoding=utf-8
"""
功 能：导入EasyOps巡检补丁包
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2022-09-19 23:17 创建
"""
import os
import re
import sys
import json
import time
import shutil
from io import BytesIO
from rest import restclient
from util import httpclient, httputils, common
from commonlog import Logger

LOG = Logger().getinstance(sys.argv[0])
WAIT_TIME = 2
MAX_TIMES = 60
BOUNDARY = "----WebKitFormBoundary5nhMJDLgOtFdBJCq"
UPLOAD_OPS_URL = "/rest/smpmanagerwebsite/v1/healthcheck/uploadhealthcheckzip"
QUERY_UPLOAD_URL = "/rest/smpmanagerservice/v1/healthcheck/getimportprogress?taskId={}"
REPEAT_IMPORT_URL = "/rest/smpmanagerwebsite/v1/healthcheck/reimportservice"
QUERY_PRODUCT_URL = "/rest/smpmanagerservice/v1/project/query-product"
POST = httputils.HTTP_METHOD_POST
GET = httputils.HTTP_METHOD_GET
# x-user-role 管理面外部接口用户权限鉴权, 1为Administrator管理员用户ID,拥有最高权限
HEADERS = {'Accept-Encoding': 'gzip, deflate, br', 'Accept': '*/*',
           'Content-Encoding': 'gzip', 'x-user-roles': '1',
           'Content-Type': 'multipart/form-data; boundary={}'.format(BOUNDARY)}


def get_request_data(file_path, product_id):
    """
    功能说明:请求文件数据流
    :param file_path:
    :param product_id:
    """
    with open(file_path, 'rb') as fb:
        file = fb.read()
    # 拼接请求body
    body = BytesIO()
    body.write(("--%s\r\n" % BOUNDARY).encode("latin-1"))
    body.write(
        (f'Content-Disposition: form-data; name="file"; '
         f'filename="{os.path.basename(file_path)}"\r\n').encode("latin-1"))
    body.write('Content-Type: zip\r\n'.encode("latin-1"))
    body.write(b"\r\n")
    body.write(file)
    body.write(b"\r\n")
    if product_id:
        body.write(f"--{BOUNDARY}\r\n".encode("latin-1"))
        body.write('Content-Disposition: form-data; name="projectId"\r\n'.encode("latin-1"))
        body.write(b"\r\n")
        body.write(f"{product_id}\r\n".encode("latin-1"))
    body.write(f"--{BOUNDARY}--\r\n".encode("latin-1"))
    return body.getvalue()


def ir_request(method, url, data):
    """
    发送IR请求
    :param method: 请求类型
    :param url: 请求地址
    :param data: 请求数据
    """
    ip_func = common.getMgmtIRIPs if hasattr(common, 'getMgmtIRIPs') \
        else common.get_mgmt_ir_ips
    port_func = common.getMgmtIRPort if hasattr(common, 'getMgmtIRPort') \
        else common.get_mgmt_ir_port
    ips = ip_func()
    port = port_func()
    LOG.info(f"MgmtIRIPs : {ips} MgmtIRPort : {port}")
    session = [restclient.HttpsClient(ip, port, False, headers=HEADERS) for ip in ips]
    status, response = (600, '')
    for one_session in session:
        one_session.connect()
        # _invoke暂无替代方法
        status, response = one_session._invoke(method, url, data, HEADERS)
        LOG.info(f"status:{status} response:{response}")
        LOG.info("(url:%s)' [status:%s, response:%s]" % (url, status, response))
        if status != 200:
            break
    if isinstance(response, bytes):
        response = response.decode()
    return status, response


def query_import_progress(task_id):
    """
    功能说明:查询包导入进度
    :param task_id:任务ID
    """
    progress = 0
    for _ in range(MAX_TIMES):
        time.sleep(WAIT_TIME)
        status, response = httpclient.IRHttpClient().get(QUERY_UPLOAD_URL.format(task_id))
        LOG.info(f"status:{status} response:{response}")
        if status != 200 or not response:
            continue
        if isinstance(response, bytes):
            response = response.decode()
            progress = int(json.loads(response).get("total", "0"))
        if progress == 100:
            break
    else:
        # 轮询结束返回失败
        return False
    return True


def repeated_import(response, product_id):
    """
    功能说明:重复导入
    :param response: 响应
    :param product_id:产品ID
    """
    task_id = get_task_id(response)
    data = {"dirName": get_task_time_stamp(response), "projectId": product_id,
            "repeatedSceneDir": [], "status": True, "taskId": task_id,
            "uploadType": get_upload_type(response)}
    local_headers = HEADERS.copy()
    local_headers.update({'Content-Type': 'application/json;charset=UTF-8'})
    status, response = httpclient.IRHttpClient(headers=local_headers).post(REPEAT_IMPORT_URL, data)
    LOG.info(f"status:{status} response:{response}")
    if status != 200 or not response:
        return False
    if isinstance(response, bytes) and json.loads(response).get('result') == '0':
        return query_import_progress(task_id)
    return False


def import_result(status, response):
    """
    功能说明:判断是否导入成功
    :param status: 返回码
    :param response: 响应
    :return 'error', 'finish', 'repeat_import'
    """
    # 请求失败
    if status != 200 or not response:
        return "error"
    # 请求成功
    if response.lower().startswith("uploadcompleted"):
        return "finish"
    # 请求成功,重复导入
    if response.lower().startswith("uploadrepeatservice"):
        return "repeat_import"
    return "error"


def get_task_id(response):
    """
    功能说明:获取任务ID
    :param response: 响应
    """
    # 导入成功response为两段式,重复导入为3段式
    if len(response.split('_')) == 2:
        return response.split('_')[1]
    elif len(response.split('_')) > 2:
        return response.split('_')[2]
    return ''


def get_task_time_stamp(response):
    """
    功能说明:获取任务时间戳
    :param response: 响应
    """
    if len(response.split('_')) > 1:
        return response.split('_')[1]
    return ''


def get_upload_type(response):
    """
    功能说明:获取上传类型
    :param response: 响应
    """
    if len(response.split('_')) > 3:
        return response.split('_')[3]
    return ''


def upload_zip(file_name, product_id):
    """
    功能说明:导入healthcheck包
    :param file_name: healthcheck包全路径
    :param product_id: 产品ID
    """
    result = False
    data = get_request_data(file_name, product_id)
    try:
        # 调用上传文件的IR接口
        status, response = ir_request(POST, UPLOAD_OPS_URL, data)
        result = import_result(status, response)
        if result == "finish":
            result = query_import_progress(get_task_id(response))
        elif result == "repeat_import":
            result = repeated_import(response, product_id)
        else:
            result = False
    except Exception as e:
        LOG.error("Failed to upload the zip_file,due to Exception:%s" % e)
        return result
    if not result:
        LOG.info(f"Failed to import easy_ops.result:{result}")
    return result


def unzip_easy_ops(pkg_path, unzip_path):
    """
    功能说明:解压easy_ops补丁包
    :param pkg_path:包路径
    :param unzip_path: 解压目录
    :return: True, False
    """
    from taskmgr_util import Taskmgrutil
    cmd = f"unzip -oq {pkg_path} -d {unzip_path}"
    ret_code, out_put = Taskmgrutil.execute_cmd(cmd)
    if ret_code == 0:
        return True
    LOG.info(f"Failed to unzip easy_ops.result:{ret_code, out_put}")
    return False


def get_product_ids():
    """
    功能说明:获取产品ID
    :return: product_ids:环境各个产品ID
    """
    product_ids = {}
    status, response = httpclient.IRHttpClient().get(QUERY_PRODUCT_URL)
    if status == 200:
        product_ids = {x.get('productName'): x.get('productId') for x in json.loads(response)}
    return product_ids


def get_product_desc():
    """
    功能说明:获取产品描述
    :return:
    """
    product_desc = {}
    status, response = httpclient.IRHttpClient().get('/rest/plat/omp/v1/main/resmgr/products')
    if status == 200:
        product_desc = {x.get('productName'): x.get('productDesc').split('(')[0].strip()
                        for x in json.loads(response)}
    return product_desc


def initial_path(path):
    """
    功能说明：初始化指定路径
    :param path: 目录全路径
    :return:
    """
    if os.path.exists(path):
        shutil.rmtree(path)
    os.makedirs(path)


def get_real_path(*args):
    """
    功能说明:获取拼接的真实路径
    :param args: 目录、文件可选参数
    :return:
    """
    return os.path.realpath(os.path.join(*args))


def get_deploy_ops_pkg(filter_path, filter_pattern=r'^healthcheck.*\.zip$'):
    """
    功能说明:获取EasyOps待导入包清单
    :param filter_path: 软件包过滤路径
    :param filter_pattern: 过滤正则表达式
    :return:
    """
    valid_zip_pkg = []
    for root_dir, child_dir, file in os.walk(filter_path):
        valid_zip_pkg.extend([get_real_path(root_dir, x) for x in file
                              if re.fullmatch(filter_pattern, x)])
    # 产品描述匹配的直接导入
    deploy_ops_pkg = [x for x in valid_zip_pkg for y in get_product_desc().values()
                      if x.startswith(get_real_path(filter_path, y))]

    # 跨域场景特殊判断正则表达式；R22C10及之后版本提供子域配置能力由子域自行判断；R22C00版本采用工具硬编码逻辑进行支持
    cross_domain_pattern = r'.+IP\+T.*|.+IP\+FAN.*|.+T\+FAN.*'
    cross_domain_path = 'NCE-CrossDomain'
    deploy_ops_pkg.extend([x for x in valid_zip_pkg for y in get_product_desc().values()
                           if re.match(cross_domain_pattern, y)
                           and x.startswith(get_real_path(filter_path, cross_domain_path))])
    # 最外层目录下文件满足条件直接导入
    deploy_ops_pkg.extend([x for x in valid_zip_pkg if os.path.dirname(x) == filter_path])
    LOG.info(f"get_deploy_ops_pkg.deploy_ops_pkg:{deploy_ops_pkg}")
    return deploy_ops_pkg


def import_easy_ops(argv):
    """
    功能说明:总入口
    :param argv:脚本入参
    """
    pkg_path = argv[1]
    easy_ops_pkg = argv[2]

    easy_ops_path = get_real_path(pkg_path, easy_ops_pkg)
    unzip_pkg_path = get_real_path(pkg_path, 'easy_ops_temp')
    # 初始化解压目录
    initial_path(unzip_pkg_path)
    # 解压EasyOps包
    unzip_easy_ops(easy_ops_path, unzip_pkg_path)

    # 无待部署Ops包，退出
    pre_deploy_pkgs = get_deploy_ops_pkg(unzip_pkg_path)
    if not pre_deploy_pkgs:
        # 退出，清空解压目录
        shutil.rmtree(unzip_pkg_path)
        return False

    product_ids = get_product_ids()
    results = []
    if not product_ids:
        results = [upload_zip(x, "") for x in pre_deploy_pkgs]

    for product_name, one_id in product_ids.items():
        # 管理面ID特殊需要特殊处理
        if product_name == "manager":
            one_id = "default"
        results = [upload_zip(x, one_id) for x in pre_deploy_pkgs]
    # 退出，清空解压目录
    shutil.rmtree(unzip_pkg_path)
    return False if False in results else True


if __name__ == '__main__':
    import_easy_ops(sys.argv)