#!/bin/bash

RET_ERR=1
RET_OK=0
declare -i retVal=0

function precheck_nsswitch_conf()
{
    local file_path="/etc/nsswitch.conf"
    local nss_key_string="passwd:,group:,services:,netgroup:,aliases:"
    local nss_key_list=($(echo "${nss_key_string}" | awk -F',' '{for(i=1; i<=NF; i++){print $i;};}'))
    for nss_key in "${nss_key_list[@]}"
    do
        nss_line=$(grep -w "${nss_key}" "${file_path}" | grep -v "#")
        if [[ -z "${nss_line}" ]];then
            fn_sys_log_error "${host_ip}: The configuration item "${nss_key}" cannot be found in /etc/nsswitch.conf. \
        Please update /etc/nsswitch.conf and ensure that the configuration items are correct."
            return $RET_ERR
        fi
        ((index++))
    done
    
    return $RET_OK
}

getPamUnixKey()
{
    local pam_unix2_key="pam_unix2.so"
    
    if [[ -f "/etc/SuSE-release" ]]; then
        VERSION=$(cat /etc/SuSE-release | grep "VERSION" | awk -F '=' '{print $2}')
        if [[ "${VERSION}" -eq "12" ]]; then
            pam_unix2_key="pam_unix.so"
        fi
    fi
    
    echo "${pam_unix2_key}"
}


function precheck_suse_ldap_pamd_conf()
{
    local pam_common_session="/etc/pam.d/common-session"
    local pam_common_password="/etc/pam.d/common-password"
    local pam_common_auth="/etc/pam.d/common-auth"
    local pam_common_account="/etc/pam.d/common-account"
    
    local pam_unix2_line=""
    local pam_unix2_key="$(getPamUnixKey)"
    
    pam_unix2_line=$(grep -w "${pam_unix2_key}" "${pam_common_session}" | grep -v "#" )
    if [[ -z "${pam_unix2_line}" ]];then
        fn_sys_log_warn "${host_ip}: The configuration item pam_unix2_key cannot be found in /etc/pam.d/common-session. \
    Please update /etc/pam.d/common-session and ensure that the configuration item is correct."            
    fi 
    
    local pam_limits_line=""
    local pam_limits_key="pam_limits.so"
    
    pam_limits_line=$(grep -w "${pam_limits_key}" "${pam_common_session}" | grep -v "#" )
    if [[ -z "${pam_limits_line}" ]];then
        fn_sys_log_warn "${host_ip}: The configuration item pam_limits_key cannot be found in /etc/pam.d/common-session. \
    Please update /etc/pam.d/common-session and ensure that the configuration item is correct."     
    fi
    
    pam_unix2_line=$(grep -w "${pam_unix2_key}" "${pam_common_password}" | grep -v "#" )
    if [[ -z "${pam_unix2_line}" ]];then
        fn_sys_log_warn "${host_ip}: The configuration item pam_unix2_key cannot be found in /etc/pam.d/common-password. \
    Please update /etc/pam.d/common-password and ensure that the configuration item is correct."           
    fi  
    
    pam_unix2_line=$(grep -w "${pam_unix2_key}" "${pam_common_auth}" | grep -v "#" )
    if [[ -z "${pam_unix2_line}" ]];then
        fn_sys_log_warn "${host_ip}: The configuration item pam_unix2_key cannot be found in /etc/pam.d/common-auth. \
    Please update /etc/pam.d/common-auth and ensure that the configuration item is correct."    
    fi  
    
    pam_unix2_line=$(grep -w "${pam_unix2_key}" "${pam_common_account}" | grep -v "#" )
    if [[ -z "${pam_unix2_line}" ]];then
        fn_sys_log_warn "${host_ip}: The configuration item pam_unix2_key cannot be found in /etc/pam.d/common-account. \
    Please update /etc/pam.d/common-account and ensure that the configuration item is correct."        
    fi    
}

function precheck_suse_ldap_nscd_conf()
{
    local sysnscd_path="/etc/nscd.conf"
    if [[ ! -f "$sysnscd_path" ]];then
        fn_sys_log_warn "${host_ip}: The /etc/nscd.conf configuration file cannot be found. \
    Please restore the file and ensure that the file is correct."         
    fi    
    
    local sysnscd__bin_path="/usr/sbin/nscd"
    if [[ ! -f "$sysnscd__bin_path" ]];then
        fn_sys_log_warn "${host_ip}:   The /usr/sbin/nscd configuration file cannot be found. \
    Please restore the file and ensure that the file is correct."        
    fi
}

function precheck_suse_ldap_env()
{
    local sysldap_file_env="/usr/lib/openldap/slapd"
    
    if [[ ! -f "$sysldap_file_env" ]];then
        fn_sys_log_warn "${host_ip}: The /usr/lib/openldap/slapd configuration file cannot be found. \
    Please restore the file and ensure that the file is correct."      
    fi    
}

function precheck_redhat_ldap_env()
{
    local sysldap_file_env="/usr/sbin/slapd"
    
    if [[ ! -f "$sysldap_file_env" ]];then
         fn_sys_log_warn "${host_ip}: The /usr/sbin/slapd configuration file cannot be found. \
     Please restore the file and ensure that the file is correct."     
    fi
    
    return $RET_OK
}

function precheck_ldapclient_env()
{
    local sysldap_client_env="/usr/bin/ldapsearch"
    if [[ ! -f "$sysldap_client_env" ]];then
        fn_sys_log_warn "${host_ip}: The /usr/bin/ldapsearch configuration file cannot be found. \
    Please restore the file and ensure that the file is correct."        
    fi
    
    return $RET_OK
}

function precheck_redhat_ldap_rsyslog_conf()
{
    local sys_ldap_rsyslog_conf="/etc/rsyslog.conf"    
    if [[ ! -f "$sys_ldap_rsyslog_conf" ]];then
        fn_sys_log_warn "${host_ip}: The /etc/rsyslog.conf configuration file cannot be found. \
    Please restore the file and ensure that the file is correct."         
    fi    
    
    local sys_ldap_rsyslog_bin="/sbin/rsyslogd"
    if [[ ! -f "$sys_ldap_rsyslog_bin" ]];then
        fn_sys_log_warn "${host_ip}: The /sbin/rsyslogd configuration file cannot be found. \
    Please restore the file and ensure that the file is correct."        
    fi    
}

precheck_redhat_ldap()
{
    precheck_redhat_ldap_env
        
    precheck_redhat_ldap_rsyslog_conf

}

precheck_suse_ldap()
{
    precheck_suse_ldap_env
    
    precheck_suse_ldap_nscd_conf

    precheck_suse_ldap_pamd_conf    
}

version_eq=1
version_err=3
version_compare()
{
    if [[ $# -ne 2 ]]; then
        fn_sys_log_error "${host_ip}: Version_compare need two parameter"
        return 3
    fi
    old_version=$( echo "$1" | sed -e "s/ //g")
    new_version=$( echo "$2" | sed -e "s/ //g")
    if [[ "${old_version}" != "" ]] && [[ "${new_version}" != "" ]]; then
        old_version_array=( $(echo "${old_version}" | cut -d "." --output-delimiter=" " -f 1-) )
        new_version_array=( $(echo "${new_version}" | cut -d "." --output-delimiter=" " -f 1-) )
        old_version_array_length=${#old_version_array[@]}
        new_version_array_length=${#new_version_array[@]}

        length=$((${new_version_array_length}>${old_version_array_length}?${old_version_array_length}:${new_version_array_length}))
        index=0
        eq_tag=3
        while [[ $index -lt "${length}" ]]
        do
            v1=${old_version_array[$index]}
            v2=${new_version_array[$index]}
            if [[ "${v1}" -lt "${v2}" ]]; then
                return 2
            elif [[ "${v1}" -gt "${v2}" ]] ; then
                return 0
            elif [[ "${v1}" -eq "${v2}" ]] ; then
                eq_tag=1
                let index+=1
                continue
            else
                return 3
            fi
        done
        return ${eq_tag}
    else
        fn_sys_log_error "${host_ip}: Parameters can not be null."
        return 3
    fi
}

version_check()
{
    local re=${version_err}
    $(version_compare "${1}" "${2}")
    re=$?
    if [[ ${re} -eq ${version_err} ]];then
        fn_sys_log_error "${host_ip}: Detect ${3} version failed. \
    Please ensure ${3} has been installed correctly."
        return ${RET_ERR}
    else
        if [[ ${re} -gt ${version_eq} ]];then
        fn_sys_log_error "${host_ip}: The ${3} version is weakly secure, \
    it cannot support TLSv1.3.  You are advised to install the recommended OS.\
        Proceeding with the installation indicates that you accepted the security risks."
        fi
    fi
    return ${RET_OK}
}

precheck_ldap_version()
{
    local system_type="$1"
    local e_ldap_version="2.4.40"
    local slapd_bin="/usr/sbin/slapd"
    if [[ -n "$system_type" ]];then
        slapd_bin="/usr/lib/openldap/slapd"    
    fi
    local ldapversion=$(${slapd_bin} -VV 2>&1 | grep "OpenLDAP: slapd" | tr -cd "[0-9 .]" | awk '{print $1}' | sed -e "s/ //g")
    local ldap="openldap"
    version_check "${ldapversion}" "${e_ldap_version}" "${ldap}"
    return $?
}

precheck_openssl_version()
{
    local e_ssl_version="1.0.1"
    local sslversion=$(openssl version 2>&1 | grep "OpenSSL" | tr -cd "[0-9 .]" | awk '{print $1}' | sed -e "s/ //g")
    local ssl="openssl"
    version_check "${sslversion}" "${e_ssl_version}" "${ssl}"
    return $?
}

check_ldap()
{

    host_ip=$1
    fn_sys_log_info "${host_ip}: start checking ldap."
    
    local system_type=$(cat /proc/version | grep "SUSE")  
    
    if [[ -z "$system_type" ]];then
        precheck_redhat_ldap      
    else
        precheck_suse_ldap        
    fi    
    
    precheck_ldap_version "${system_type}"
    
    precheck_nsswitch_conf    

    precheck_openssl_version
    fn_sys_log_info "${host_ip}: Success checking ldap."
}
