#!/bin/bash
################################################################################
## Copyright:   HUAWEI Tech. Co., Ltd.
## Filename:    log.sh
## Description: Log Records
## Version:     CampusInsight V100R20C10
## Created:     Saturday, 11 28, 2020
################################################################################


################################################################################
# Function: logDef
# Description: 记录到日志文件
# Parameter:
#   input:
#   $1: function name ; $2: Log level ; $3: line num ; $4: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: 该函数是最低层日志函数，不会被外部函数直接调用
#################################################################################
logDef()
{
    # 调用日志打印函数的函数名称
    local funcName="$1"
    shift

    # 打印的日志级别
    local logLevel="$1"
    shift

    # 外部调用日志打印函数时所在的行号
    local lineNO="$1"
    shift

    if [ -d "${g_logPath}" ] ; then
        # 打印时间、日志级别、日志内容、脚本名称、调用日志打印函数的函数、打印时的行号及脚本的进程号
        local logTime="$(date -d today +'%Y-%m-%d %H:%M:%S,%N' | cut -b 1-23)"
        printf "[${logTime}] [ ${logLevel} ] : $* [${g_scriptName}(${funcName}):${lineNO}]\n" \
            >> "${g_logPath}/${g_logFile}" 2>&1
    fi

    return 0
}

################################################################################
# Function: log_error
# Description: 对外部提供的日志打印函数：记录EEROR级别日志到日志文件
# Parameter:
#   input:
#   $1: line num ; $2: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
log_error()
{
    # FUNCNAME是shell的内置环境变量，是一个数组变量，其中包含了整个调用链上所有
    # 的函数名字，通过该变量取出调用该函数的函数名
    logDef "${FUNCNAME[1]}" "ERROR" "$@"
}

################################################################################
# Function: log_info
# Description: 对外部提供的日志打印函数：记录INFO级别日志到日志文件
# Parameter:
#   input:
#   $1: line num ; $2: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
log_info()
{
    # FUNCNAME是shell的内置环境变量，是一个数组变量，其中包含了整个调用链上所有
    # 的函数名字，通过该变量取出调用该函数的函数名
    logDef "${FUNCNAME[1]}" "INFO" "$@"
}


################################################################################
# Function: log_warn
# Description: 对外部提供的日志打印函数：记录WARNING级别日志到日志文件
# Parameter:
#   input:
#   $1: line num ; $2: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
log_warn()
{
    # FUNCNAME是shell的内置环境变量，是一个数组变量，其中包含了整个调用链上所有
    # 的函数名字，通过该变量取出调用该函数的函数名
    logDef "${FUNCNAME[1]}" "WARNING" "$@"
}

################################################################################
# Function: showLog
# Description: 记录日志到文件并显示到屏幕
# Parameter:
#   input:
#   $1: Log level ; $2: line num ; $3: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: 该函数是低层日志函数，不会被外部函数直接调用
################################################################################
showLog()
{
    # 把日志打印到日志文件。FUNCNAME是shell的内置环境变量，是一个数组变量，其中
    # 包含了整个调用链上所有的函数名字，通过该变量取出调用该函数的函数名
    logDef "${FUNCNAME[2]}" "$@"
    local logTime="$(date -d today +'%Y-%m-%d %H:%M:%S,%N' | cut -b 1-23)"
    # 设置所有级别日志都显示在屏幕上
    echo -e "[${logTime}] [ $1 ] : $3 [${g_scriptName}(${funcName}):$2]"
}

################################################################################
# Function: showLog_error
# Description: 对外部提供的日志打印函数：记录ERROR级别日志到文件并显示到屏幕
# Parameter:
#   input:
#   $1: line num ; $2: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
showLog_error()
{
    showLog ERROR "$@"
}

################################################################################
# Function: showLog_info
# Description: 对外部提供的日志打印函数：记录INFO级别日志到文件并显示到屏幕
# Parameter:
#   input:
#   $1: line num ; $2: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
showLog_info()
{
    showLog INFO "$@"
}

################################################################################
# Function: showLog_warn
# Description: 对外部提供的日志打印函数：记录INFO级别日志到文件并显示到屏幕
# Parameter:
#   input:
#   $1: line num ; $2: desc
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
showLog_warn()
{
    showLog WARNING "$@"
}

################################################################################
# Function: syslog
# Description: Important operation must record to syslog
# Parameters :
#   input:
#   $1: component name ; $2: filename ;  $3: message
#   output:
#   N/A
# Return: 0 -- success; not 0 -- failure
# Others: N/A
################################################################################
function syslog()
{
    local component=$1
    local filename=$2
    local message=$3

    which logger >/dev/null 2>&1
    [ "$?" -ne "0" ] && return 0;
    login_user_ip="$(who -m | sed 's/.*(//g;s/)*$//g')"
    execute_user_name="$(whoami)"
    logger -p local0.notice -i "${component};[${filename}];${login_user_ip};${execute_user_name};${message}"

}
