#!/usr/bin/python
# -*- coding: utf-8 -*-
import time
import traceback
import json
import uuid

from getDBConnection import get_zenith_session
from common_tasks.base_task import BaseTask
from common_tasks import const_sql

ONE_PACKAGE = 500


class UpgradeNetconfPara(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradeNetconfPara, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeNetconfPara init product_name is %s" % product_name)
        self.src_db_session = get_zenith_session('ucommonDB', 'ucommonDB', product_name)
        if self.src_db_session is None:
            self.error("ucommonDB is None")
            return
        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('neresdb', 'neresdb', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "t_netconf_para"
        self.src_table_cols = ("devID", "devIP", "userName", "password", "port", "privateKey",
                               "passwordPhrase", "authMode", "loginTimeout", "responseTimeout")
        self.src_table_cols_index = {y:x for x, y in enumerate(self.src_table_cols)}
        self.dst_table = "NetConfPara"
        self.dst_table_cols = ("modelId", "fModelId", "tenantId", "neResId", "neId", "purpose",
                               "connectStatus", "userName", "password", "port", "userPrivateKey",
                               "userPrivateKeyPwd", "authMode", "loginTimeout", "responseTimeout",
                               "createAt", "updateAt")
        self.idmapping_dic = {}

    def get_resid_from_idmapping(self, paras):
        idmappingdb_sess = get_zenith_session('idmappingdb', 'idmappingdb', self.product_name)
        if idmappingdb_sess is None:
            self.error("get idmappingdb_sess session fail")
            return
        idmappingdb_sess.autocommit(True)
        id_mapping_cursor = idmappingdb_sess.cursor()

        tmp_table_name = "tmp_neid_%s" % self.dst_table
        id_mapping_cursor.execute(const_sql.DROP_TEMP_TABLE % tmp_table_name)
        id_mapping_cursor.execute(const_sql.CREATE_TEMP_TABLE % tmp_table_name)
        insert_stmt = "insert into tmp_%s values(:1)" % tmp_table_name

        nativeIds = []
        for para in paras:
            data = []
            nativeId = "NE=%s" % para[self.src_table_cols_index.get("devID")]
            data.append(nativeId)
            tuple_data = tuple(data)
            nativeIds.append(tuple_data)
            if len(nativeIds) == ONE_PACKAGE:
                id_mapping_cursor.executemany(insert_stmt, nativeIds)
                self.debug("one package:%s" % nativeIds)
                nativeIds = []

        if len(nativeIds) != 0:
            id_mapping_cursor.executemany(insert_stmt, nativeIds)
            self.debug("last package datas: %s" % nativeIds)

        query_stmt = const_sql.INNER_JOIN_TEMP_TABLE % tmp_table_name
        self.debug("query sql: %s" % query_stmt)
        id_mapping_cursor.execute(query_stmt)
        result = id_mapping_cursor.fetchall()

        for r in result:
            l = list(r)
            self.idmapping_dic[l[1].lstrip("NE=")] = l[0]
            self.debug("result: %s: %s" % (l[1], l[0]))

        id_mapping_cursor.execute(const_sql.DROP_TEMP_TABLE % tmp_table_name)
        id_mapping_cursor.close()
        idmappingdb_sess.close()

    def convert_data(self, paras):
        modelId = str(uuid.uuid1())
        fModelId = None
        tenantId = "default-organization-id"
        neResId = self.idmapping_dic.get(str(paras[self.src_table_cols_index.get("devID")]))
        neId = paras[self.src_table_cols_index.get("devID")]
        purpose = "default"
        connectStatus = "-1"

        userName = paras[self.src_table_cols_index.get("userName")]
        password = paras[self.src_table_cols_index.get("password")]
        port = paras[self.src_table_cols_index.get("port")]
        userPrivateKey = paras[self.src_table_cols_index.get("privateKey")]
        userPrivateKeyPwd = paras[self.src_table_cols_index.get("passwordPhrase")]
        # Done: int转str，缺少环境，枚举值待定
        # 升级前 升级后
        # 1 User
        # 2 PrivateKey
        # 3 PrivateKeyPassword
        authModeMap = {
            "1": "User",
            "2": "PrivateKey",
            "3": "PrivateKeyPassword",
        }
        authMode = authModeMap.get(str(paras[self.src_table_cols_index.get("authMode")]))
        loginTimeout = paras[self.src_table_cols_index.get("loginTimeout")]
        responseTimeout = paras[self.src_table_cols_index.get("responseTimeout")]

        createAt = str(int(time.time() * 1000))
        updateAt = str(int(time.time() * 1000))

        return tuple(None if x is None else str(x) for x in (modelId, fModelId, tenantId, neResId, neId, purpose,
                               connectStatus, userName, password, port, userPrivateKey,
                               userPrivateKeyPwd, authMode, loginTimeout, responseTimeout,
                               createAt, updateAt))

    def to_UpgradePara(self, datas):
        self.get_resid_from_idmapping(datas)
        col_names = "`" + ("`, `".join(self.dst_table_cols)) + "`"
        val_ids = ":" + (",:".join((str(x+1) for x in range(len(self.dst_table_cols)))))
        insert_stmt = "INSERT INTO `%s` (%s) VALUES(%s)" % (self.dst_table, col_names, val_ids)
        self.debug("insert sql stmt: %s" % insert_stmt)
        list_datas = []
        for data in datas:
            data = self.convert_data(data)
            if len(data) == len(self.dst_table_cols):
                tuple_data = tuple(data)
                list_datas.append(tuple_data)
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def do(self):
        try:
            self.info('UpgradeNetconfPara::do start!')
            select_stmt = "select %s from %s" % (",".join(self.src_table_cols), self.src_table)
            self.debug("execute sql: %s" % select_stmt)
            try:
                self.src_db_cursor.execute(select_stmt)
                datas = self.src_db_cursor.fetchall()
            except BaseException as be:
                self.warning("execute sql failed, err is: %s" % str(be))
                datas = []

            final_data = []
            if datas:
                final_data = list(datas)
            self.info('get para data success count: %d' % len(final_data))

            self.to_UpgradePara(final_data)
            self.info('to_UpgradePara done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeNetconfPara got exception')
            self.error("$s" % traceback.format_exc())
            return -1
        self.info('UpgradeNetconfPara::do done')
        return 0

if __name__ == '__main__':

    tool = UpgradeNetconfPara()
    print('[INFO] UpgradeNetconfPara start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeNetconfPara finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
