#!/bin/bash -

###############################################################################################
function fn_sys_assert_not_null()
{
    local l_var_name="$1"

    local l_var_value="${!l_var_name}"
    [ -z "$l_var_value" ] && fn_sys_log_error "Variable [${l_var_name}] must not be NULL" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_number()
{
    local l_var_name="$1"

    local l_var_value="${!l_var_name}"
    fn_sys_assert_not_null "$l_var_name" || return 1
    [ $(fn_sys_is_valid_number "$l_var_value") == "false" ] && fn_sys_log_error "Variable [${l_var_name}] with value [${l_var_value}] is not a valid number" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_bool_number()
{
    local l_var_name="$1"

    local l_var_value="${!l_var_name}"
    fn_sys_assert_not_null "$l_var_name" || return 1
    fn_sys_assert_is_number "$l_var_name" || return 1

    ([ "$l_var_value" != "0" ] && [ "$l_var_value" != "1" ]) && fn_sys_log_error "Variable [${l_var_name}] with value [${l_var_value} is not a bool number [0, 1]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_value_greater_than()
{
    local l_var_name="$1"
    local l_min_value="$2"

    fn_sys_assert_not_null "$l_var_name" || return 1
    fn_sys_assert_is_number "$l_var_name" || return 1

    local l_var_value=${!l_var_name}
    [ -z "$l_min_value" ] && fn_sys_log_warn "Min value is null. Ignore the validate and return true" && return 0
    [ "$l_var_value" -le "$l_min_value" ] && fn_sys_log_error "Variable [${l_var_name}] value [${l_var_value}] is less than or equal to value [${l_min_value}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_value_greater_than_or_equal_to()
{
    local l_var_name="$1"
    local l_min_value="$2"

    fn_sys_assert_not_null "$l_var_name" || return 1
    fn_sys_assert_is_number "$l_var_name" || return 1

    local l_var_value="${!l_var_name}"
    [ -z "$l_min_value" ] && fn_sys_log_warn "Min value is null. Ignore the validate and return true" && return 0
    [ "$l_var_value" -lt "$l_min_value" ] && fn_sys_log_error "Variable [${l_var_name}] value [${l_var_value}] is less than value [${l_min_value}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_value_less_than()
{
    local l_var_name="$1"
    local l_max_value="$2"

    fn_sys_assert_not_null "$l_var_name" || return 1
    fn_sys_assert_is_number "$l_var_name" || return 1

    local l_var_value="${!l_var_name}"
    [ -z "$l_max_value" ] && fn_sys_log_warn "Max value is null. Ignore the validate and return true" && return 0
    [ "$l_var_value" -ge "$l_max_value" ] && fn_sys_log_error "Variable [${l_var_name}] value [${l_var_value}] is greater than or equal to value [${l_max_value}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_value_less_than_or_equal_to()
{
    local l_var_name="$1"
    local l_max_value="$2"

    fn_sys_assert_not_null "$l_var_name" || return 1
    fn_sys_assert_is_number "$l_var_name" || return 1

    local l_var_value="${!l_var_name}"
    [ -z "$l_max_value" ] && fn_sys_log_warn "Max value is null. Ignore the validate and return true" && return 0
    [ "$l_var_value" -gt "$l_max_value" ] && fn_sys_log_error "Variable [${l_var_name}] value [${l_var_value}] is greater than value [${l_max_value}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_number_in_range()
{
    local l_var_name="$1"
    local l_min_value="$2"
    local l_max_value="$3"
    local l_include_min_value="$4"
    local l_include_max_value="$5"

    local l_var_value="${!l_var_name}"
    if [ "$(fn_sys_is_number_in_range "$l_var_value" "$l_min_value" "$l_max_value" "$l_include_min_value" "$l_include_max_value")" != "true" ]; then
        fn_sys_log_error "Variable [${l_var_name}] value [${l_var_value}] is not in specified range" && return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_assert_is_path_existing()
{
    local l_path_var_name="$1"

    local l_path="${!l_path_var_name}"

    [ -z "$l_path" ] && fn_sys_log_error "Path [${l_path_var_name}] value must not be null" && return 1
    [ ! -e "$l_path" ] && fn_sys_log_error "Path [${l_path}] is not existing" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_is_directory()
{
    local l_path_var_name="$1"

    local l_path="${!l_path_var_name}"

    [ -z "$l_path" ] && fn_sys_log_error "Path [${l_path_var_name}] value must not be null" && return 1
    [ ! -d "$l_path" ] && fn_sys_log_error "Directory [${l_path}] is not existing" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_is_physical_directory()
{
    local l_path_var_name="$1"

    local l_path="${!l_path_var_name}"

    if [ "$(fn_sys_is_physical_directory "$l_path")" == "false" ]; then
        fn_sys_log_error "Path [${l_path}] is not a physical directory"
        return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_assert_is_file()
{
    local l_path_var_name="$1"

    local l_path="${!l_path_var_name}"

    [ -z "$l_path" ] && fn_sys_log_error "Path [${l_path_var_name}] value must not be null" && return 1
    [ ! -f "$l_path" ] && fn_sys_log_error "File [${l_path}] is not existing" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_is_not_the_same_path()
{
    local l_path_1_var_name="$1"
    local l_path_2_var_name="$2"

    local l_path_1="${!l_path_1_var_name}"
    local l_path_2="${!l_path_2_var_name}"

    if [ "$l_path_1" == "$l_path_2" ]; then
        fn_sys_log_error "Path [${l_path_1}] for [${l_path_1_var_name}] is the same as path [${l_path_2}] for [${l_path_2_var_name}]"
        return 1
    fi
    return 0
}

###############################################################################################
# $1: Required. Variable name
# $2: Required. Range list. Delimited by ","
function fn_sys_assert_in_range()
{
    local l_token_var_name="$1"
    local l_value_list="$2"

    local l_token_value="${!l_token_var_name}"

    fn_sys_assert_not_null "$l_token_var_name" || return 1
    [ -z "$l_value_list" ] && fn_sys_log_error "Value list must not be null" && return 1

    [ $(fn_sys_in "$l_value_list" "$l_token_value") == "false" ] && fn_sys_log_error "Value [${l_token_value}] of token [${l_token_var_name}] is not in range [${l_value_list}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_directory_writable()
{
    local l_directory_var_name="$1"

    local l_directory="${!l_directory_var_name}"
    fn_sys_assert_not_null "$l_directory_var_name" || return 1
    fn_sys_assert_is_directory "$l_directory_var_name" || return 1

    local l_file=${l_directory}/dummy_$$_${RANDOM}_$$_${RANDOM}_$$_${RANDOM}.txt
    touch "$l_file" &> /dev/null
    if [ "$?" -ne "0" ]; then
        fn_sys_log_error "Directory [${l_directory}] is not writable"
        return 1
    else
        rm -rf "$l_file"
    fi

    return 0
}

###############################################################################################
function fn_sys_assert_current_user_matched()
{
    local l_expected_user="$1"

    [ -z "$l_expected_user" ] && fn_sys_log_error "expected user can not be null" && return 1
    [ "$(whoami)" != "$l_expected_user" ] && fn_sys_log_error "Current user is not expected user: ${l_expected_user}" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_is_ip()
{
    local l_ip_var_name="$1"

    local l_ip="${!l_ip_var_name}"
    fn_sys_assert_not_null "$l_ip_var_name" || return 1

    [ "$(fn_sys_is_valid_ip "$l_ip")" == "false" ] && fn_sys_log_error "Token [${l_ip}] is not a valid ip" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_is_local_ip()
{
    local l_ip_var_name="$1"

    local l_ip="${!l_ip_var_name}"
    fn_sys_assert_not_null "$l_ip_var_name" || return 1

    [ "$(fn_sys_is_valid_ip "$l_ip")" == "false" ] && fn_sys_log_error "Token [${l_ip}] is not a valid ip" && return 1

    ip a | grep "${l_ip}/"
    [[ $? -ne 0 ]] && fn_sys_log_warn "This IP address [${l_ip}] is not in the current node." && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_is_ip_list()
{
    local l_ip_list_var_name="$1"
    local l_delimiter="${2:-;}"

    local l_ip_list="${!l_ip_list_var_name}"
    fn_sys_assert_not_null "$l_ip_list_var_name" || return 1

    for each_ip in $(echo "$l_ip_list" | awk -F"$l_delimiter" '{ for(i=1; i<=NF; i++) print $i }'); do
        [ "$(fn_sys_is_valid_ip "$each_ip")" == "false" ] && fn_sys_log_error "Token [${each_ip}] is not a valid ip" && return 1
    done

    return 0
}

###############################################################################################
function fn_sys_assert_is_symbol_link()
{
    local l_file_var_name="$1"
    local l_file_path="${!l_file_var_name}"
    fn_sys_assert_not_null "l_file_path" || return 1

    [ "$(fn_sys_is_symbol_link "$l_file_path")" == "false" ] && fn_sys_log_error "File [${l_file_path}] is not a symbol link" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_not_symbol_link()
{
    local l_file_var_name="$1"
    local l_file_path="${!l_file_var_name}"
    fn_sys_assert_not_null "l_file_path" || return 1

    [ "$(fn_sys_is_symbol_link "$l_file_path")" == "true" ] && fn_sys_log_error "File [${l_file_path}] is a symbol link" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_symbol_link_to_file()
{
    local l_file_var_name="$1"
    local l_file_path="${!l_file_var_name}"
    fn_sys_assert_not_null "l_file_path" || return 1

    [ "$(fn_sys_is_symbol_link "$l_file_path")" == "false" ] && fn_sys_log_error "Path [${l_file_path}] is not a symbol link" && return 1
    [ ! -f "$l_file_path" ] && fn_sys_log_error "Path [${l_file_path}] is not a symbol link for file" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_symbol_link_to_directory()
{
    local l_file_var_name="$1"
    local l_file_path="${!l_file_var_name}"
    fn_sys_assert_not_null "l_file_path" || return 1

    [ "$(fn_sys_is_symbol_link "$l_file_path")" == "false" ] && fn_sys_log_error "Path [${l_file_path}] is not a symbol link" && return 1
    [ ! -d "$l_file_path" ] && fn_sys_log_error "Path [${l_file_path}] is not a symbol link for directory" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_symbol_or_hard_link()
{
    local l_file_var_name="$1"
    local l_file_path="${!l_file_var_name}"
    fn_sys_assert_not_null "l_file_path" || return 1

    [ "$(fn_sys_is_link_file "$l_file_path")" == "false" ] && fn_sys_log_error "File [${l_file_path}] is not neither symbol nor hard link" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_mount_point()
{
    local l_mount_point_path_var="$1"

    local l_mount_point_path="${!l_mount_point_path_var}"
    fn_sys_assert_is_directory "l_mount_point_path" || return 1

    [ "$(fn_sys_is_mount_point_existing "$l_mount_point_path")" == "false" ] && fn_sys_log_error "Directory [${l_mount_point_path}] is not a mount point" && return 1
    return 0
}

###############################################################################################
function fn_sys_assert_is_simple_word()
{
    local l_original_word_var_name="$1"

    local l_original_word="${!l_original_word_var_name}"
    fn_sys_assert_not_null "$l_original_word_var_name" || return 1

    local l_normalized_word="$(echo "$l_original_word" | sed 's#[A-Za-z0-9_]##g')"
    if [ -n "$l_normalized_word" ]; then
        fn_sys_log_error "Value [${l_original_word}] for variable [${l_original_word_var_name}] is invalid"
        return 1
    fi
    return 0
}

###############################################################################################
function fn_sys_assert_is_valid_host_name()
{
    local l_host_name_var="$1"

    local l_host_name="${!l_host_name_var}"
    fn_sys_assert_not_null "$l_host_name_var" || return 1

    if [ "$(fn_sys_is_valid_host_name "$l_host_name")" != "true" ]; then
        fn_sys_log_error "Value [${l_host_name}] for variable [${l_host_name_var}] is not a valid host name"
        return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_assert_is_valid_nic()
{
    local l_nic_name_var="$1"

    local l_nic_name="${!l_nic_name_var}"
    fn_sys_assert_not_null "$l_nic_name_var" || return 1

    if [ "$(fn_sys_is_valid_nic_name "$l_nic_name")" != "true" ]; then
        fn_sys_log_error "Value [${l_nic_name}] for variable [${l_nic_name_var}] is not a valid nic name"
        return 1
    fi
}

###############################################################################################
function fn_sys_assert_is_real_os_nic()
{
    local l_nic_name_var="$1"

    fn_sys_assert_is_valid_nic "$l_nic_name_var" || return 1

    local l_nic_name="${!l_nic_name_var}"
    if [ "$(fn_sys_is_real_os_nic "$l_nic_name")" != "true" ]; then
        fn_sys_log_error "Value [${l_nic_name}] for variable [${l_nic_name_var}] is not a real nic name"
        return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_assert_is_reserved_user()
{
    local l_user_name_var="$1"
    local readonly c_reserved_user_list="root,ossuser,ossadm,dbuser"

    local l_user_name="${!l_user_name_var}"
    fn_sys_assert_not_null "$l_user_name_var" || return 1

    if [ "$(fn_sys_is_reserved_user "$l_user_name")" == "true" ]; then
        fn_sys_log_error "User [${l_user_name}] is a reserved user"
        return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_assert_file_permission_matched()
{
    local l_file_path_var_name="$1"
    local l_file_permission="$2"

    local l_file_path="${!l_file_path_var_name}"
    fn_sys_assert_is_path_existing "l_file_path" || return 1

    [ "$(fn_sys_is_valid_number "$l_file_permission")" == "false" ] && fn_sys_log_error "Permission [${l_file_permission}] is not valid" && return 1

    local l_file_actual_permission="$(fn_sys_get_file_permission_in_octal "$l_file_path")"
    [ -z "$l_file_actual_permission" ] && fn_sys_log_error "Permission denied to get file permission for [${l_file_path}]" && return 1

    local l_expected_permission="$(echo "$l_file_permission" | sed 's#^0##g')"

    [ "$l_expected_permission" != "$l_file_actual_permission" ] && fn_sys_log_error "File permission does not match. Expected [${l_expected_permission}]; Actual [${l_file_actual_permission}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_file_owner_matched()
{
    local l_file_path_var_name="$1"
    local l_file_owner="$2"

    local l_file_path="${!l_file_path_var_name}"
    fn_sys_assert_is_path_existing "l_file_path" || return 1

    local l_file_user="$(echo "$l_file_owner" | awk -F':' '{ print $1 }')"
    local l_file_group="$(echo "$l_file_owner" | awk -F':' '{ print $2 }')"
    fn_sys_assert_is_simple_word "l_file_user" || return 1
    fn_sys_assert_is_simple_word "l_file_group" || return 1

    local l_file_actual_owner="$(fn_sys_get_file_owner_and_group_name "$l_file_path")"
    [ -z "$l_file_actual_owner" ] && fn_sys_log_error "Permission denied to get file owner for [${l_file_path}]" && return 1

    [ "$l_file_owner" != "$l_file_actual_owner" ] && fn_sys_log_error "File owner does not match. Expected [${l_file_owner}]; Actual [${l_file_actual_owner}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_os_user_exists()
{
    local l_user_name_var_name="$1"

    fn_sys_assert_not_null "$l_user_name_var_name" || return 1

    local l_user_name="${!l_user_name_var_name}"
    id "$l_user_name" &> /dev/null
    [ "$?" -ne "0" ] && fn_sys_log_error "OS user [${l_user_name}] is not existing" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_os_group_exists()
{
    local l_group_name_var_name="$1"

    fn_sys_assert_not_null "$l_group_name_var_name" || return 1

    local l_group_name="${!l_group_name_var_name}"
    local l_gid="$(fn_sys_get_os_gid "$l_group_name")"
    [ -z "$l_gid" ] && fn_sys_log_error "OS group [${l_group_name}] is not existing" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_os_user_uid_matched()
{
    local l_user_name_var_name="$1"
    local l_expected_uid="$2"

    local l_user_name="${!l_user_name_var_name}"
    fn_sys_assert_not_null "$l_user_name_var_name" || return 1

    local l_actual_uid=$(fn_sys_get_os_uid "$l_user_name")
    [ -z "$l_actual_uid" ] && fn_sys_log_error "User [${l_user_name}] is not existing" && return 1
    [ "$l_actual_uid" != "$l_expected_uid" ] && fn_sys_log_error "Actual uid [${l_actual_uid}] for user [${l_user_name}] does not match the expected uid [${l_expected_uid}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_os_group_gid_matched()
{
    local l_group_name_var_name="$1"
    local l_expected_gid="$2"

    fn_sys_assert_os_group_exists "$l_group_name_var_name" || return 1

    local l_group_name="${!l_group_name_var_name}"
    local l_actual_gid="$(fn_sys_get_os_gid "$l_group_name")"
    [ "$l_actual_gid" != "$l_expected_gid" ] && fn_sys_log_error "Actual gid [${l_actual_gid}] for group [${l_group_name}] does not match the expected gid [${l_expected_gid}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_assert_user_group_matched()
{
    local l_user_name_var_name="$1"
    local l_group_list="$2"

    fn_sys_assert_os_user_exists "$l_user_name_var_name" || return 1

    local l_user_name="${!l_user_name_var_name}"

    local l_actual_groups="$(id -Gn "$l_user_name" | awk '{ for(i=1; i<=NF; i++) print $i }' | sort -u | xargs)"
    local l_expected_groups="$(echo "$l_group_list" | awk -F',' '{ for(i=1; i<=NF; i++) print $i }' | sort -u | xargs)"
    [ "$l_actual_groups" != "$l_expected_groups" ] && fn_sys_log_error "Actual groups for user [${l_user_name}] does not match the expected groups [${l_group_list}]" && return 1

    return 0
}
