#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.

set +x

xuser=$(id -un)
if [ "${xuser}" != "arbiter" ]; then
    echo "please login as arbiter, then excute the script again."
    exit 1
fi

function get_work_path() {
    dir=$(dirname "$0")
    ispointstart=$(echo "${dir}" | awk -F/ '{print $1}')
    if [ "${ispointstart}" == "." ]; then
        dir=$(echo "${dir}" | sed 's/^.//')
        cur_path="$(pwd)${dir}"
    elif [ "${ispointstart}" == ".." ]; then
        cur_path="$(pwd)/${dir}"
    else
        cur_path="${dir}"
    fi
    return 0
}
get_work_path

. /etc/profile >/dev/null 2>&1
. "${cur_path}"/../env/env.properties
TMOUT=0

root_dir=$(echo $(
    cd "$(dirname "$0")"
    pwd
))
. $root_dir/logging.sh
LOG_FILE="${ETCD_LOG}/service.log"
PROCESS_NAME="arbitration-etcd_${INDEX}"
ALARM_PROCESS_NAME="arbitration-monitor-alarm"
BASE_DIR="${MONITOR_HOME}"
LIB_DIR="${MONITOR_HOME}/lib/"
CONF_DIR="${MONITOR_HOME}/conf/"
LOG4J_DIR="${ETCD_HOME}/conf/"
PROP_FILE="${MONITOR_HOME}/conf/aam.properties"
PID_FILE="${ETCD_HOME}/arbitration-etcd.pid"

JAVA_OPTS="-Dprocess.name=$PROCESS_NAME -Dbase.dir=$BASE_DIR -Xss256k -Xms128M -Xmx128M -XX:MaxDirectMemorySize=256m"
JAVA_OPTS="$JAVA_OPTS -Djava.nio.channels.spi.SelectorProvider=sun.nio.ch.EPollSelectorProvider"
JAVA_OPTS="$JAVA_OPTS -Dbeetle.application.home.path=$CONF_DIR -Dlog4j.configurationFile=$LOG4J_DIR/log4j2.xml -Dlog4j2.formatMsgNoLookups=true"
JAVA_OPTS="$JAVA_OPTS -XX:+UnlockExperimentalVMOptions -XX:+UseG1GC -XX:G1PeriodicGCLoadThreshold=20"
JAVA_OPTS="$JAVA_OPTS -XX:G1PeriodicGCInterval=3000 -XX:+G1Uncommit"
JAVA_OPTS="$JAVA_OPTS -XX:CompileCommand=exclude,org.springframework.core.ResolvableType::forMethodParameter"
cat "${PROP_FILE}" | grep aam.local.ip | grep "]" >/dev/null 2>&1
if [ $? -ne 0 ]; then
    JAVA_OPTS="$JAVA_OPTS -Djava.net.preferIPv4Stack=true -Djava.net.preferIPv4Addresses=true"
fi
LOCAL_JAR=$(ls $LIB_DIR | sed "s|^|$LIB_DIR|" | tr '\n' ':')
JAVA_CP=$CLASSPATH:$LOCAL_JAR:$CONF_DIR
JAVA_MAIN="com.huawei.raft.RaftServerMain"

function kill_process() {
    kill_single_process "${PROCESS_NAME}"
    if [ "${INDEX}" = "0" ]; then
        kill_single_process "arbitration-etcd"
    fi
}

function kill_single_process() {
    process_name="$1"
    local pid=$(ps -efww | grep -w "Dprocess.name=${process_name}" | grep -v grep | awk '{print $2}')
    local num=$(ps -efww | grep -w "Dprocess.name=${process_name}" | grep -v grep | wc -l)
    if [ $num -eq 0 ]; then
        g_LOG_Info "the arbitration-etcd process already stoped."
    elif [ $num -eq 1 ]; then
        #use systemd to stop this process
        g_LOG_Info "find one arbitration-etcd process, pid=${pid}."
        kill -9 $pid >>${LOG_FILE} 2>&1
        sleep 2
    else
        g_LOG_Warn "find one more arbitration-etcd process, pid=${pid}."
        #force stop all process
        kill -9 $pid >>${LOG_FILE} 2>&1
        sleep 2
    fi
}

function try_get_pid() {
    for i in {1..3}; do
       pid=$(ps -efww | grep "Dprocess.name=${PROCESS_NAME}" | grep -v grep | awk '{print $2}')
       if [ "x${pid}" = "x" ]; then
           g_LOG_Error "query arbitration-etcd process pid failed. retry: ${i}"
           sleep 3
       else
           g_LOG_Info "start arbitration-etcd process with pid=${pid}."
           echo $pid >$PID_FILE
           return 0
       fi
    done
    g_LOG_Error "query arbitration-etcd process pid failed. over retry."
    return 1
}

function do_start() {
    g_LOG_Info "begin to start arbitration-etcd ..."

    sed -i 's/etcd.mode=false/etcd.mode=true/' $PROP_FILE

    pid=$(ps -efww | grep "Dprocess.name=${PROCESS_NAME}" | grep -v grep | awk '{print $2}')
    if [ -n "${pid}" ]; then
       g_LOG_Info "arbitration-etcd is running, no need start. Pid: ${pid}"
       return 0
    fi

    export CIPHER_ROOT=${COMMON_HOME}/cipher
    export OSS_ROOT=${MONITOR_HOME}
    export MONITOR_ROOT=${BASE_DIR}
    export ETCD_ROOT=${ETCD_HOME}
    export OSS_ROOT=${BASE_DIR}

    ${ARBITRATION_JAVA_HOME}/bin/java $JAVA_OPTS -cp $JAVA_CP $JAVA_MAIN >/dev/null 2>&1 &
    if [ $? -ne 0 ]; then
        g_LOG_Error "start arbitration-etcd failed."
        return 1
    fi

    sleep 2

    try_get_pid || return 1

    g_LOG_Info "start arbitration-etcd success."

    sleep 10

    return 0
}

function do_stop() {
    g_LOG_Info "begin to stop arbitration-etcd ..."
    sed -i 's/etcd.mode=true/etcd.mode=false/' $PROP_FILE

    kill_process
    g_LOG_Info "stop arbitration-etcd success."
    return 0
}

function do_restart() {
    g_LOG_Info "begin to restart arbitration-etcd ..."
    kill_process
    if [ $? -ne 0 ]; then
        return 1
    fi
    g_LOG_Info "stop arbitration-etcd success."

    sleep 1

    do_start
    if [ $? -ne 0 ]; then
        return 1
    fi

    g_LOG_Info "restart arbitration-etcd success."
    return 0
}

case "$1" in
start)
    echo -n "Starting arbitration-etcd daemon ... "
    do_start
    exit $?
    ;;
stop)
    echo -n "Stoping arbitration-etcd daemon ... "
    do_stop
    exit $?
    ;;
restart)
    echo -n "Restarting arbitration-etcd daemon ... "
    do_restart
    exit $?
    ;;
*)
    echo "Usage: $0 {start|stop|restart}"
    exit 1
    ;;
esac
