#!/bin/bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2020-2020. All rights reserved.
# File name: arbitration_generate_ca.sh
# Description: 生成ca证书并修改OMP节点下的ha.properties文件
###########################################################
function get_work_path() {
    local dir=$(dirname $0)
    local ispointstart=$(echo $dir | awk -F/ '{print $1}')
    if [ "${ispointstart}" = "." ]; then
        local dir=$(echo ${dir} | sed 's/^.//')
        workfolder="$(pwd)${dir}"
    elif [ "${ispointstart}" = ".." ]; then
        workfolder="$(pwd)/${dir}"
    else
        workfolder="${dir}"
    fi
}

function create_check_file() {
    result_code=$1
    create_check_result_file ${result_code} "check_generate_ca_result.sh"
}

function get_params() {
    master_omp_node_ip=$1
    master_omp_node_uname=$2
    read -p "es_input: master_omp_node_pwd" -s -r master_omp_node_pwd
    read -p "es_input: master_omp_node_ossadm_pwd" -s -r master_omp_node_ossadm_pwd

    slave_omp_node_ip=$3
    slave_omp_node_uname=$4
    read -p "es_input: slave_omp_node_pwd" -s -r slave_omp_node_pwd
    read -p "es_input: slave_omp_node_ossadm_pwd" -s -r slave_omp_node_ossadm_pwd

    # 获取备NCE的common_server节点间通信IP
    slave_inner_comm_ip=$5

    remote_node_ips=$6
    all_access_ip_list=($(echo ${remote_node_ips} | tr ',' ' '))
    node_size=${#all_access_ip_list[@]}

    # 转义特殊字符
    master_omp_node_pwd=$(echo ${master_omp_node_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    master_omp_node_ossadm_pwd=$(echo ${master_omp_node_ossadm_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')

    slave_omp_node_pwd=$(echo ${slave_omp_node_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    slave_omp_node_ossadm_pwd=$(echo ${slave_omp_node_ossadm_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')

    get_root_params
}

function get_root_params() {
    read -p "es_input: node_2_root_pwd" -s -r node_2_root_pwd
    read -p "es_input: node_3_root_pwd" -s -r node_3_root_pwd
    read -p "es_input: node_4_root_pwd" -s -r node_4_root_pwd
    read -p "es_input: node_5_root_pwd" -s -r node_5_root_pwd

    node_3_root_pwd=$(echo ${node_3_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    node_5_root_pwd=$(echo ${node_5_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')

    node_root_pwd_list=("${node_3_root_pwd}" "${node_5_root_pwd}")

    if [ ${node_size} -eq 4 ]; then
        node_2_root_pwd=$(echo ${node_2_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
        node_4_root_pwd=$(echo ${node_4_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')

        node_root_pwd_list=("${node_2_root_pwd}" "${node_3_root_pwd}" "${node_4_root_pwd}" "${node_5_root_pwd}")
    fi
}

# 更新远端manifest中的密码
function update_remote_ca() {
    node_ip=$1
    node_root_pwd=$2
    random_password=$3
    expect <<EOF >/dev/null 2>&1
set timeout 30
spawn ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null root@${node_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${node_root_pwd}\r"}
}
expect {
    "*#" {
        send ". /opt/arbitration_file/env.properties && chown arbiter: \$\{ETCD_HOME\}/ca/* && chmod 600 \$\{ETCD_HOME\}/ca/* && sudo -u arbiter bash \$\{MONITOR_HOME\}/script/arb_cipher_keys.sh update_cipher CA\r"
        expect {
            "please input random password:" {send -- "${random_password}\r"}
        }
        expect {
            "update cipher success." { exit 100 }
        }
    }
}

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]
EOF
    update_result=$?
    if [ ${update_result} -eq 100 ]; then
        arbitration_log INFO "GENERATE_CA" "update ${node_ip} ca cipher success."
        return 0
    fi
    arbitration_log ERROR "GENERATE_CA" "update ${node_ip} ca cipher failed. code: ${update_result}"
    return 1
}

# 同步ca
function sync_ca() {
    local node_ip=$1
    local node_root_pwd=$2
    local node_etcd_home=$3
    local file_name=$4
    expect <<EOF >/dev/null 2>&1
set timeout 30
spawn scp -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null /opt/arbitration-etcd/ca/${file_name} root@\[${node_ip}\]:${node_etcd_home}/ca/${file_name}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${node_root_pwd}\r"}
    "100%" { exit 0 }
}
expect eof
EOF
    if [ $? -ne 0 ]; then
        arbitration_log ERROR "GENERATE_CA" "sync ca file ${file_name} to ${node_ip} failed."
        return 1
    fi
}

function get_remote_etcd_home() {
    local node_ip=$1
    local node_root_pwd=$2
    local tmp_result=/opt/arbitration_file/remote_tmp_result
    expect <<EOF >${tmp_result}
set timeout 30
spawn ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null root@${node_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${node_root_pwd}\r"}
}
expect {
    "*#" {
        send ". /opt/arbitration_file/env.properties && echo \$\{ETCD_HOME\}\r"
        expect {
            "/opt/arbitration-etcd*" { exit 100 }
        }
    }
}
expect timeout { exit 1 }
expect eof
catch wait result
exit [lindex $result 3]
EOF
    if [ ! -f ${tmp_result} ]; then
        arbitration_log ERROR "GENERATE_CA" "get remote etcd home failed, no result."
        return 1
    fi
    if grep -Eo "^/opt/arbitration-etcd_[1-9]" ${tmp_result}; then
        rm -f ${tmp_result}
        return 0
    elif grep -Eo "^/opt/arbitration-etcd" ${tmp_result}; then
        rm -f ${tmp_result}
        return 0
    fi
    arbitration_log ERROR "GENERATE_CA" "get remote etcd home failed."
    return 1
}

# 同步ca并更新远端ca
function sync_ca_and_update() {
    for j in ${!all_access_ip_list[@]}; do
        local node_ip=${all_access_ip_list[j]}
        local node_root_pwd=${node_root_pwd_list[j]}
        local remote_etcd_home

        # 如果已有互信先删除，处理对端重装互信失效的情况
        [ -f /root/.ssh/known_hosts ] && sed -i "/^${node_ip} /d" /root/.ssh/known_hosts

        remote_etcd_home=$(get_remote_etcd_home "${node_ip}" "${node_root_pwd}") || return 1
        sync_ca "${node_ip}" "${node_root_pwd}" "${remote_etcd_home}" ca.cer || return 1
        sync_ca "${node_ip}" "${node_root_pwd}" "${remote_etcd_home}" ca_key.pem || return 1
        sync_ca "${node_ip}" "${node_root_pwd}" "${remote_etcd_home}" manifest.json || return 1
        update_remote_ca "${node_ip}" "${node_root_pwd}" "${random_password}" || return 1
    done
}

function generate_ca() {
    generate_local_ca || return 1
    sync_ca_and_update || return 1
}

function generate_local_ca() {
    # 生成CA证书前，如果存在先删除目录
    [ -d "${ETCD_HOME}/ca" ] && rm -rf "${ETCD_HOME}/ca"

    rm -f /opt/arbitration_file/check_generate_ca_result.sh

    arbitration_log INFO "GENERATE_CA" "Start generate ca."

    # 调用随机数生成随机密码，至少含有一个大写字母、小写字母、数字、特殊字符
    random_password=$(bash ${workfolder}/generate_random_pwd.sh)
    random_password=$(echo ${random_password} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')

    expect <<EOF
set timeout 30
spawn sudo -u arbiter bash ${ETCD_HOME}/script/init-ca.sh
expect {
        "please input random password:" {send -- "${random_password}\r"}
    }

expect eof

EOF

    # 检查是否生成证书
    if [[ -f "${ETCD_HOME}/ca/ca.cer" && -f "${ETCD_HOME}/ca/ca.csr" && -f "${ETCD_HOME}/ca/ca_key.pem" && -f "${ETCD_HOME}/ca/manifest.json" ]]; then
        arbitration_log INFO "GENERATE_CA" "Generate ca successfully."
    else
        arbitration_log ERROR "GENERATE_CA" "Generate ca failed."
        return 1
    fi

    return 0
}

function remote_modify_ha() {
    exec_file="${workfolder}/remote_modify_ha.sh"

    # 修改主站点omp的/opt/oss/share/manager/DRService/configuration/ha.properties 的 etcd_pwd_of_ts 字段密文
    expect <<EOF >/dev/null 2>&1
set timeout 30
spawn bash ${exec_file} ${master_omp_node_uname} "etcd_pwd_of_ts"
    expect {
        "please input login password:" {send -- "${master_omp_node_pwd}\r"}
    }


    expect {
        "please input ossadm password:" {send -- "${master_omp_node_ossadm_pwd}\r"}
    }

    expect {
        "please input random password:" {send -- "${random_password}\r"}
    }

    expect {
        "*modify properties failed*" { exit 255 }
        "*modify properties successfully*" { exit 100 }
    }

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]

EOF
    result_master=$?
    sleep 1

    # 如果已有互信先删除，处理对端重装互信失效的情况
    [ -f /root/.ssh/known_hosts ] && sed -i "/^${slave_inner_comm_ip} /d" /root/.ssh/known_hosts

    # 修改备站点omp的/opt/oss/share/manager/DRService/configuration/ha.properties 的 etcd_pwd_of_ts 字段密文
    expect <<EOF >/dev/null 2>&1
set timeout 60
spawn ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null root@${slave_inner_comm_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${node_3_root_pwd}\r"}
}

expect {
    "*#" {
        send "bash ${exec_file} ${slave_omp_node_uname} \"etcd_pwd_of_ts\"\r"
        expect {
            "please input login password:" {send -- "${slave_omp_node_pwd}\r"}
        }

        expect {
            "please input ossadm password:" {send -- "${slave_omp_node_ossadm_pwd}\r"}
        }

        expect {
            "please input random password:" {send -- "${random_password}\r"}
        }

        expect {
            "*modify properties failed*" { exit 255 }
            "*modify properties successfully*" { exit 100 }
        }

    }
}

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]

EOF

    result_slave=$?
}

function update_local_ca_cipher() {
    sudo -u arbiter bash "${MONITOR_HOME}/script/arb_cipher_keys.sh" update_cipher_from_kmc1 CA
    update_local_result=$?
    if [ ${update_local_result} -ne 0 ]; then
        arbitration_log ERROR "UPDATE_CA_CIPHER" "update local ca cipher failed."
        return 1
    fi
    arbitration_log INFO "UPDATE_CA_CIPHER" "update local ca cipher success."
}

function update_remote_ca_cipher_item() {
    node_ip=$1
    node_root_pwd=$2
    random_password=$3
    expect <<EOF >/dev/null 2>&1
set timeout 30
spawn ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null root@${node_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${node_root_pwd}\r"}
}
expect {
    "*#" {
        send ". /opt/arbitration_file/env.properties && chown arbiter: \$\{ETCD_HOME\}/ca/* && chmod 600 \$\{ETCD_HOME\}/ca/* && sudo -u arbiter bash \$\{MONITOR_HOME\}/script/arb_cipher_keys.sh update_cipher_from_kmc1 CA\r"
        expect {
            "update cipher from KMC1.0 success." { exit 100 }
        }
    }
}

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]
EOF
    update_result=$?
    if [ ${update_result} -eq 100 ]; then
        arbitration_log INFO "UPDATE_CIPHER_FROM_KMC1" "update ${node_ip} ca cipher success."
        return 0
    fi
    arbitration_log ERROR "UPDATE_CIPHER_FROM_KMC1" "update ${node_ip} ca cipher failed. code: ${update_result}"
    return 1
}

function update_remote_ca_cipher() {
    arbitration_log INFO "UPDATE_REMOTE_CA_CIPHER" "remote nodes: ${all_access_ip_list}"
    for i in ${!all_access_ip_list[@]}; do
        node_ip=${all_access_ip_list[i]}
        node_root_pwd=${node_root_pwd_list[i]}
        update_remote_ca_cipher_item "${node_ip}" "${node_root_pwd}" || return 1
    done
}

function update_ca_cipher() {
    arbitration_log INFO "UPDATE_CA_CIPHER" "start update ca cipher"
    update_local_ca_cipher || return 1
    update_remote_ca_cipher || return 1
}

function main() {
    . ./arbitration_opertate_lib
    . ./env.properties
    . ./public.sh

    get_work_path

    # 获取参数
    get_params "$@"

    source /opt/arbitration_file/config/installEnv.properties
    if [ "${GENERATE_CERT}" == "false" ]; then
        update_ca_cipher
        if [ $? -ne 0 ]; then
            arbitration_log ERROR "UPDATE_CA_CIPHER" "update ca cipher failed."
            $(create_check_file 255)
            return 1
        fi
        $(create_check_file 100)
        return 0
    fi

    # 生成CA
    for i in {1..3}; do
        arbitration_log INFO "GENERATE_CA" "generate ca by ${i}"
        generate_ca
        if [ $? -eq 0 ]; then
            break
        fi

        if [ $i -eq 3 ]; then
            $(create_check_file 255)
            exit
        fi
    done

    # 刷新ha.properties
    remote_modify_ha
    if [ ${result_master} -eq 100 ] && [ ${result_slave} -eq 100 ]; then
        arbitration_log INFO "GENERATE_CA" "Modify ha.properties successfully."
    else
        # 重试一次
        remote_modify_ha
        if [ ${result_master} -ne 100 ] || [ ${result_slave} -ne 100 ]; then
            if [ ${result_master} -ne 100 ]; then
                arbitration_log ERROR "GENERATE_CA" "Modify master ha.properties failed."
            fi
            if [ ${result_slave} -ne 100 ]; then
                arbitration_log ERROR "GENERATE_CA" "Modify slave ha.properties failed."
            fi

            $(create_check_file 255)
            exit
        fi
    fi

    $(create_check_file 100)
}

main "$@"
