#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
. /opt/arbitration_file/env.properties
. /opt/arbitration_file/config/installEnv.properties
. ./arbitration_opertate_lib
# 备份证书文件

function init() {
    UPGRADE_RUNNING_FLAG=/opt/backupArb/arbitration_upgrade_running.flag
    UPGRADE_FLAG=/opt/backupArb/arbitration_upgrade.flag
    touch ${UPGRADE_FLAG}
    chmod 600 ${UPGRADE_FLAG}
}

function is_backup_cert() {
    arbitration_log INFO "${TYPE}" "start query if backup the cert"

    # 仲裁目录被破坏场景
    [ -f /opt/backupArb/arbitration_error.flag ] && return 1

    # 重试场景，判断上轮升级且生成证书是否完成，如果生成证书没有完全，则这里要重新生成证书
    [ -f ${UPGRADE_RUNNING_FLAG} ] && return 1

    # 默认不备份证书，安装仲裁时自动生成
    local is_backup_cert=1
    local ca_cer="${ETCD_HOME}/ca/ca.cer"
    if [ -f "${ca_cer}" ]; then
        # 如果原仲裁存在证书，则默认应该备份
        local is_backup_cert=0
        local local_ca_sum=$(sha256sum ${ca_cer} | awk '{print $1}')
        local config_ca_sum_array=($(echo ${AAM_OLD_CERT_SUM} | tr ',' ' '))
        for config_ca_sum in ${config_ca_sum_array[@]}; do
            if [ ${local_ca_sum} == ${config_ca_sum} ]; then
                # 如果原仲裁存在证书，但是为原华为预置证书，需要重新生成
                local is_backup_cert=1
                break
            fi
        done
    fi
    arbitration_log INFO "${TYPE}" "the backup the cert restult is ${is_backup_cert}"
    return ${is_backup_cert}
}

function refresh_properties() {
    local installEnv=/opt/arbitration_file/config/installEnv.properties
    chmod 750 ${installEnv}
    sed -i '/GENERATE_CERT=/d' "${installEnv}" >/dev/null 2>&1
    echo "export GENERATE_CERT=false" >>"${installEnv}"
}

function backup_certificate() {
    arbitration_log INFO "${TYPE}" "enter to backup the cert, the index is ${INDEX}"
    $(is_backup_cert)
    if [ $? -ne 0 ]; then
        # 需要重新生成证书
        if [ ! -f ${UPGRADE_RUNNING_FLAG} ]; then
            touch ${UPGRADE_RUNNING_FLAG}
            chmod 600 ${UPGRADE_RUNNING_FLAG}
            arbitration_log INFO "${TYPE}" "touch new flag file ${UPGRADE_RUNNING_FLAG}"
        fi
    else
        # 不需要重新生成证书，备份原有CA证书，用于后续恢复
        if [ ${INDEX} -eq 0 -o -z "${INDEX}" ]; then
            local upgrade_tmp="/opt/backupArb/upgrade_tmp"
        else
            local upgrade_tmp="/opt/backupArb/upgrade_tmp_${INDEX}"
        fi

        arbitration_log INFO "${TYPE}" "enter to backup the cert, the upgrade_tmp path is ${upgrade_tmp}"
        echo "Backup old certificate, index is ${INDEX}."
        if [ ! -d "${upgrade_tmp}" ]; then
            mkdir -p "${upgrade_tmp}" || return 1
        fi

        if [ ! -f "${upgrade_tmp}/ca.zip" ]; then
            cd "${ETCD_HOME}/ca" && zip -rp "${upgrade_tmp}/ca.zip" ./* || return 1
        fi
        arbitration_log INFO "${TYPE}" "success to backup the ca: ${upgrade_tmp}/ca.zip"

        refresh_properties
        return $?
    fi
    arbitration_log INFO "${TYPE}" "end to backup the cert, the index is ${INDEX}"
}

function unzip_and_modify_permission() {
    local zip_package=$1
    local target_dir=$2
    if [ ! -d ${target_dir} ]; then
        mkdir -p ${target_dir}
        chmod 700 ${target_dir}
    fi
    arbitration_log INFO "${TYPE}" "start to unzip and modify the permission: ${zip_package} to ${target_dir}"
    rm -rf "${target_dir}"/*
    unzip "${zip_package}" -d "${target_dir}" || return 1
    arbitration_log INFO "${TYPE}" "success to unzip ${zip_package}"
    chown -R arbiter:ArbiterGroup "${target_dir}" || return 1
    arbitration_log INFO "${TYPE}" "success to chown ${zip_package}"
    chmod 600 "${target_dir}"/* || return 1
    arbitration_log INFO "${TYPE}" "success to unzip and modify the permission: ${zip_package} to ${target_dir}"
}

# 恢复证书文件
function restore_certificate() {
    [ ${GENERATE_CERT} == "true" ] && return 0
    arbitration_log INFO "${TYPE}" "enter to restore the cert, the index is ${INDEX}"

    if [ ${INDEX} -eq 0 -o -z "${INDEX}" ]; then
        local upgrade_tmp="/opt/backupArb/upgrade_tmp"
    else
        local upgrade_tmp="/opt/backupArb/upgrade_tmp_${INDEX}"
    fi

    local ca_zip="${upgrade_tmp}/ca.zip"
    local zip_list=$(ls ${upgrade_tmp})
    arbitration_log INFO "${TYPE}" "the cert list: ${zip_list}"
    unzip_and_modify_permission "${ca_zip}" "${ETCD_HOME}/ca" || return 1
    rm -rf "${upgrade_tmp}"
    arbitration_log INFO "${TYPE}" "success to restore the cert, the index is ${INDEX}"
}

function create_check_file() {
    local result_code=$1
    local type=$2
    local script_name="check_${type}_certificate_result.sh"
    create_check_result_file ${result_code} "${script_name}"
}

function get_cipher_key_tmp_path() {
    if [ "${INDEX}" == 0 ]; then
        CK_TMP_PATH="/opt/backupArb/backup_cipher_key"
    else
        CK_TMP_PATH="/opt/backupArb/backup_cipher_key_${INDEX}"
    fi
}

function backup_cipher_key() {
    arbitration_log INFO "${TYPE}" "start to backup cipher key, the index is ${INDEX}"
    local cipher_path="${MONITOR_HOME}/conf/cipher"
    if [ ! -e "${cipher_path}" ]; then
        arbitration_log INFO "${TYPE}" "no need backup cipher path"
        return 0
    fi

    get_cipher_key_tmp_path
    if [ ! -d "${CK_TMP_PATH}" ]; then
        mkdir -p "${CK_TMP_PATH}" || return 1
    fi

    local backup_file="${CK_TMP_PATH}/ck.zip"
    if [ ! -f "${backup_file}" ]; then
        cd "${cipher_path}" && zip -rp "${backup_file}" ./* || return 1
    fi
    arbitration_log INFO "${TYPE}" "success to backup cipher key: ${backup_file}"
}

function restore_cipher_key() {
    arbitration_log INFO "${TYPE}" "start to restore cipher key, the index is ${INDEX}"
    get_cipher_key_tmp_path
    local backup_file="${CK_TMP_PATH}/ck.zip"
    if [ ! -f "${backup_file}" ]; then
        arbitration_log WARN "${TYPE}" "restore cipher key failed, ${backup_file} not exist"
        return 1
    fi
    unzip_and_modify_permission "${backup_file}" "${MONITOR_HOME}/conf/cipher" || return 1
    rm -rf "${CK_TMP_PATH}"
    arbitration_log INFO "${TYPE}" "success to restore cipher key"
}

function main() {
    TYPE=$1
    arbitration_log INFO "${TYPE}" "start to ${TYPE} the cert"
    init
    local env_file="/opt/arbitration_file/env.properties"
    INDEX=$(grep "export INDEX=" "${env_file}" | awk -F'=' '{print $2}')
    case ${TYPE} in
    "backup")
        backup_certificate
        backup_cipher_key
        return $?
        ;;
    "restore")
        restore_certificate
        return $?
        ;;
    "restore_ck")
        restore_cipher_key
        return $?
        ;;
    esac
    arbitration_log INFO "${TYPE}" "success to ${TYPE} the cert"
}

main $@
if [ $? -eq 0 ]; then
    $(create_check_file 100 ${TYPE})
else
    $(create_check_file 255 ${TYPE})
fi
exit $?
