#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
. ./arbitration_opertate_lib
. ./public.sh
env_ips=$(get_current_ips)

NCE_MANAGE_AGENT_IPMC_PATH="/opt/oss/manager/agent/bin"
# 最大索引为9
MAX_INDEX=9
NEW_VERSION=0
PUBLIC_IP=""
INPUT_IPS=""
CONF_IPS=""
AB_INNER_IPS=""
SAME_IPS=""
IS_UPGRADE=0

###################################################################
#
#
#    安装或升级仲裁软件中临时解压一个python路径
#
#
###################################################################
function unzip_temp_python() {
    ARBITRATION_TEMP_PATH=/opt/arbitration_file/arbitration_temp
    bash /opt/arbitration_file/get_arbitration_python.sh ${ARBITRATION_TEMP_PATH} >/dev/null
    TEMP_ARBITRATION_PYTHON_HOME=${ARBITRATION_TEMP_PATH}/arbitration_python/bin/python
}

###########################################################################################
#
#
#       老版本由于使用cron，导致/var/spool/clientmqueue爆满而升级仲裁节点操作系统失败
#       升级仲裁软件的时候，删除/var/spool/clientmqueue下的文件
#
#
###########################################################################################
function clear_arb_inode_backstage() {
    find /var/spool/clientmqueue/ -type f -exec egrep -l "arbitration" {} \; | xargs rm -f
    find /var/spool/clientmqueue/ -type f -exec egrep -l "check_IO_status" {} \; | xargs rm -f
}

function clear_arb_inode() {
    local fun_name=CLEAR_ARB_INODE
    arbitration_log INFO "${fun_name}" "start to clean inode"
    local threshold="90"
    local var_inode_percent=$(df -i | grep var$ | awk '{print int($5)}')
    if [ "${var_inode_percent}" != "" ]; then
        if [ "${var_inode_percent}" -gt "${threshold}" ]; then
            echo "The inode usage rate on the arbitration node exceeds 90%. Clearing..."
            arbitration_log INFO "${fun_name}" "The inode usage rate on the arbitration node exceeds 90%. Clearing..."
            export -f clear_arb_inode_backstage
            nohup bash -c clear_arb_inode_backstage </dev/null &>/dev/null &
            sleep 30
        fi
    fi
    arbitration_log INFO "${fun_name}" "end to clean inode"
}

###################################################################
#
#
#    初始化参数
#
#
###################################################################
function init() {
    clear_arb_inode
    local master_inner_card_ips=$1
    local slave_inner_card_ips=$2
    local third_inner_card_ip=$3
    #install or  precheck
    MODE=$4

    local master_inner_card_ip_list=($(echo ${master_inner_card_ips} | tr ',' ' '))
    local slave_inner_card_ip_list=($(echo ${slave_inner_card_ips} | tr ',' ' '))
    INPUT_IPS=$(echo "${master_inner_card_ip_list[@]} ${slave_inner_card_ip_list[@]} ${third_inner_card_ip}")
    AB_INNER_IPS=$(echo "${master_inner_card_ip_list[@]} ${slave_inner_card_ip_list[@]}")
    PUBLIC_IP=${third_inner_card_ip}

    # 解压临时python路径，安装、升级过程中使用
    unzip_temp_python
    # 更新文件权限
    chmod 400 /opt/arbitration_file/*.pyc >/dev/null 2>&1
    chmod 400 /opt/arbitration_file/*.zip >/dev/null 2>&1
}

###################################################################
#
#
#       校验内部通信ip是否
#
#
###################################################################
function check_inner_card_ip() {
    local fun_name="CHECK_CONFIG_IP"
    arbitration_log INFO "${fun_name}" "Start check configured ip."

    # 检查输入IP是否是浮动IP
    local float_ip_list=""
    ip addr | grep -w "inet ${real_inner_ip}" | egrep "eth[0-9]+:[0-9]+" >/dev/null 2>&1 && float_ip_list="${float_ip_list}${real_inner_ip}(Communication IP Address) "
    if [[ "${float_ip_list}" != "" ]]; then
        echo "The ip address ${float_ip_list}is float ip, please modify your input." >&2
        return 1
    fi

    arbitration_log INFO "${fun_name}" "Check configured ip successfully."
}

# 校验仲裁节点间的连通性
function check_arbitration_connect() {
    for node_ip in ${INPUT_IPS[@]}; do
        echo "${node_ip}" | grep ":" &>/dev/null
        if [[ $? -eq 0 ]]; then
            ping6 -c 2 ${node_ip} &>/dev/null
        else
            ping -c 2 ${node_ip} &>/dev/null
        fi
        if [[ $? -eq 0 ]]; then
            echo "Connect ${node_ip} is normal." >>/opt/arbitration_file/check_arbitration_result.sh
        else
            local check_connect_pyscript=/opt/arbitration_file/check_arbitration_connect.pyc
            ${TEMP_ARBITRATION_PYTHON_HOME} "${check_connect_pyscript}" "${node_ip}"
            if [ $? -eq 0 ]; then
                arbitration_log INFO "check_arbitration_connect" "Connect ${node_ip} is normal."
            else
                echo "${node_ip} is down" >>/opt/arbitration_file/check_arbitration_result.sh
                echo "[ERROR] Connect ${node_ip} is abnormal." >&2
                return 1
            fi
        fi
    done
}

###########################################################################
#
#
#       校验主备站点是否已安装nce版本
#
#
#
###########################################################################
function check_master_slave_already_install() {
    local fun_name="CHECK_MASTER_SALVE_ALREADY_INSTALL"
    arbitration_log INFO "${fun_name}" "Start check whether already install nce."
    real_third_inner_card_ip=$(get_real_full_ip ${PUBLIC_IP})
    if [[ $(echo "${env_ips}" | grep -iwF "${real_third_inner_card_ip}") == "" ]]; then
        arbitration_log INFO "${fun_name}" "Not third site node, need to check whether install version."
        su - ossadm -c ". $NCE_MANAGE_AGENT_IPMC_PATH/engr_profile.sh;sh $NCE_MANAGE_AGENT_IPMC_PATH/ipmc_adm -cmd statusapp" >/dev/null 2>&1
        if [[ "$?" != "0" ]]; then
            echo "The node hasnot installed nce version, you should install nce version before install arbitration software." >&2
            return 1
        fi
    else
        arbitration_log INFO "${fun_name}" "Third site node, no need to check whether install version."
    fi
}

# 检查当前节点是否已安装的仲裁软件，仅在三方站点执行
# return 0：找到了完全匹配的目录，安装和升级场景均使用此目录
# return 1：找到了部分匹配的目录，安装场景直接报错；升级场景继续往下走
# return 2：未找到匹配的目录，安装场景直接新增目录；升级场景直接报错
# return 3: 无目录
function check_installed() {
    ls /opt | grep -E "^arbitration-etcd$|^arbitration-etcd_[1-9]$" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "0"
        return 3
    fi
    # 遍历环境上的仲裁软件，判断是否是已安装场景
    for check_installed_item in $(ls /opt); do
        echo "${check_installed_item}" | grep -E "^arbitration-monitor$|^arbitration-monitor_[1-9]$" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            aam_properties=/opt/"${check_installed_item}"/conf/aam.properties
            if [ -f "${aam_properties}" ]; then
                NEW_VERSION=1
            else
                NEW_VERSION
                continue
            fi
            current_index=$(get_index_from_path ${check_installed_item})
            aam_properties=$(get_config_file ${current_index})
            if [ ! -f "${aam_properties}" ]; then
                continue
            fi
            
            echo "${aam_properties}" | grep -q "arbitration-etcd"
            if [ $? -eq 0 ]; then
                get_ip_from_etcd_conf "${aam_properties}"
            else
                get_ip_from_aam_conf "${aam_properties}"
            fi
            check_ip_include
            result=$?

            if [ ${result} -eq 0 ]; then
                echo "${current_index}"
                return 0
            elif [ ${result} -eq 1 ]; then
                echo "Internal IP \"${SAME_IPS}\" has been used in \"${CONF_IPS}\"." >&2
                arbitration_log INFO "Internal IP has been used in \"${CONF_IPS}\"."
                return 1
            elif [ ${result} -eq 2 ]; then
                continue
            elif [ ${result} -eq 3 ]; then
                # 安装失败的INDEX，需要判断是否有IP完全一致的INDEX
                if [ -z ${temp_index} ]; then
                    local temp_index=${current_index}
                fi
            fi
        fi
    done
    if [ ! -z ${temp_index} ]; then
        echo ${temp_index}
        return 0
    fi
    return 2
}

function get_config_file() {
    local index=$1
    if [ ${index} -eq 0 ]; then
        local etcd_path="arbitration-etcd"
        local monitor_path="arbitration-monitor"
    else
        local etcd_path="arbitration-etcd_${index}"
        local monitor_path="arbitration-monitor_${index}"
    fi
    if [ -f "/opt/${etcd_path}/bin/etcd" ]; then
        local aam_properties=/opt/"${etcd_path}"/conf/etcd_server.yaml
        if [ ! -f "${aam_properties}" ]; then
            aam_properties=/opt/"${etcd_path}"/conf/etcd_server.properties
        fi
    else
        local aam_properties=/opt/"${monitor_path}"/conf/aam.properties
    fi
    echo ${aam_properties}
}

# 根据etcd安装路径获取仲裁软件索引
function get_index_from_path() {
    path=$1
    echo "${path}" | grep "_" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "0"
        return 0
    fi
    index=$(echo "${path}" | sed "s/arbitration-monitor_//g")
    echo "${index}"
    return 0
}

# 获取下一个可以安装的目录
# 0: 已找到可以安装的目录INDEX，回显为待安装的index
# 1: 仲裁软件已满，无法再次安装
function get_install_index() {
    for get_install_index_item in $(seq 0 ${MAX_INDEX}); do
        if [ ${get_install_index_item} -eq 0 ]; then
            index=
        else
            index="_${get_install_index_item}"
        fi

        if [ ! -d /opt/arbitration-etcd${index} ]; then
            echo "${get_install_index_item}"
            return 0
        fi
    done
    echo "Already install $(expr ${MAX_INDEX} + 1) arbitration softwares." >&2
    return 1
}

# 初始化将要安装的仲裁软件的索引。NCE端的仲裁软件索引始终为0
function init_index() {
    real_third_inner_card_ip=$(get_real_full_ip "${PUBLIC_IP}")
    if [[ $(echo "${env_ips}" | grep -iwF "${real_third_inner_card_ip}") == "" ]]; then
        arbitration_log INFO "Not public node, index=0"
        INDEX=0
        return 0
    fi
    # 全新安装场景，INDEX=0
    if [ -f /opt/arbitration_file/new ]; then
        INDEX=0
        return 0
    fi
    INDEX=$(check_installed)
    result=$?
    if [ ${result} -eq 1 ]; then
        # 校验失败，退出
        echo "${INDEX}"
        return 1
    elif [ ${result} -eq 2 ]; then
        INDEX=$(get_install_index)
        result=$?
        if [ ${result} -ne 0 ]; then
            echo "${INDEX}"
            return 1
        fi
    fi
    return 0
}

# 判断配置文件的IP和传入的IP是否一致
# return 0 完全一致
# return 1 不完全一致
# return 2 完全不一致
# return 3 无效路径
function check_ip_include() {
    if [ "${CONF_IPS}" = "" ]; then
        return 2
    fi
    echo "${CONF_IPS}" | grep "\*.\*.\*.\*" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        # 安装失败的仲裁软件，直接使用
        arbitration_log INFO "installed failed path"
        return 3
    fi
    ip_count=0
    include_ip_count=0
    for input_ip in ${AB_INNER_IPS[@]}; do
        echo "${CONF_IPS}" | grep -wF "${input_ip}" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            include_ip_count=$(expr ${include_ip_count} + 1)
            SAME_IPS=$(echo "${SAME_IPS} ${input_ip}")
        fi
        ip_count=$(expr ${ip_count} + 1)
    done
    if [ ${include_ip_count} -eq 0 ]; then
        # 都不一致
        arbitration_log INFO "${CONF_IPS} totally different"
        return 2
    elif [ ${ip_count} -eq ${include_ip_count} ]; then
        arbitration_log INFO "${CONF_IPS} totally same"
        # 完全一致
        return 0
    else
        arbitration_log INFO "${CONF_IPS} partially same"
        # 不完全一致
        return 1
    fi
}

# 获取IP地址，区分Ipv4和ipv6
function split_ip_and_port() {
    local ip_from_aam_conf_item=$1
    echo ${ip_from_aam_conf_item} | grep '\[' >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "${ip_from_aam_conf_item}" | awk -F'//' '{print $2}' | awk -F':' '{print $1}'
        return
    fi
    local ip_and_port=$(echo "${ip_from_aam_conf_item}" | awk -F'//' '{print $2}')
    local port=$(echo "${ip_from_aam_conf_item}" | awk -F':' '{print $NF}')
    port=":$port"
    echo "${ip_and_port}" | awk -F"$port" '{print $1}'
}

# 从配置文件中获取仲裁软件使用到的IP
function get_ip_from_aam_conf() {
    conf_file=$1

    local local_etcd_servers=$(cat ${conf_file} | grep "aam.local.etcdservers" | awk -F"=" '{print $2}')
    local remote_etcd_servers=$(cat ${conf_file} | grep "aam.remote.1.etcdservers" | awk -F"=" '{print $2}')
    local third_etcd_servers=$(cat ${conf_file} | grep "aam.public.etcdservers" | awk -F"=" '{print $2}')

    local_etcd_servers=$(echo "${local_etcd_servers}" | sed -E 's/:[0-9]+$//g' | sed -E 's/\[//g' | sed -E 's/\]//g')
    remote_etcd_servers=$(echo "${remote_etcd_servers}" | sed -E 's/:[0-9]+$//g' | sed -E 's/:[0-9]+,/,/g' | sed -E 's/\[//g' | sed -E 's/\]//g')
    third_etcd_servers=$(echo "${third_etcd_servers}" | sed -E 's/:[0-9]+$//g' | sed -E 's/\[//g' | sed -E 's/\]//g')

    local current_dc=$(cat "${conf_file}" | grep "aam.local.dcserial" | awk -F"=" '{print $2}')
    if [ "${current_dc}" == "public" ]; then
        CONF_IPS="${local_etcd_servers},${remote_etcd_servers}"
    else
        CONF_IPS="${local_etcd_servers},${remote_etcd_servers},${third_etcd_servers}"
    fi

    return 0
}

# 适配从老版本配置文件中获取仲裁软件使用到的IP
function get_ip_from_etcd_conf() {
    local conf_file=$1
    local ips=""

    echo "${conf_file}" | grep -q 'yaml$'
    if [ $? -ne 0 ]; then
        local arr_conf_ips=$(cat ${conf_file} | grep 'initial_cluster=' | sed 's/initial_cluster=//g' | sed 's/,/ /g')
    else
        local arr_conf_ips=$(cat ${conf_file} | grep 'initial-cluster:' | sed 's/initial_cluster://g' | sed 's/ //g' | sed 's/,/ /g')
    fi
    for get_ip_from_etcd_conf_item in ${arr_conf_ips[@]}
    do
        local temp_ip=$(split_ip_and_port "${get_ip_from_etcd_conf_item}")
        ips=$(echo "${ips} ${temp_ip}")
    done
    CONF_IPS="${ips}"
}

# 写入配置文件
function write_properties() {
    path=/opt/arbitration_file/env.properties
    rm -rf ${path}
    echo "export APP_ROOT=${COMMON_HOME}" >>${path}
    echo "export ETCD_HOME=${ETCD_HOME}" >>${path}
    echo "export MONITOR_HOME=${MONITOR_HOME}" >>${path}
    echo "export COMMON_HOME=${COMMON_HOME}" >>${path}
    echo "export ETCD_LOG=${ETCD_LOG}" >>${path}
    echo "export MONITOR_LOG=${MONITOR_LOG}" >>${path}
    echo "export INDEX=${INDEX}" >>${path}
    echo "export ARBITRATION_HOME=${ARBITRATION_HOME}" >>${path}
    echo "export ARBITRATION_INSTALL_LOG=${ARBITRATION_INSTALL_LOG}" >>${path}
    echo "export ARBITTRATION_WATCHDOG_LOG=${ARBITTRATION_WATCHDOG_LOG}" >>${path}
    echo "export ARBITRATION_LOG=${ARBITRATION_LOG}" >>${path}
    echo "export ETCD_LOG_BASE=/var/log/arbitration-etcd" >>${path}
    echo "export MONITOR_LOG_BASE=/var/log/arbitration-monitor" >>${path}
    echo "export COMMON_LOG=${COMMON_LOG}" >>${path}
    echo "export TEMP_ARBITRATION_PYTHON_HOME=${TEMP_ARBITRATION_PYTHON_HOME}" >>${path}
    echo "export ARBITRATION_TEMP_PATH=${ARBITRATION_TEMP_PATH}" >>${path}
    echo "export LANG=en_US.UTF-8" >>${path}
    real_third_inner_card_ip=$(get_real_full_ip "${PUBLIC_IP}")
    if [[ $(echo "${env_ips}" | grep -iwF "${real_third_inner_card_ip}") != "" ]]; then
        times=$(expr ${INDEX} + ${INDEX})
        echo "export ETCD_PORT=$(expr ${times} + 2391)" >>${path}
        echo "export MONITOR_PORT=$(expr ${times} + 2390)" >>${path}
    fi
}

# 检查可用的安装索引，并写入配置文件
function check_avaliable_index() {
    init_index || return 1
    echo "Available index is ${INDEX}"
    arbitration_log INFO "Available index is ${INDEX}"
    if [ ${INDEX} -eq 0 ]; then
        ETCD_HOME=/opt/arbitration-etcd
        MONITOR_HOME=/opt/arbitration-monitor
        COMMON_HOME=/opt/arbitration-common
        ARBITRATION_HOME=/opt/arbitration
    else
        ETCD_HOME=/opt/arbitration-etcd_${INDEX}
        MONITOR_HOME=/opt/arbitration-monitor_${INDEX}
        COMMON_HOME=/opt/arbitration-common_${INDEX}
        ARBITRATION_HOME=/opt/arbitration_${INDEX}
    fi
    ARBITRATION_INSTALL_LOG=/var/log/arbitration_install/${INDEX}
    ARBITTRATION_WATCHDOG_LOG=/var/log/arbitration_watchdog/${INDEX}
    ARBITRATION_LOG=/var/log/arbitration/${INDEX}
    COMMON_LOG=/var/log/arbitration-common/${INDEX}
    MONITOR_LOG=/var/log/arbitration-monitor/${INDEX}
    ETCD_LOG=/var/log/arbitration-etcd/${INDEX}

    if [ "${MODE}" == "install" ]; then
        delete_recover_etcd
    fi
    write_properties
}

function delete_recover_etcd() {
    rm -f ${ARBITRATION_HOME}/script/aamonitor_watchdog.sh >/dev/null 2>&1
    kill -9 $(ps -efww | grep "check_etcd_health" | grep -v grep | awk '{print $2}') >/dev/null 2>&1
}

###################################################################
#       校验三方站点内部通信ip是否侦听22端口
###################################################################
function check_third_sshd_port()
{
    ifconfig | grep -w "${PUBLIC_IP}"> /dev/null 2>&1
    if [ $? -ne 0 ]; then
        arbitration_log INFO "current does not exist third ip:${PUBLIC_IP}"
        return 0
    fi
    local listen_address_list=$(grep "^\s*ListenAddress" "/etc/ssh/sshd_config" | egrep -wv "0.0.0.0|::|::1|127.0.0.1" | awk '{print $2}')
    if [ -z "${listen_address_list}" ]; then
        return 0
    fi
    echo "${listen_address_list}" | grep -w "${PUBLIC_IP}"  > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "[ERROR] Port 22 is not listening on ${PUBLIC_IP}."
        echo "[ERROR] Suggestion: Add ListenAddress ${PUBLIC_IP} to the /etc/ssh/sshd_config file and restart the sshd service."
        arbitration_log ERROR "Port 22 is not listening on ${PUBLIC_IP}"
        return 1
    fi
}

function main()
{
    init $@
    check_inner_card_ip || return 1
    check_arbitration_connect || return 1
    check_third_sshd_port || return 1
    check_master_slave_already_install || return 1
    check_avaliable_index || return 1
}

main $@
if [ $? -eq 0 ]; then
    create_check_result_file 100 "check_arbitration_config_result.sh"
else
    create_check_result_file 255 "check_arbitration_config_result.sh"
fi
exit $?
