#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.

curpath=$(cd $(dirname $0);pwd)
cd $curpath
LOG=/var/log/config_routes.log
ADDRESS=""
NETMASK=""
GATEWAY=""
IS_IPV6=1

function is_suse_os() {
    if [ -e /etc/SuSE-release ]; then
        return 0
    fi
    return 1
}

function check_user() {
    cur_user=$(id -un)
    [ "${cur_user}" == "root" ] && return 0
    echo "The script must run with root user."
    exit 1
}
check_user

echo "start to execut config_routes.sh" >${LOG}

is_suse_os
if [ $? -eq 0 ]; then
    PATH_PRE="/etc/sysconfig/network/ifroute-"
    PATH_PRE_IPV6="/etc/sysconfig/network/ifroute6-"
else
    PATH_PRE="/etc/sysconfig/network-scripts/route-"
    PATH_PRE_IPV6="/etc/sysconfig/network-scripts/route6-"
fi

#########################################
# 是当前修改的路由生效
#########################################
function enable() {
    read -p "Whether the configuration takes effect immediately  y/n: " yn1
    case "${yn1}" in
    Y | y)
        read -p "To make the configuration take effect immediately, the network service needs to be restarted, which may cause network interruption  y/n: " yn2
        case "${yn2}" in
        Y | y)
            systemctl restart network.service >>${LOG} 2>&1
            if [ $? -eq 0 ]; then
                echo "Enable successfully." | tee ${LOG}
            else
                echo "Enable failed." | tee ${LOG}
            fi
            ;;
        *)
            echo "The configured route does not take effect immediately. For details about how to take effect, see the guide"
            ;;
        esac
        ;;
    *)
        echo "The configured route does not take effect immediately. For details about how to take effect, see the guide"
        ;;
    esac
}

#########################################
# 检查IP格式是否为IPv4
#########################################
function check_ipv4() {
    IP=$1
    if [[ "$IP" =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
        FIELD1=$(echo "$IP" | cut -d. -f1)
        FIELD2=$(echo "$IP" | cut -d. -f2)
        FIELD3=$(echo "$IP" | cut -d. -f3)
        FIELD4=$(echo "$IP" | cut -d. -f4)
        if [ "$FIELD1" -le 255 -a "$FIELD2" -le 255 -a "$FIELD3" -le 255 -a "$FIELD4" -le 255 ]; then
            return 0
        fi
    fi
    return 1
}

#########################################
# 检查IP格式是否为IPv6
#########################################
function check_ipv6() {
    local ip=$1
    if [[ "${ip}" =~ ^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$ ]]; then
        return 0
    else
        echo "Invalid IP Address"
        return 1
    fi
}

#########################################
# 检查掩码是否满足IPv6格式
#########################################
function check_netmask_ipv6() {
    local netmask=$1

    egrep -q '^[0-9]+$' <<<"${netmask}" >/dev/null
    if [ $? -ne 0 ]; then
        echo "Warning, The subnet mask must be an integer number."
        return 1
    fi

    if [ "${netmask}" -lt 0 -o "${netmask}" -gt 128 ]; then
        echo "Warning, The subnet mask number ranges is (0 - 128)."
        return 1
    fi
}

function check_routefile() {
    routefile=$1
    if [[ "${routefile}" =~ ^[a-zA-Z0-9]+$ ]]; then
        ip addr | grep -v "^ " | awk '{print $2}' | grep -w "${routefile}" >>${LOG}
        if [ $? -eq 0 ]; then
            return 0
        fi
    fi
    return 1
}

function keep_pope_dom() {
    local sfile=$1
    local mfile=$2
    local user_pope_dom=$(ls -l $sfile | awk '{print $1}' | cut -c 2-4 | sed 's/-//g')
    local group_pope_dom=$(ls -l $sfile | awk '{print $1}' | cut -c 5-7 | sed 's/-//g')
    local other_pope_dom=$(ls -l $sfile | awk '{print $1}' | cut -c 8-10 | sed 's/-//g')
    mv -f $mfile $sfile
    chmod u+${user_pope_dom} $sfile
    chmod g+${group_pope_dom} $sfile
    chmod o+${other_pope_dom} $sfile
}

function sort_route_index() {
    local route_filename="$1"
    echo "run sort_route_index start" >>${LOG}
    cat "${route_filename}" >>"${LOG}"
    route_filename_bck="${route_filename}_bck"
    local index=0
    local count=1
    if [ -f "${route_filename_bck}" ]; then
        rm -rf "${route_filename_bck}"
    fi
    touch "${route_filename_bck}"
    cat "${route_filename}" | while read line || [[ -n "${line}" ]]; do
        local flag=0
        item=$(echo "${line}" | awk -F"=" '{print $1}')
        if [[ "$item" =~ ^ADDRESS[0-9]+ ]]; then
            local item="ADDRESS"
            local flag=1
        elif [[ "$item" =~ ^NETMASK[0-9]+ ]]; then
            local item="NETMASK"
            local flag=1
        elif [[ "$item" =~ ^GATEWAY[0-9]+ ]]; then
            local item="GATEWAY"
            local flag=1
        fi
        if [[ "$flag" == 1 ]]; then
            echo "${line}" | sed "s/^${item}[0-9]\+/${item}${index}/g" >>${route_filename_bck}
            if [[ "${count}" == 3 ]]; then
                local count=1
                local index=$((${index} + 1))
            else
                local count=$((${count} + 1))
            fi
        else
            echo "${line}" >>${route_filename_bck}
        fi
    done
    keep_pope_dom "${route_filename}" "${route_filename_bck}"
    echo "run sort_route_index end" >>${LOG}
}

#########################################
# 增加路由入口
#########################################
function add_routes() {
    echo ${DEVIDE}
    echo -n "${MY_TAB}Enter the name of the network port for which you want to add a route: "
    read routefile
    while true; do
        check_routefile ${routefile}
        if [ $? -eq 0 ]; then
            break
        else
            echo "${MY_TAB}Enter a correct network port name "
            echo -n "${MY_TAB}Enter the name of the network port for which you want to add a route: "
            read routefile
        fi
    done

    read_address
    read_netmask
    read_gateway

    echo "ADDRESS=${ADDRESS} NETMASK=${NETMASK} GATEWAY=${GATEWAY}" >>${LOG}
    is_suse_os
    if [ $? -eq 0 ]; then
        add_route_suse "${ADDRESS}" "${NETMASK}" "${GATEWAY}" "${routefile}"
    else
        add_route_other "${ADDRESS}" "${NETMASK}" "${GATEWAY}" "${routefile}"
    fi
}

#########################################
# 读取目的IP地址
#########################################
function read_address() {
    echo -n "${MY_TAB}Enter the destination network: "
    read ADDRESS
    while true; do
        check_ipv4 ${ADDRESS}
        if [ $? -eq 0 ]; then
            break
            IS_IPV6=1
        else
            check_ipv6 ${ADDRESS}
            if [ $? -eq 0 ]; then
                IS_IPV6=0
                break
            else
                echo "${MY_TAB}Enter a correct destination network "
                echo -n "${MY_TAB}Enter the destination network: "
                read ADDRESS
            fi
        fi
    done
}

#########################################
# 读取目的掩码
#########################################
function read_netmask() {
    echo -n "${MY_TAB}Enter the destination mask: "
    read NETMASK
    while true; do
        if [ ${IS_IPV6} -eq 0 ]; then
            check_netmask_ipv6 ${NETMASK}
            if [ $? -eq 0 ]; then
                break
            else
                echo "${MY_TAB}Current route configuration type is IPv6. Enter a correct destination mask "
                echo -n "${MY_TAB}Current route configuration type is IPv6. Enter the destination mask: "
                read NETMASK
            fi
        else
            check_ipv4 ${NETMASK}
            if [ $? -eq 0 ]; then
                break
            else
                echo "${MY_TAB}Current route configuration type is IPv4. Enter a correct destination mask "
                echo -n "${MY_TAB}Current route configuration type is IPv4. Enter the destination mask: "
                read NETMASK
            fi
        fi
    done
}

#########################################
# 读取目的网关
#########################################
function read_gateway() {
    echo -n "${MY_TAB}Enter the gateway address: "
    read GATEWAY
    while true; do
        if [ ${IS_IPV6} -eq 0 ]; then
            check_ipv6 ${GATEWAY}
            if [ $? -eq 0 ]; then
                break
            else
                echo "${MY_TAB}Current route configuration type is IPv6. Enter a correct gateway address "
                echo -n "${MY_TAB}Current route configuration type is IPv6. Enter the gateway address: "
                read GATEWAY
            fi
        else
            check_ipv4 ${GATEWAY}
            if [ $? -eq 0 ]; then
                break
            else
                echo "${MY_TAB}Current route configuration type is IPv4. Enter a correct gateway address "
                echo -n "${MY_TAB}Current route configuration type is IPv4. Enter the gateway address: "
                read GATEWAY
            fi
        fi
    done
}

#########################################
# SuSE场景下增加路由配置
#########################################
function add_route_suse() {
    local address=$1
    local netmask=$2
    local gateway=$3
    local route_port=$4
    local route_filename="${PATH_PRE}${route_port}"
    if [ ${IS_IPV6} -eq 0 ]; then
        route_filename="${PATH_PRE_IPV6}${route_port}"
    fi
    echo "route_filename=${route_filename}" >>${LOG}
    echo "${address} ${gateway} ${netmask} ${route_port}" >>${route_filename}
    echo "Configure ${route_filename} successfully." | tee ${LOG}
}

#########################################
# Euler场景下增加路由配置
#########################################
function add_route_other() {
    local address=$1
    local netmask=$2
    local gateway=$3
    local route_port=$4
    if [ ${IS_IPV6} -eq 0 ]; then
        add_route_other_ipv6 "${address}" "${netmask}" "${gateway}" "${route_port}"
        return
    fi
    local route_filename="${PATH_PRE}${route_port}"
    if [ -f "${route_filename}" ]; then
        index=$(cat "${route_filename}" | grep ADDRESS | awk -F"=" '{print $1}' | awk -F"ADDRESS" '{print $2}' |
            awk 'BEGIN {max = -1} {if ($1>max) max=$1 fi} END {print max+1}')
        echo "ADDRESS${index}=${address}" >>"${route_filename}"
        echo "NETMASK${index}=${netmask}" >>"${route_filename}"
        echo "GATEWAY${index}=${gateway}" >>"${route_filename}"
        if [[ "${index}" != 0 ]]; then
            sort_route_index "${route_filename}"
        fi
        echo "Configure ${route_filename} successfully." | tee ${LOG}
    else
        index=0
        echo "ADDRESS${index}=${address}" >"${route_filename}"
        echo "NETMASK${index}=${netmask}" >>"${route_filename}"
        echo "GATEWAY${index}=${gateway}" >>"${route_filename}"
        echo "Configure ${route_filename} successfully." | tee ${LOG}
    fi
}

#########################################
# Euler场景下增加ipv6路由配置
#########################################
function add_route_other_ipv6() {
    local address=$1
    local netmask=$2
    local gateway=$3
    local route_port=$4
    local route_filename="${PATH_PRE_IPV6}${route_port}"
    echo "${address}/${netmask} via ${gateway} dev ${route_port}" >>"${route_filename}"
    echo "Configure ${route_port} successfully." | tee ${LOG}
}

function delete_routes() {
    echo "${DEVIDE}"
    echo -n "${MY_TAB}Enter the name of the network port from which you want to delete a route: "
    read route_file
    while true; do
        if [[ "${route_file}" =~ ^bond[0-9]+ || "${route_file}" =~ ^eth[0-9]+ ]]; then
            break
        else
            echo "${MY_TAB}Enter a correct network port name "
            echo -n "${MY_TAB}Enter the name of the network port from which you want to delete a route: "
            read route_file
        fi
    done

    is_suse_os
    if [ $? -eq 0 ]; then
        delete_route_suse "${route_file}"
    else
        delete_route_other "${route_file}"
    fi
}

##################################################
# 删除suse系统路由
##################################################
function delete_route_suse() {
    local route_file=$1
    local route_filename="${PATH_PRE}${route_file}"
    local route_file_name_ipv6="${PATH_PRE_IPV6}${route_file}"
    if [ ! -f "${route_filename}" -o ! -s "${route_filename}" ] && [ ! -f "${route_file_name_ipv6}" -o ! -s "${route_file_name_ipv6}" ]; then
        echo "${MY_TAB}No route can be deleted from the network port "
        start
        return
    fi
    local index=1
    arr=()
    if [ -f "${route_filename}" -a -s "${route_filename}" ]; then
        while read line; do
            echo "${line// /}" | grep -E "^#" >/dev/null
            if [ $? -eq 0 ]; then
                continue
            fi
            local address=$(echo "${line}" | awk '{print $1}')
            local netmask=$(echo "${line}" | awk '{print $3}')
            local gateway=$(echo "${line}" | awk '{print $2}')
            arr[${index}]="${line}"
            echo "${MY_TAB}${index}  ADDRESS: ${address} NETMASK: ${netmask} GATEWAY: ${gateway}"
            let index++
        done <${route_filename}
    fi
    if [ -f "${route_file_name_ipv6}" -a -s "${route_file_name_ipv6}" ]; then
        while read line; do
            echo "${line// /}" | grep -E "^#" >/dev/null
            if [ $? -eq 0 ]; then
                continue
            fi
            local address=$(echo "${line}" | awk '{print $1}')
            local netmask=$(echo "${line}" | awk '{print $3}')
            local gateway=$(echo "${line}" | awk '{print $2}')
            arr[${index}]="${line}"
            echo "${MY_TAB}${index}  ADDRESS: ${address} NETMASK: ${netmask} GATEWAY: ${gateway}"
            let index++
        done <"${route_file_name_ipv6}"
    fi
    # 获取界面输入路由索引
    read_select "${index}"
    local select=$?
    local del_route_line=${arr[${select}]}
    echo "${del_route_line}" | grep ":" &>/dev/null
    if [ $? -eq 0 ]; then
        route_filename=${route_file_name_ipv6}
    fi
    sed -i "/${del_route_line}/d" ${route_filename} >>${LOG}
    grep "${del_route_line}" ${route_filename} &>/dev/null
    if [ $? -eq 0 ]; then
        echo "Failed to delete the route" | tee ${LOG}
        exit 1
    else
        echo "Route deleted successfully" | tee ${LOG}
    fi
}

##################################################
# 删除欧拉系统路由
##################################################
function delete_route_other() {
    local route_file=$1
    local route_filename="${PATH_PRE}${route_file}"
    local route_file_name_ipv6="${PATH_PRE_IPV6}${route_file}"
    if [ ! -f "${route_filename}" -o ! -s "${route_filename}" ] && [ ! -f "${route_file_name_ipv6}" -o ! -s "${route_file_name_ipv6}" ]; then
        echo "${MY_TAB}No route can be deleted from the network port "
        start
        return
    fi
    index=1
    arr=()
    # 处理ipv4路由
    if [ -f "${route_filename}" -a -s "${route_filename}" ]; then
        counts=$(cat "${route_filename}" | grep ADDRESS | awk -F"=" '{print $1}' | awk -F"ADDRESS" '{print $2}')
        for count in ${counts}; do
            arr[${index}]=${count}
            local address=$(cat "${route_filename}" | grep "ADDRESS${count}" | awk -F"=" '{print $2}')
            local netmask=$(cat "${route_filename}" | grep "NETMASK${count}" | awk -F"=" '{print $2}')
            local gateway=$(cat "${route_filename}" | grep "GATEWAY${count}" | awk -F"=" '{print $2}')
            echo "${MY_TAB}${index}  ADDRESS: ${address} NETMASK: ${netmask} GATEWAY: ${gateway}"
            let index++
        done
    fi
    # 处理ipv6路由
    if [ -f "${route_file_name_ipv6}" -a -s "${route_file_name_ipv6}" ]; then
        while read line; do
            echo "${line// /}" | grep -E "^#" >/dev/null
            if [ $? -eq 0 ]; then
                continue
            fi
            local address=$(echo "${line}" | awk '{print $1}' | awk -F"/" '{print $1}')
            local netmask=$(echo "${line}" | awk '{print $1}' | awk -F"/" '{print $2}')
            local gateway=$(echo "${line}" | awk '{print $3}')
            arr[${index}]="${line}"
            echo "${MY_TAB}${index}  ADDRESS: ${address} NETMASK: ${netmask} GATEWAY: ${gateway}"
            let index++
        done <"${route_file_name_ipv6}"
    fi
    # 获取界面输入路由索引
    read_select "${index}"
    local select=$?
    local del_route_line=${arr[${select}]}
    # 数字索引值IPv4，非数字Ipv6
    if [[ "${del_route_line}" =~ ^[0-9]{1,2}$ ]]; then
        delete_other_ipv4 "${del_route_line}" "${route_filename}"
    else
        delete_other_ipv6 "${del_route_line}" "${route_file_name_ipv6}"
    fi
}

##################################################
# 获取界面输入路由索引
##################################################
function read_select() {
    local index=$1
    # 获取界面输入路由索引
    echo -n "${MY_TAB}Enter the serial number of the route to be deleted: "
    read select
    while true; do
        if [[ "${select}" =~ ^[1-9]{1,2}$ ]]; then
            if [ "${select}" -ge 1 -a "${select}" -lt "${index}" ]; then
                break
            else
                echo "${MY_TAB}Enter a correct serial number"
                echo -n "${MY_TAB}Enter the serial number of the route to be deleted: "
                read select
            fi
        else
            echo "${MY_TAB}Enter a correct serial number "
            echo -n "${MY_TAB}Enter the serial number of the route to be deleted: "
            read select
        fi
    done
    return ${select}
}

##################################################
# 删除欧拉IPv4路由
##################################################
function delete_other_ipv4() {
    local del_route_line="$1"
    local route_filename="$2"
    sed -i "/^ADDRESS${del_route_line}=.*/d" ${route_filename} >>${LOG}
    sed -i "/^NETMASK${del_route_line}=.*/d" ${route_filename} >>${LOG}
    sed -i "/^GATEWAY${del_route_line}=.*/d" ${route_filename} >>${LOG}
    local del_address=$(cat ${route_filename} | egrep "^ADDRESS${del_route_line}=.*$")
    local del_nermask=$(cat ${route_filename} | egrep "^NETMASK${del_route_line}=.*$")
    local del_gateway=$(cat ${route_filename} | egrep "^GATEWAY${del_route_line}=.*$")
    if [ -n "${del_address}" -o -n "${del_nermask}" -o -n "${del_gateway}" ]; then
        echo "Failed to delete the route" | tee ${LOG}
        exit 1
    else
        sort_route_index "${route_filename}"
        echo "Route deleted successfully" | tee ${LOG}
    fi
}

##################################################
# 删除欧拉IPv4路由
##################################################
function delete_other_ipv6() {
    local del_route_line="$1"
    local route_file_name_ipv6="$2"
    # 掩码/转义
    del_route_line=${del_route_line/\//\\/}
    sed -i "/${del_route_line}/d" ${route_file_name_ipv6} >>${LOG}
    grep "${del_route_line}" ${route_file_name_ipv6} &>/dev/null
    if [ $? -eq 0 ]; then
        echo "Failed to delete the route" | tee ${LOG}
        exit 1
    else
        echo "Route deleted successfully" | tee ${LOG}
    fi
}

function start() {
    echo ${DEVIDE}
    echo "${MY_TAB}1: Add routes"
    echo "${MY_TAB}2: Delete routes"
    echo "${MY_TAB}3: Exit"
    echo -n "${MY_TAB}Select a task: "
    read choice
    while [[ "$choice" != 1 && "$choice" != 2 && "$choice" != 3 ]]; do
        echo "${MY_TAB}Incorrect input. Enter a value again."
        start
    done
    if [[ "$choice" == 1 ]]; then
        add_routes
        enable
        start
    elif [[ "$choice" == 2 ]]; then
        delete_routes
        enable
        start
    elif [[ "$choice" == 3 ]]; then
        exit 0
    fi
}

MY_TAB="        "
DEVIDE="------------------------------------"
start
