#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
cur_path=$(cd $(dirname $0);pwd)
RETURN_AUTH_FAILED=-1
RETURN_SUCCESS=0
RETURN_ERROR=1

TIME_STAMP=$(date '+%s')
BASH_PID=$$

#################################
# 初始化日志
#################################
function init_log() {
    # 预装使用，默认环境未安装仲裁
    LOG_PATH=/var/log/arbitration/0
    if [ ! -d ${LOG_PATH} ]; then
        mkdir -p ${LOG_PATH}
    fi
    LOG_FILE="${LOG_PATH}/config_third_node.log"
}

#################################
# 加载环境变量
#################################
function init_env() {
    # 加载系统环境变量
    . /usr/local/ostools/bond_tools/get_os_info.sh >>"${LOG_FILE}" 2>&1
    PATH_PERFIX=""
    PATH_ROUTE_PERFIX=""
    if [ "${linux_type}" = "SUSE" ]; then
        PATH_PERFIX="/etc/sysconfig/network/ifcfg-"
        PATH_PER="/etc/sysconfig/network/"
        PATH_ROUTE_PERFIX="/etc/sysconfig/network/ifroute-"
    elif [ "${linux_type}" = "CentOS" -o ${linux_type} = "Redhat" -o ${linux_type} = "EulerOS" ]; then
        PATH_PERFIX="/etc/sysconfig/network-scripts/ifcfg-"
        PATH_PER="/etc/sysconfig/network-scripts/"
        PATH_ROUTE_PERFIX="/etc/sysconfig/network-scripts/route-"
    fi
}

#################################
# 脚本退出
#################################
function do_exit() {
    local errorCode="$1"
    local logType="$2"
    local logInfo="$3"

    echo "${logInfo}"
    write_logger "${logType}" "${logInfo}"
    exit ${errorCode}
}

#################################
# 日志记录
#################################
function write_logger() {
    local logType="$1"
    local logInfo="$2"

    echo ${logType} | grep -i "ERROR" &>/dev/null && Level="ERROR"
    echo ${logType} | grep -i "WARN" &>/dev/null && Level="WARN"
    [ -z "${Level}" ] && Level="INFO"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${BASH_PID}] | ${Level} ${logInfo}" >>"${LOG_FILE}"
}

#################################
# 获取参数
#################################
function get_params() {
    for i in {1..14}; do
        [[ ${1#-} == "login_ip" ]] && {
            LOGIN_IP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "login_nic" ]] && {
            LOGIN_NIC=$2
            shift 2
            continue
        }
        [[ ${1#-} == "login_mask" ]] && {
            LOGIN_MASK=$2
            shift 2
            continue
        }
        [[ ${1#-} == "login_desk_ip" ]] && {
            LOGIN_DESK_IP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "login_desk_mask" ]] && {
            LOGIN_DESK_MASK=$2
            shift 2
            continue
        }
        [[ ${1#-} == "login_gateway" ]] && {
            LOGIN_GATEWAY=$2
            shift 2
            continue
        }
        [[ ${1#-} == "inner_ip" ]] && {
            INNER_IP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "inner_nic" ]] && {
            INNER_NIC=$2
            shift 2
            continue
        }
        [[ ${1#-} == "inner_mask" ]] && {
            INNER_MASK=$2
            shift 2
            continue
        }
        [[ ${1#-} == "inner_desk_ip" ]] && {
            INNER_DESK_IP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "inner_desk_mask" ]] && {
            INNER_DESK_MASK=$2
            shift 2
            continue
        }
        [[ ${1#-} == "inner_gateway" ]] && {
            INNER_GATEWAY=$2
            shift 2
            continue
        }
        [[ ${1#-} == "ntp_time_ip" ]] && {
            NTP_TIME_IP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "time_zone" ]] && {
            TIME_ZONE=$2
            shift 2
            continue
        }
    done
    [ -z "${LOGIN_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_NIC}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_DESK_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${LOGIN_DESK_IP}" = "None" ] && LOGIN_DESK_IP=$(get_desk_ip_default ${LOGIN_IP})
    [ -z "${LOGIN_DESK_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${LOGIN_DESK_MASK}" = "None" ] && LOGIN_DESK_MASK=$(get_desk_mask_default ${LOGIN_IP})
    [ -z "${LOGIN_GATEWAY}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_NIC}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_DESK_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${INNER_DESK_IP}" = "None" ] && INNER_DESK_IP=$(get_desk_ip_default ${INNER_IP})
    [ -z "${INNER_DESK_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${INNER_DESK_MASK}" = "None" ] && INNER_DESK_MASK=$(get_desk_mask_default ${INNER_IP})
    [ -z "${INNER_GATEWAY}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${NTP_TIME_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${TIME_ZONE}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
}

#################################
# 获取默认目标ip
#################################
function get_desk_ip_default() {
    local ip=$1
    echo "${ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        echo "::"
    else
        echo "0.0.0.0"
    fi
}

#################################
# 获取默认目标掩码
#################################
function get_desk_mask_default() {
    local ip=$1
    echo "${ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        echo "32"
    else
        echo "0.0.0.0"
    fi
}

function config_third_sys() {
    backup_config
    modify_ip "${LOGIN_NIC}" "${LOGIN_IP}" "${LOGIN_MASK}"
    modify_route "${LOGIN_NIC}" "${LOGIN_GATEWAY}" "${LOGIN_DESK_IP}" "${LOGIN_DESK_MASK}"
    if [ "${LOGIN_NIC}" != "${INNER_NIC}" ]; then
        modify_ip "${INNER_NIC}" "${INNER_IP}" "${INNER_MASK}"
        modify_route "${INNER_NIC}" "${INNER_GATEWAY}" "${INNER_DESK_IP}" "${INNER_DESK_MASK}"
    fi

    bash /usr/local/ostools/bond_tools/config_netcard.sh "enable" >>"${LOG_FILE}" 2>&1
    restart_sshd
}

#################################
# 修改主机名
#################################
function config_third_host() {
    hostnamectl set-hostname "third-party-site" >>"${LOG_FILE}" 2>&1
    if [ $? -ne 0 ]; then
        do_exit ${RETURN_ERROR} "ERROR" "modify hostname failed."
    fi
}

#################################
# 修改ip
#################################
function modify_ip() {
    local nic=$1
    local ip_addr=$2
    local netmask=$3
    local ip_type=$(get_ip_type ${ip_addr})
    local ip_name="IPADDR"
    if [ "${linux_type}" = "SUSE" ]; then
        [[ "${ip_type}" == "IPv6" ]] && ip_name="IPADDR6"
    elif [ "${linux_type}" = "CentOS" -o "${linux_type}" = "Redhat" -o "${linux_type}" = "EulerOS" ]; then
        [[ "${ip_type}" == "IPv6" ]] && ip_name="IPV6ADDR"
    fi

    ip_config_file=${PATH_PERFIX}${nic}
    if [ ! -f ${ip_config_file} ]; then
        do_exit ${RETURN_AUTH_FAILED} "ERROR" "The arbitration config file ${file_name} does not exist, exit."
    fi
    old_ip=$(cat ${ip_config_file} | grep ${ip_name} | awk -F"=" '{print $2}' | awk -F"/" '{print $1}')
    old_ip=${old_ip//\'/}
    local ret=0
    if [ "${ip_type}" = "IPv6" ]; then
        sed -i "/^NETMASK=/d" "${ip_config_file}" || let ret=ret+1
        sed -i "s#^IPADDR=.*\|^${ip_name}=.*#${ip_name}=${ip_addr}/${netmask}#g" "${ip_config_file}" || let ret=ret+1
        enable_ipv6
    else
        sed -i "s#^${ip_name}.*#${ip_name}='${ip_addr}'#g" $ip_config_file 2>/dev/null || let ret=ret+1
        sed -i "s#^NETMASK.*#NETMASK='${netmask}'#g" $ip_config_file 2>/dev/null || let ret=ret+1
    fi
    # 防止ssh未配置监听ip情况下配置新ip监听失败
    local ssh_ip=$(cat /etc/ssh/sshd_config | grep -w ${old_ip} 2>/dev/null)
    if [ "${ssh_ip}" != "" ]; then
        sed -i "/^ListenAddress.*\<${old_ip}\>/d" /etc/ssh/sshd_config 2>/dev/null || let ret=ret+1
        #刷新旧的公钥，失败不影响
        sed -i "s#\<${old_ip}\>#${ip_addr}#g" /root/.ssh/known_hosts >>"${LOG_FILE}" 2>&1
    fi
    modify_ssh "${ip_addr}" || let ret=ret+1
    if [ ${ret} -ne 0 ]; then
        rollback_config
        do_exit ${RETURN_ERROR} "ERROR" "modify ${nic} ip address failed."
    fi
    write_logger "INFO" "modify ${nic} ip address success."
}

#################################
# 修改ssh配置,仅新增，不删除原有配置
#################################
function modify_ssh() {
    local ip_addr=$1
    local listen_address_list=$(grep "^\s*ListenAddress" "/etc/ssh/sshd_config" | egrep -wv "0.0.0.0|::|::1|127.0.0.1" | awk '{print $2}')
    if [ -z "${listen_address_list}" ]; then
        return 0
    fi
    echo ${listen_address_list} | grep -wq "${ip_addr}"
    if [ $? -eq 0 ]; then
        return 0
    fi
    local line_num=$(grep -n "^ListenAddress" /etc/ssh/sshd_config | head -n 1 | awk -F":" '{print $1}')
    sed -i "${line_num} iListenAddress ${ip_addr}" /etc/ssh/sshd_config
    return $?
}

#################################
# 获取ip类型
#################################
function get_ip_type() {
    local ip=$1
    echo "${ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        echo "IPv6"
    else
        echo "IPv4"
    fi
}

#################################
# 修改路由
#################################
function modify_route() {
    local nic=$1
    local gateway=$2
    local desk_ip=$3
    local desk_mask=$4
    if [ "${gateway}" = "None" ]; then
        write_logger "INFO" "gateway of ${nic} is None, no need to config."
        return
    fi
    echo "${desk_ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        bash /usr/local/ostools/bond_tools/config_netcard_ipv6.sh "route:${nic};${gateway},${desk_ip},${desk_mask}"
    else
        bash /usr/local/ostools/bond_tools/config_netcard.sh "route:${nic};${gateway},${desk_ip},${desk_mask}"
    fi
    if [ $? -ne 0 ]; then
        rollback_config
        do_exit ${RETURN_ERROR} "ERROR" "modify ${nic} route failed."
    fi
    write_logger "INFO" "modify ${nic} route success."
}

#################################
# 备份网络配置文件
#################################
function backup_config() {
    login_ip_file="${PATH_PERFIX}${LOGIN_NIC}_${TIME_STAMP}"
    copy_file "${PATH_PERFIX}${LOGIN_NIC}" "${login_ip_file}"

    login_route_file="${PATH_ROUTE_PERFIX}${LOGIN_NIC}_${TIME_STAMP}"
    copy_file "${PATH_ROUTE_PERFIX}${LOGIN_NIC}" "${login_route_file}"

    if [ "${LOGIN_NIC}" = "${INNER_NIC}" ]; then
        return
    fi
    inner_ip_file="${PATH_PERFIX}${INNER_NIC}_${TIME_STAMP}"
    copy_file "${PATH_PERFIX}${INNER_NIC}" "${inner_ip_file}"

    inner_route_file="${PATH_ROUTE_PERFIX}${INNER_NIC}_${TIME_STAMP}"
    copy_file "${PATH_ROUTE_PERFIX}${INNER_NIC}" "${inner_route_file}"

    sshd_config_file="/etc/ssh/sshd_config_${TIME_STAMP}"
    copy_file /etc/ssh/sshd_config "${sshd_config_file}"
}

function copy_file() {
    local src_file=$1
    local dest_file=$2
    if [ -f "${src_file}" ]; then
        cp -rp "${src_file}" "${dest_file}"
    fi
}

#################################
# 恢复网络配置文件
#################################
function rollback_config() {
    move_file "${login_ip_file}" "${PATH_PERFIX}${LOGIN_NIC}"
    move_file "${login_route_file}" "${PATH_ROUTE_PERFIX}${LOGIN_NIC}"
    move_file "${inner_ip_file}" "${PATH_PERFIX}${INNER_NIC}"
    move_file "${inner_route_file}" "${PATH_ROUTE_PERFIX}${INNER_NIC}"
    move_file "${sshd_config_file}" /etc/ssh/sshd_config

    restart_network
    restart_sshd
}

function move_file() {
    local dest_file=$1
    local src_file=$2
    if [ -f "${dest_file}" ]; then
        mv -f "${dest_file}" ${src_file}
    fi
}

#################################
# 成功后清除备份文件
#################################
function clear_backup_file() {
    for file_name in "${login_ip_file}" "${login_route_file}" "${inner_ip_file}" "${inner_route_file}" "${sshd_config_file}"; do
        rm -f "${file_name}" >/dev/null 2>&1
    done
}

#################################
# 重启sshd服务
#################################
function restart_sshd() {
    if [ ${linux_type} = "SUSE" ]; then
        service sshd restart >>"${LOG_FILE}" 2>&1
    elif [ ${linux_type} = "CentOS" -o ${linux_type} = "Redhat" -o ${linux_type} = "EulerOS" ]; then
        systemctl restart sshd.service >>"${LOG_FILE}" 2>&1
    fi
}

#################################
# 允许ipv6
#################################
function enable_ipv6() {
    local file="/etc/sysctl.conf"
    for line in "net.ipv6.conf.all.disable_ipv6" "net.ipv6.conf.default.disable_ipv6" "net.ipv6.conf.lo.disable_ipv6"; do
        cat "${file}" | grep "^${line}" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            sed -i "s/${line}.*/${line} = 0/g" "${file}"
        else
            echo "${line} = 0" >>"${file}"
        fi
    done
}

#################################
# 配置三方站点时区
#################################
function config_zone() {
    timedatectl set-ntp no
    ret=1
    retry_time=3
    echo "time zone is ${TIME_ZONE}" >>"${LOG_FILE}"
    while (($retry_time > 0)); do
        # 重复执行三次防止修改失败
        timedatectl set-timezone "${TIME_ZONE}" >>"${LOG_FILE}" 2>&1
        if [ $? -eq 0 ]; then
            ret=0
            break
        fi
        ((retry_time--))
        sleep 1
    done
    if [ ${ret} -ne 0 ]; then
        do_exit ${RETURN_ERROR} "ERROR" "file to modify third node Time zone."
    fi
}

#################################
#配置三方站点时间
#################################
function config_time() {
    echo "${NTP_TIME_IP}" | grep ":" >/dev/null 2>&1
    if [ $? == 0 ]; then
        ping6 -c 3 ${NTP_TIME_IP} >>"${LOG_FILE}" 2>&1
    else
        ping -c 3 ${NTP_TIME_IP} >>"${LOG_FILE}" 2>&1
    fi
    if [ $? != 0 ]; then
        do_exit ${RETURN_ERROR} "ERROR" "file to modify third node time, can not connect to the ntp node."
    fi
    service ntpd status | grep "active (running)" >>"${LOG_FILE}" 2>&1
    if [ $? == 0 ]; then
        service ntpd stop >>"${LOG_FILE}" 2>&1
        ntp_type=ntpd
    fi
    service chronyd status | grep "active (running)" >>"${LOG_FILE}" 2>&1
    if [ $? == 0 ]; then
        service chronyd stop >>"${LOG_FILE}" 2>&1
        ntp_type=chronyd
    fi
    ntpdate ${NTP_TIME_IP} >>"${LOG_FILE}" 2>&1
    service ${ntp_type} start
}

#################################
#后置处理，清除多余网络配置
#################################
function restart_network() {
    bash /usr/local/ostools/bond_tools/config_netcard.sh "enable" &
}

function main() {
    init_log
    init_env
    get_params "$@"
    config_third_sys
    clear_backup_file
    config_zone
    config_time
    config_third_host
}

main "$@"
