#!/bin/bash
## ##############################################################################
## # Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
## 修改三方站点NTP
## # ----------------------------------------------------------------
## # 参数为时钟源ip
## #
## ##############################################################################

# NTP 类型
NTP_TYPE="chronyd"
# 主用时钟源ip
ACTIVE_IP=""
# 备用时钟源ips
STANDY_IPS=""
# 备用时钟源数组
STANDY_IP_ARR=()

##################################################
# 将参数赋值给全局变量
##################################################
function init_params() {
    local args=$(getopt -a -o h -l active:,standby:,help -- "$@" 2>/dev/null)
    if [ $? != 0 ]; then
        params_example
        return 1
    fi
    eval set -- "${args}"
    while :; do
        case "$1" in
        --active)
            ACTIVE_IP=$2
            shift
            ;;
        --standby)
            STANDY_IPS=$2
            shift
            ;;
        --)
            shift
            break
            ;;
        *)
            params_example && return 1
            ;;
        esac
        shift
    done
    [ -z "${ACTIVE_IP}" ] && params_example && return 1
    STANDY_IP_ARR=(${STANDY_IPS//,/ })
}

##################################################
# 打印命令执行样例
##################################################
function params_example() {
    echo "Incorrect parameter, for example: bash config_third_ntp.sh -active 192.168.10.10 -standby 192.168.10.11,192.168.10.12"
}

##################################################
# 检查输入参数
##################################################
function param_check() {
    # 检查主用
    local ret=0
    check_ip ${ACTIVE_IP} || return 1
    check_ntp ${ACTIVE_IP} || return 1

    # 检查备用
    for standy_ip in ${STANDY_IP_ARR[@]}; do
        check_ip ${standy_ip} || return 1
        check_ntp ${standy_ip} || return 1
    done
}

##################################################
# 修改NTP前置检查
##################################################
function pre_check() {
    check_run_user || return 1
    check_node_role || return 1
    check_local_ntp_service || return 1
}

##################################################
# 检查本地NTP服务
##################################################
function check_local_ntp_service() {
    systemctl status chronyd | grep "Active" | grep "running" &>/dev/null
    if [ $? -eq 0 ]; then
        return 0
    fi
    systemctl status ntpd | grep "Active" | grep "running" &>/dev/null
    if [ $? -eq 0 ]; then
        NTP_TYPE="ntpd"
        return 0
    fi
    echo "The local NTP service is not started. Run the systemctl start chronyd command to start the NTP service."
    return 1
}

##################################################
# 校验时钟源是否可连通
##################################################
function check_ntp() {
    systemctl status ${NTP_TYPE} | grep "Active" | grep "running" &>/dev/null
    if [ $? -eq 0 ]; then
        systemctl stop ${NTP_TYPE} &>/dev/null
    fi
    local ntp_server_ip=$1
    ntpdate -q ${ntp_server_ip} >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "NTP server ${ntp_server_ip} is not reachable"
        return 1
    fi
}

##################################################
## 校验IP合法性
#
## @param 输入ip
##################################################
function check_ip() {
    [[ "$1" =~ ^([1-9][0-9]?|1[0-9][0-9]|2[0-4][0-9]|25[0-5])(\.([1-9]?[0-9]|1[0-9][0-9]?|2[0-4][0-9]|25[0-5])){3}$ ]] ||
        [[ "$1" =~ ^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$ ]]
    if [ $? -ne 0 ]; then
        echo "The ip address format of $1 is incorrect"
        return 1
    fi
}

##################################################
# 检查执行节点是否是三方站点
##################################################
function check_node_role() {
    local arbitration_dirs=$(ls /opt | grep arbitration-monitor)
    local monitor_arr=(${arbitration_dirs})
    for monitor_dir in ${monitor_arr[@]}; do
        if [ ! -f "/opt/${monitor_dir}/conf/aam.properties" ]; then
            continue
        fi
        local node_role=$(grep "aam.local.role" "/opt/${monitor_dir}/conf/aam.properties" | awk -F"=" '{print $2}')
        if [ -n "${node_role}" ]; then
            break
        fi
    done
    if [ "third" != "${node_role}" ]; then
        echo "This script can be run only on the third-party quorum node."
        return 1
    fi
}

##################################################
# 检查执行用户是否是root
##################################################
function check_run_user() {
    local cur_user=$(id -un)
    [ "${cur_user}" == "root" ] && return 0
    echo "The script must run with root user."
    return 1
}

##################################################
# 修改三方站点NTP配置
##################################################
function modify_ntp_config() {
    local ntp_config_file="/etc/chrony.conf"
    if [ "ntpd" = "${NTP_TYPE}" ]; then
        ntp_config_file="/etc/ntp.conf"
    fi
    cp -p "${ntp_config_file}" "${ntp_config_file}_for_arbitration" >/dev/null 2>&1
    local ret=0
    sed -i '/^server/d' "${ntp_config_file}" || let ret=ret+1
    systemctl stop ${NTP_TYPE} >/dev/null 2>&1
    if [ "ntpd" = "${NTP_TYPE}" ]; then
        add_ntpd_conf "${ntp_config_file}"
    else
        add_chronyd_conf "${ntp_config_file}"
    fi
    let ret=ret+$?
    if [ ${ret} -ne 0 ]; then
        mv "${ntp_config_file}_for_arbitration" "${ntp_config_file}"
        echo "NTP server at the third-party site configured failed."
    else
        echo "Start to synchronize time from the clock source."
        ntpdate ${ACTIVE_IP} >/dev/null
        hwclock -w >/dev/null
        echo "NTP server at the third-party site configured successfully."
    fi
    systemctl start ${NTP_TYPE}
    return ${ret}
}

##################################################
# ntpd服务NTP配置
##################################################
function add_ntpd_conf() {
    local ntp_config_file=$1
    echo "server ${ACTIVE_IP} burst iburst maxpoll 7 prefer true" >>${ntp_config_file}
    for standy_ip in ${STANDY_IP_ARR[@]}; do
        echo "server ${standy_ip} burst iburst maxpoll 7" >>${ntp_config_file}
    done
}

##################################################
# chrond服务NTP配置
##################################################
function add_chronyd_conf() {
    local ntp_config_file=$1
    echo "server ${ACTIVE_IP} iburst trust prefer version 4" >>${ntp_config_file}
    for standy_ip in ${STANDY_IP_ARR[@]}; do
        echo "server ${standy_ip} iburst version 4" >>${ntp_config_file}
    done
}

function main() {
    pre_check || return 1
    init_params "$@" || return 1
    param_check
    if [ $? -ne 0 ]; then
        systemctl start ${NTP_TYPE}
        return 1
    fi
    modify_ntp_config || return 1
}

main "$@"
exit $?
