 #!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

# 包装错误信息，打印到easysuite前台界面
function echo_message() {
    echo "$@" 1>&2
}

##################################################
# 校验执行用户
# 脚本要求使用root用户执行
##################################################
function check_user() {
    local user
    user=$(whoami)
    if [ "${user}" != "root" ]; then
        echo_message "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##################################################
# 日志记录进/opt/backupArb/log/${SELF_FILE}_*.log
##################################################
function LOG() {
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]; then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]; then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2" | tee -a ${LOG_FILE}
    if [ "${Level}" == "ERROR" ]; then
        echo_message "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2"
    fi
}

############################################
# 清理多余的日志文件
##################################################
function clear_log() {
    local log_path=$1
    local need_clear_num=$(echo $(($(ls ${log_path} | grep ".log.gz" | wc -l) - 50)))
    if [ ${need_clear_num} -lt 0 ]; then
        need_clear_num=0
    fi
    find ${log_path} -name "*.log.gz" | sort | head -n ${need_clear_num} | xargs rm -f > /dev/null 2>&1
}

# 获取输入参数
function init_params() {
    local num=$#
    if [ $((${num} % 2)) -eq 1 ]; then
        num=$((${num} / 2 + 1))
    else
        num=$((${num} / 2))
    fi
    local count=1
    for ((i = 1; i <= ${num}; i++)); do
        [[ ${1#-} == "iplist" ]] && {
            IPLIST=$2
            shift 2
            continue
        }
        [[ ${1#-} == "backup_dir" ]] && {
            BACKUP_DIR=$2
            shift 2
            continue
        }
        [[ ${1#-} == "timestamp" ]] && {
            TIMESTAMP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "thirdip" ]] && {
            THIRDIP=$2
            shift 2
            continue
        }
        [[ ${1#-} == "action" ]] && {
            ACTION=$2
            shift 2
            continue
        }
    done

    local log_path=/opt/backupArb/log
    if [ ! -d ${log_path} ]; then
        mkdir -p /opt/backupArb/log >/dev/null 2>&1
    fi
    LOG_FILE="${log_path}/${SELF_FILE//.sh/}_${BACKUP_DIR}.log"
}

##################################################
# 校验入参,入参顺序无要求
# @param
#   -iplist         --192.168.75.195,192.168.75.197,192.168.75.199
#   -pkg            --目标软件包名
#   -timestamp      --当前任务时间戳
#   -thirdip        --三方站点IP
#   -action         --任务动作
##################################################
function check_input() {
    init_params "$@"
    if [ $? -ne 0 ]; then
        echo "Example: bash ${SELF_FILE} -iplist 192.168.75.195,192.168.75.197,192.168.75.199"\
        "-pkg NCE_AAMonitor.zip -timestamp 1605078004164 -thirdip 192.168.75.199 -action stop"
        LOG "ERROR" "The input param is invalid."
        return 1
    fi
}

function create_check_file() {
    result_code=$1
    echo "#!/bin/bash
echo  RESULT:${result_code} && rm -f /opt/arbitration_file/es_script/check_backup_arb.sh
   " >/opt/arbitration_file/es_script/check_backup_arb.sh

    chmod 500 /opt/arbitration_file/es_script/check_backup_arb.sh
}

function get_params() {
    ARB_COUNT=$(echo "${IPLIST}" | awk -F',' '{print NF}')
    node_1_ip=$(echo "${IPLIST}" | awk -F',' '{print $1}')
    node_3_ip=$(echo "${IPLIST}" | awk -F',' '{print $2}')
    node_5_ip=$(echo "${IPLIST}" | awk -F',' '{print $3}')

    read -p "es_input: node_1_root_pwd" -sr node_1_root_pwd
    read -p "es_input: node_3_root_pwd" -sr node_3_root_pwd
    read -p "es_input: node_5_root_pwd" -sr node_5_root_pwd

    node_1_root_pwd=$(echo ${node_1_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    node_3_root_pwd=$(echo ${node_3_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    node_5_root_pwd=$(echo ${node_5_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')

    if [ "${ARB_COUNT}" -eq "5" ]; then
        node_2_ip=$(echo "${IPLIST}" | awk -F',' '{print $4}')
        node_4_ip=$(echo "${IPLIST}" | awk -F',' '{print $5}')

        read -p "es_input: node_2_root_pwd" -sr node_2_root_pwd
        read -p "es_input: node_4_root_pwd" -sr node_4_root_pwd

        node_2_root_pwd=$(echo ${node_2_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
        node_4_root_pwd=$(echo ${node_4_root_pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    fi
}

function scp_file() {
    local node_ip=$1
    local node_pwd=$2
    local local_file=$3
    local remote_file=$4
    [ -f /root/.ssh/known_hosts ] && sed -i "/^${node_ip} /d" /root/.ssh/known_hosts
    node_ip="\["${node_ip}"\]"
    expect <<EOF >/dev/null 2>&1
    set timeout 30
    spawn scp -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null ${local_file} root@${node_ip}:${remote_file}
        expect {
            "*yes/no*" {send "yes\r";exp_continue}
            "*assword:" {send -- "${node_pwd}\r"}
        }
        expect "100%" { exit 0 }
        expect timeout { exit 1 }
    expect eof
EOF
    return $?
}

function scp_file_retry() {
    local node_ip=$1
    local node_pwd=$2
    local local_file=$3
    local remote_file=$4

    LOG "WARN" "scp local flag file to ${node_ip}"
    for i in {1..3}; do
        scp_file "${node_ip}" "${node_pwd}" "${local_file}" "${remote_file}"
        [ $? -eq 0 ] && return 0
        LOG "WARN" "retry to scp local flag file to ${node_ip}"
        sleep 1
    done

    LOG "ERROR" "failed to scp local flag file to ${node_ip}"
    return 1
}

function create_error_flag() {
    LOG "WARN" "create error flag, tar failed"
    local error_flag=/opt/backupArb/arbitration_error.flag
    [ ! -d /opt/backupArb ] && mkdir -p /opt/backupArb
    [ ! -f ${error_flag} ] && touch ${error_flag}

    LOG "WARN" "scp local error flag file to other node"
    scp_file_retry "${node_1_ip}" "${node_1_root_pwd}" "${error_flag}" "${error_flag}" || return 1
    scp_file_retry "${node_3_ip}" "${node_3_root_pwd}" "${error_flag}" "${error_flag}" || return 1
    scp_file_retry "${node_5_ip}" "${node_5_root_pwd}" "${error_flag}" "${error_flag}" || return 1
    if [ "${ARB_COUNT}" -eq "5" ]; then
        LOG "WARN" "5 node scp local error flag file to other node"
        scp_file_retry "${node_2_ip}" "${node_2_root_pwd}" "${error_flag}" "${error_flag}" || return 1
        scp_file_retry "${node_4_ip}" "${node_4_root_pwd}" "${error_flag}" "${error_flag}" || return 1
    fi
    LOG "WARN" "end to scp local error flag file to other node"
}

function get_index() {
    index=""
    for check_installed_item in $(ls /opt | grep arbitration-etcd); do
        echo "${check_installed_item}" | grep "_" >/dev/null 2>&1
        if [ $? -ne 0 ]; then
            tmp_index="0"
        else
            tmp_index=$(echo "${check_installed_item}" | sed "s/arbitration-etcd_//g")
        fi
        index="${index} ${tmp_index}"
    done

    for i in {0..10}; do
        echo "${index}" | grep -qw "${i}"
        if [ $? -eq 0 ]; then
            echo ${i}
            return 0
        fi
    done
}

# 初始化任务相关信息
function init_task_info() {
    # 获取输入参数
    check_input "$@" || return 1

    # 解析输入参数,确定当前站点类型
    # NCE:NCE主备 THIRD:三方仲裁
    ifconfig | grep -qw "${THIRDIP}"
    if [ $? -ne 0 ]; then
        SITE_TYPE="NCE"
        LOG "INFO" "The current site is a NCE site."
    else
        SITE_TYPE="THIRD"
        LOG "INFO" "The current site is a third site."
    fi

    # 当前环境为三方站点时，确定尾缀参数
    if [ "${SITE_TYPE}" = "THIRD" ]; then
        LOG "INFO" "Start to query index."
        local query_script="/opt/arbitration_file/es_script/query_arbitration.sh"
        if [ -z "${query_script}" ]; then
            LOG "WARN" "Failed to confirm query scripts."
            TAIL_INDEX=$(get_index)
            return 0
        fi
        bash ${query_script} >/var/log/arb_stie_info.log
        if [ $? -ne 0 ]; then
            LOG "WARN" "Failed to query arb info."
            TAIL_INDEX=$(get_index)
            return 0
        fi
        local cmd="cat /var/log/arb_stie_info.log | grep -v INDEX"
        local ips=$(echo ${IPLIST} | awk -F',' '{for(i=1;i<=NF;++i) print $i}')
        for ip in ${ips}; do
            cmd="${cmd} | grep -w '${ip}'"
        done
        TAIL_INDEX=$(eval ${cmd} | awk '{print $1}')
        if [ -z "${TAIL_INDEX}" ]; then
            TAIL_INDEX=$(get_index)
        fi
        LOG "INFO" "Index is ${TAIL_INDEX}."
        LOG "INFO" "Finished to query index."
    else
        TAIL_INDEX="0"
    fi
}

function init_path() {
    LOG "INFO" "Index is ${TAIL_INDEX}"
    if [ "${TAIL_INDEX}" = "0" ]; then
        TARGET="/opt/arbitration/"
        TARGET_M="/opt/arbitration-monitor/"
        TARGET_E="/opt/arbitration-etcd/"
    else
        TARGET="/opt/arbitration_${TAIL_INDEX}/"
        TARGET_M="/opt/arbitration-monitor_${TAIL_INDEX}/"
        TARGET_E="/opt/arbitration-etcd_${TAIL_INDEX}/"
    fi

    LOG "INFO" "TARGET is ${TARGET}."
    LOG "INFO" "TARGET_M is ${TARGET_M}."
    LOG "INFO" "TARGET_E is ${TARGET_E}."

    BACKUP_PATH="/opt/backupArb/${BACKUP_DIR}/${TAIL_INDEX}"
    mkdir -p "${BACKUP_PATH}" &>>${LOG_FILE}
    if [ $? -ne 0 ]; then
        LOG "ERROR" "Failed to create backup path:${BACKUP_PATH}."
        return 1
    fi
}

function kill_process() {
    local process=$1
    pid=$(ps -ef | grep "${process}" | grep -v grep | awk '{print $2}')
    if [ -n "${pid}" ]; then
        kill -9 "${pid}" &>>"${LOG_FILE}"
    fi
}

# 停止指定服务
function stop_specified_service() {
    local index_file=$1
    local server_name=$2
    LOG "INFO" "Start to stop arbitration-${server_name}."
    if [ "${index_file}" == "0" ]; then
        local tail=""
    else
        local tail="_${index_file}"
    fi

    if [ ! -f /opt/arbitration-${server_name}${tail}/script/service.sh ]; then
        # 脚本如果不存在，则直接kill
        LOG "WARN" "/opt/arbitration-${server_name}${tail}/script/service.sh is not exist, kill the process"
        kill_process "/opt/arbitration-${server_name}${tail}"
        return 0
    fi

    bash /opt/arbitration-${server_name}${tail}/script/service.sh stop &>>${LOG_FILE}
    if [ $? -ne 0 ]; then
        sudo -u arbiter bash /opt/arbitration-${server_name}${tail}/script/service.sh stop &>>${LOG_FILE}
        if [ $? -ne 0 ]; then
            LOG "ERROR" "Failed to stop arbitration-${server_name}."
            return 1
        fi
    fi
    LOG "INFO" "Finished to stop arbitration-${server_name}."
}

# 备份仲裁
function backup_application() {
    LOG "INFO" "Start to backup arbitration application."
    local index_file=$1
    cd /opt/
    rm -f ${BACKUP_PATH}/arbitrationBackup.tar.gz
    rm -f ${BACKUP_PATH}/arbitrationLog.tar.gz
    clear_log "/var/log/arbitration-etcd/${index_file}"
    clear_log "/var/log/arbitration-monitor/${index_file}"
    if [ "${index_file}" == "0" ]; then
        # 备份应用程序
        tar --warning=no-file-changed -czf ${BACKUP_PATH}/arbitrationBackup.tar.gz arbitration arbitration-common arbitration-etcd arbitration_file.bak arbitration-monitor &>>${LOG_FILE}
        result=$?
        rm -rf /opt/arbitration_file.bak
        if [ ${result} -ne 0 ]; then
            # 打包失败，不直接返回失败
            LOG "WARN" "Failed to create backup file:${BACKUP_PATH}/arbitrationBackup.tar.gz. Return Code:${result}"
            create_error_flag || return 1
        fi
    else
        # 备份应用程序
        tar --warning=no-file-changed -czf ${BACKUP_PATH}/arbitrationBackup.tar.gz arbitration_${index_file} arbitration-common_${index_file} arbitration-etcd_${index_file} arbitration_file.bak arbitration-monitor_${index_file} &>>${LOG_FILE}
        result=$?
        rm -rf /opt/arbitration_file.bak
        if [ ${result} -ne 0 ]; then
            # 打包失败，不直接返回失败
            LOG "WARN" "Failed to create backup file:${BACKUP_PATH}/arbitrationBackup.tar.gz. Return Code:${result}"
            create_error_flag || return 1
        fi
    fi
    # 备份日志
    cd /var/log
    tar --warning=no-file-changed -czf ${BACKUP_PATH}/arbitrationLog.tar.gz arbitration/${index_file} arbitration-etcd/${index_file} arbitration-monitor/${index_file} arbitration_install/${index_file} &>>${LOG_FILE}
    result=$?
    if [ ${result} -ne 0 ]; then
        # 打包失败，不直接返回失败
        LOG "WARN" "Failed to create backup file:${BACKUP_PATH}/arbitrationLog.tar.gz. Return Code:${result}"
    fi
    cat /etc/passwd 2>/dev/null | grep ^arbiter >${BACKUP_PATH}/passwd.bak
    LOG "INFO" "Finished to backup arbitration application."
}

# 如果上次已经备份成功，重入直接跳过
function check_flag() {
    grep -q "backup:${TIMESTAMP}" /opt/backupArb/arb_backup_rollback.flag 2>/dev/null
    if [ $? -eq 0 ]; then
        LOG "INFO" "The current project has been successfully backed up. Skip this step."
        return 0
    else
        return 1
    fi
}

# 等待定时任务结束
function wait_cron_task_stop() {
    sleep 5
    for i in {1..10}; do
        ps -ef | grep $1 | grep -v grep >/dev/null 2>&1
        if [ $? -ne 0 ]; then
            return 0
        fi
        sleep 1
        LOG "INFO" "Waiting cron task to end: ${i}"
    done
    LOG "WARN" "cron task is running timeout"
}

# 删除定时任务并重启
function clean_restart_crontab() {
    local aamonitor_watchdog_path_format=$(echo "${TARGET%/}/script/aamonitor_watchdog.sh" | sed 's/\//\\\//g')
    sed -i "/${aamonitor_watchdog_path_format}/d" /etc/crontab 2>/dev/null
    sed -i "/${aamonitor_watchdog_path_format}/d" /var/spool/cron/root 2>/dev/null

    local cron_service_name="crond.service"
    systemctl -l | grep -q cron.service
    if [ $? -eq 0 ]; then
        cron_service_name="cron.service"
    fi
    clear_watch_dog_pid
    LOG "INFO" "Start to restart ${cron_service_name}"
    systemctl restart ${cron_service_name} &>>${LOG_FILE}
    if [ $? -ne 0 ]; then
        LOG "ERROR" "Failed to restart ${cron_service_name}"
        return 1
    fi
    LOG "INFO" "Finished to restart ${cron_service_name}"

    wait_cron_task_stop ${aamonitor_watchdog_path_format}
}

function clear_watch_dog_pid() {
    rm -rf "${TARGET%/}/script/Pid"
    local watchdog_script="${TARGET%/}/script/aamonitor_watchdog.sh"
    if [ ! -f "${watchdog_script}" ]; then
        return
    fi
    cat "${watchdog_script}" | grep -q "check_running"
    if [ $? -eq 0 ]; then
        sed -i 's#exit 1#pid_list=$(ls /opt/arbitration/script/Pid/ 2>/dev/null);for pid in ${pid_list[@]};do ps -ef | grep -w ${pid} | grep -v grep >/dev/null || rm -f /opt/arbitration/script/Pid/${pid};done;exit 0#g' "${watchdog_script}"
    fi
}

function set_expiration() {
    LOG "INFO" "Set the user password validity period."
    chage -M 99999 arbiter >/dev/null 2>&1
}

function backup_cronb() {
    if [ ! -f "${BACKUP_PATH}/crontab.bak" ]; then
        cp -f /etc/crontab ${BACKUP_PATH}/crontab.bak 2>/dev/null
    fi

    if [ -f "/var/spool/cron/root" -a ! -f "${BACKUP_PATH}/crontab_spool.bak" ]; then
        cp -f /var/spool/cron/root ${BACKUP_PATH}/crontab_spool.bak 2>/dev/null
    fi
}

# 停止仲裁软件monitor+etcd
function stop_arb() {
    set_expiration
    backup_cronb
    clean_restart_crontab || return 1
    stop_specified_service "${TAIL_INDEX}" "etcd" || return 1
    stop_specified_service "${TAIL_INDEX}" "monitor" || return 1
    LOG INFO "Kill ${TARGET}"
    kill_process "${TARGET}"
    LOG INFO "Kill ${TARGET_M}"
    kill_process "${TARGET_M}"
    LOG INFO "Kill ${TARGET_E}"
    kill_process "${TARGET_E}"
}

function backup_arb0() {
    backup_arb
    if [ $? -eq 0 ]; then
        $(create_check_file 100)
        return 0
    else
        $(create_check_file 255)
        return 1
    fi
}

# 备份仲裁软件
function backup_arb() {
    rm -rf /opt/upgrade/easysuite_upgrade/scripts/common/NCE-Common/check_backup_arb.sh
    check_flag && return 0
    get_params || return 1
    backup_application "${TAIL_INDEX}" || return 1
    echo "backup:${TIMESTAMP}" >>/opt/backupArb/arb_backup_rollback.flag
}

function main() {
    export LANG=en_US.UTF-8
    check_user || return 1

    init_task_info "$@" || return 1

    init_path || return 1

    case ${ACTION} in
    "stop")
        stop_arb || return 1
        ;;
    "backup")
        backup_arb0 || return 1
        ;;
    *)
        LOG "ERROR" "Unsupported action."
        return 1
        ;;
    esac
}

main "$@"
exit $?
