#!/bin/bash
## ##############################################################################
## # Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
# 修改操作系统用户密码
## # ----------------------------------------------------------------
## # 参数为用户名
## #
## ##############################################################################

function init_log() {
    LOG_FILE="/var/log/modify_third_node_pwd.log"
    if [ ! -f "${LOG_FILE}" ]; then
        touch "${LOG_FILE}"
    fi
    echo "" >"${LOG_FILE}"
}

function init_env_cmd() {
    PYTHON_CMD=$(find /opt/arbitration-common* -name "python" -type f | head -n 1)
    ${PYTHON_CMD} -V 1>/dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "Env Python exception."
        return 1
    fi
}

function write_log() {
    local level=$1
    local write_sys=$2
    local operation=$3
    local oper_ip=$(who am i | awk '{print $NF}' | sed 's/[()]//g')
    if [ "${level}" = "Error" ]; then
        local sys_log_level="local0.err"
        local cmd_result="Error"
    else
        local sys_log_level="local0.info"
        local cmd_result="Successful"
    fi
    local my_name=$(basename $0)
    local log_time=$(date '+%Y:%m:%d:%H:%M:%S')
    echo "${operation}"
    echo "Data: ${log_time} ${my_name} level:[${level}] [message]:${oper_ip:-127.0.0.1};${operation}." >>"${LOG_FILE}"
    if [ "${write_sys}" = "yes" ]; then
        logger -t root -p ${sys_log_level} "${my_name};${cmd_result};${oper_ip:-127.0.0.1};${operation}"
    fi
}

#########################################
## 修改用户密码
#
## @return 修改结果
#########################################
function modify_passwd() {
    write_log "Info" "yes" "Begin to modify the ${USER_NAME}"
    local count=1
    while true; do
        OLD_IFS="$IFS"
        IFS=$'\n'
        echo -n "Input the new password for ${USER_NAME}:"
        read -rs pass
        echo " "
        echo -n "Re-enter new Password:"
        read -rs again
        echo " "
        IFS="$OLD_IFS"
        #判断两次输入一样
        if [ "${pass}" = "${again}" ]; then
            chack_pass "${again}"
            #判断符合规则
            if [ $? -eq 0 ]; then
                # 修改操作需要重写
                execute_passwd_cmd "${again}"
                if [ $? -eq 0 ]; then
                    write_log "Info" "yes" "Modify the ${USER_NAME} successfully."
                    return 0
                fi
                write_log "Error" "yes" "Failed to modify the ${USER_NAME}"
                return 1
            elif [ ${count} -gt 2 ]; then
                echo "The password is incorrectly entered for three consecutive times."
                write_log "Error" "yes" "Failed to modify the ${USER_NAME}"
                return 1
            else
                write_log "Info" "no" "Does not comply with security rules! again"
                count=$((count + 1))
                continue
            fi
        elif [ ${count} -gt 2 ]; then
            echo "The password is incorrectly entered for three consecutive times."
            write_log "Error" "yes" "Failed to modify the ${USER_NAME}"
            return 1
        else
            write_log "Info" "no" "The two passwords do not match. Enter the same password"
            count=$((count + 1))
            continue
        fi
    done
}

#########################################
## 修改用户密码命令
#
## @param 用户输入的密码
## @return 修改结果
#########################################
function execute_passwd_cmd() {
    local pwd=$1
    pwd=$(echo ${pwd} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
(
    expect <<EOF >> ${LOG_FILE}
       set timeout 30
       spawn passwd ${USER_NAME}
   expect {
   "*New password:*" {
       send "${pwd}\r"
       exp_continue
   }
   "Retype new password:" {
       send "${pwd}\r"
       exp_continue
   }
   "*successfully*" { exit 0 }
   "*password:*" { exit 1 }
   }
   expect timeout {
        exit 1
    }
EOF
) > /dev/null 2>&1
    if [ $? -ne 0 ];then
        return 1
    fi
}

#########################################
## 密码规则检查
#
## @param 用户输入的密码
## @return 校验结果
#########################################
function chack_pass() {
    local input_pwd=$1
    local pwd_length=${#input_pwd}
    # 密码长度校验
    if [[ ${pwd_length} -lt 8 || ${pwd_length} -gt 64 ]]; then
        write_log "Info" "no" "PassWD rules-->The length of password is not between 8 and 64."
        return 1
    fi
    # 密码必须包含大写字母
    echo "${input_pwd}" | egrep "[[:upper:]]" &>/dev/null
    if [ $? -ne 0 ]; then
        write_log "Info" "no" "PassWD rules-->No capital letters."
        return 1
    fi
    # 密码必须包含小写字母
    echo "${input_pwd}" | egrep "[[:lower:]]" &>/dev/null
    if [ $? -ne 0 ]; then
        write_log "Info" "no" "PassWD rules-->No lowercase letters."
        return 1
    fi
    # 密码必须包含数字
    echo "${input_pwd}" | egrep '[0-9]' &>/dev/null
    if [ $? -ne 0 ]; then
        write_log "Info" "no" "PassWD rules-->No numbers."
        return 1
    fi
    # 密码必须包含限定的特殊字符
    echo "${input_pwd}"| egrep -q '[~@#^*_+[{}:./?%=!]|]|-'
    if [ $? -ne 0 ]; then
        write_log "Info" "no" "PassWD rules-->No special characters."
        return 1
    fi
    # 密码不能以！开头
    echo "${input_pwd}" | egrep '^!+.*$' &>/dev/null
    if [ $? -eq 0 ]; then
        write_log "Info" "no" "PassWD rules-->Starting with ! is not allowed."
        return 1
    fi
    # 密码与账户或账户反向不能相同
    check_pwd_not_same_as_user "${input_pwd}"
    if [ $? -ne 0 ]; then
        write_log "Info" "no" "PassWD rules-->Password and account or reverse account can't same. Password can't contain account(Case Insensitive)"
        return 1
    fi
    # 密码不能与其反向相同，不区分大小写
    local upper_pass=$(echo "${input_pwd}" | tr '[:lower:]' '[:upper:]')
    local rev_upper_name=$(echo "${upper_name}" | rev)
    if [ "${upper_pass}" = "${rev_upper_name}" ]; then
        write_log "Info" "no" "PassWD rules-->Not be the same as the reverse of it, regardless of the letter case."
        return 1
    fi

    # 检查弱密码
    which cracklib-check >/dev/null
    if [ $? -eq 0 ]; then
        local check_ret=$(echo "${input_pwd}" | cracklib-check | awk -F":" '{print $NF}')
        echo "${check_ret}" | grep -w "OK" >/dev/null
        if [ $? -ne 0 ]; then
            write_log "Info" "no" "PassWD rules-->${check_ret}."
            return 1
        fi
    fi

    # 匹配数字，小写字母，大写字母，特殊字符
    local check_ret=$(echo "${input_pwd}" | ${PYTHON_CMD} -c "import re; pwd=input();pattern = re.compile(r'^(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[~@#^*-_+\[{}\]:./?%=\!])[\da-zA-Z~@#^*-_+\[{}\]:./?%=\!]{8,64}$');print(pattern.match(pwd))")
    if [ "${check_ret}" = "None" ]; then
        write_log "Info" "no" "PassWD rules-->The password contains unsupported special characters. Only the following special characters are supported: ~@#^*-_+[{}]:./?%=!"
        return 1
    fi

    # 匹配连续重复三次以上字符
    local check_ret=$(echo "${input_pwd}" | ${PYTHON_CMD} -c "import re; pwd=input();pattern = re.compile(r'^.*([\da-zA-Z~@#^*-_+\[{}\]:./?%=\!])\1{2}.*$');print(pattern.match(pwd))")
    if [ "${check_ret}" != "None" ]; then
        write_log "Info" "no" "PassWD rules-->Cannot contain more than two consecutive identical characters."
        return 1
    fi
    # 同一个字符最多使用三次
    local count=$(echo "${input_pwd}" | ${PYTHON_CMD} -c "pwd=input();print(max(pwd.count(i) for i in pwd))")
    if [ ${count} -gt 3 ]; then
        write_log "Info" "no" "PassWD rules-->The same character can be used three times at most."
        return 1
    fi
}

#########################################
## 检查密码是否包含用户名或者用户名倒叙
#
## @param 用户输入的面
## @return 校验结果
#########################################
function check_pwd_not_same_as_user() {
    local passwd=$1
    local upper_pass=$(echo ${passwd} | tr '[:lower:]' '[:upper:]')
    user_list="root arbiter ossadm ftpuser dbuser ossuser sopuser secuser iscript"
    user_arr=("${user_list}")
    for user in ${user_arr[@]}; do
        local upper_name=$(echo ${user} | tr '[:lower:]' '[:upper:]')
        local rev_upper_name=$(echo "${upper_name}" | rev)
        if [[ "${upper_pass}" =~ ${upper_name} ]] || [[ "${upper_pass}" =~ ${rev_upper_name} ]]; then
            return 1
        fi
    done
}

#########################################
## 检查执行用户是否是root
#
## @return 校验结果
#########################################
function check_run_user() {
    local cur_user=$(id -un)
    [ "${cur_user}" == "root" ] && return 0
    echo "The script must run with root user."
    return 1
}

function check_param() {
    if [ $# -ne 1 ]; then
        echo "Incorrect parameter, for example: bash modify_third_node_pwd.sh sopuser"
        return 1
    fi
    local user=$1
    cat /etc/passwd | awk -F":" '{print $1}' | grep -w "${user}" >/dev/null
    if [ $? -ne 0 ]; then
        echo "${user} does not exist."
        return 1
    fi
    USER_NAME="${user}"
}

function main() {
    check_run_user || return 1
    check_param "$@" || return 1
    init_env_cmd || return 1
    init_log
    modify_passwd
}

main "$@"
exit $?
