#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd.2019-2022. All rights reserved.
cur_path=$(cd $(dirname $0);pwd)
RETURN_AUTH_FAILED=-1
RETURN_SUCCESS=0
RETURN_ERROR=1

PRE_CONFIG_FLAG="pre_config_backup"
BASH_PID=$$

#################################
# 初始化日志
#################################
function init_log() {
    # 预装使用，默认环境未安装仲裁
    LOG_PATH="/var/log/arbitration/0"
    if [ ! -d ${LOG_PATH} ]; then
        mkdir -p ${LOG_PATH}
    fi
    LOG_FILE="${LOG_PATH}/pre_config_third_ip.log"
}

#################################
# 加载环境变量
#################################
function init_env() {
    # 加载系统环境变量
    . /usr/local/ostools/bond_tools/get_os_info.sh >>"${LOG_FILE}" 2>&1
    PATH_PERFIX=""
    PATH_ROUTE_PERFIX=""
    if [ "${linux_type}" = "SUSE" ]; then
        PATH_PERFIX="/etc/sysconfig/network/ifcfg-"
        PATH_PER="/etc/sysconfig/network/"
        PATH_ROUTE_PERFIX="/etc/sysconfig/network/ifroute-"
    elif [ "${linux_type}" = "CentOS" -o ${linux_type} = "Redhat" -o ${linux_type} = "EulerOS" ]; then
        PATH_PERFIX="/etc/sysconfig/network-scripts/ifcfg-"
        PATH_PER="/etc/sysconfig/network-scripts/"
        PATH_ROUTE_PERFIX="/etc/sysconfig/network-scripts/route-"
    fi
}

#################################
# 脚本退出
#################################
function do_exit() {
    local error_code="$1"
    local log_type="$2"
    local log_info="$3"

    echo "${log_info}"
    write_logger "${log_type}" "${log_info}"
    exit ${error_code}
}

#################################
# 日志记录
#################################
function write_logger() {
    local log_type="$1"
    local log_info="$2"

    echo ${log_type} | grep -i "ERROR" &>/dev/null && Level="ERROR"
    echo ${log_type} | grep -i "WARN" &>/dev/null && Level="WARN"
    [ -z "${Level}" ] && Level="INFO"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${BASH_PID}] | ${Level} ${log_info}" >>"${LOG_FILE}"
}

##################################################
# 将参数赋值给全局变量
##################################################
function init_params()
{
    ARGS=$(getopt -a -o h -l type:,login_ip:,login_nic:,login_mask:,login_desk_ip:,login_desk_mask:,login_gateway:,inner_ip:,inner_nic:,inner_mask:,inner_desk_ip:,inner_desk_mask:,inner_gateway:,help -- "$@" 2>/dev/null)
    if [ $? != 0 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "Failed to parse the parameter."
        return 1
    fi
    eval set -- "${ARGS}"
    while :; do
        case $1 in
            --type) TYPE=$2;shift ;;
            --login_ip) LOGIN_IP=$2;shift ;;
            --login_nic) LOGIN_NIC=$2;shift ;;
            --login_mask) LOGIN_MASK=$2;shift ;;
            --login_desk_ip) LOGIN_DESK_IP=$2;shift ;;
            --login_desk_mask) LOGIN_DESK_MASK=$2;shift ;;
            --login_gateway) LOGIN_GATEWAY=$2;shift ;;
            --inner_ip) INNER_IP=$2;shift ;;
            --inner_nic) INNER_NIC=$2;shift ;;
            --inner_mask) INNER_MASK=$2;shift ;;
            --inner_desk_ip) INNER_DESK_IP=$2;shift ;;
            --inner_desk_mask) INNER_DESK_MASK=$2;shift ;;
            --inner_gateway) INNER_GATEWAY=$2;shift ;;
            --) shift;break ;;
            *) do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect." ;;
        esac
        shift
    done
    [ -z "${TYPE}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_NIC}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${LOGIN_DESK_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${LOGIN_DESK_IP}" = "None" ] && LOGIN_DESK_IP=$(get_desk_ip_default ${LOGIN_IP})
    [ -z "${LOGIN_DESK_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${LOGIN_DESK_MASK}" = "None" ] && LOGIN_DESK_MASK=$(get_desk_mask_default ${LOGIN_IP})
    [ -z "${LOGIN_GATEWAY}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_NIC}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ -z "${INNER_DESK_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${INNER_DESK_IP}" = "None" ] && INNER_DESK_IP=$(get_desk_ip_default ${INNER_IP})
    [ -z "${INNER_DESK_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."
    [ "${INNER_DESK_MASK}" = "None" ] && INNER_DESK_MASK=$(get_desk_mask_default ${INNER_IP})
    [ -z "${INNER_GATEWAY}" ] && do_exit ${RETURN_ERROR} "ERROR" "parameter is incorrect."

    Login_IP_LOG_FILE="${PATH_PERFIX}${LOGIN_NIC}_${PRE_CONFIG_FLAG}"
    LOGIN_ROUTE_FILE="${PATH_ROUTE_PERFIX}${LOGIN_NIC}_${PRE_CONFIG_FLAG}"
    if [ "${LOGIN_NIC}" = "${INNER_NIC}" ]; then
        return
    fi
    INNER_IP_FILE="${PATH_PERFIX}${INNER_NIC}_${PRE_CONFIG_FLAG}"

    INNER_ROUTE_FILE="${PATH_ROUTE_PERFIX}${INNER_NIC}_${PRE_CONFIG_FLAG}"

    SSHD_CONFIG_FILE="/etc/ssh/sshd_config_${PRE_CONFIG_FLAG}"
}

#################################
# 获取默认目标ip
#################################
function get_desk_ip_default() {
    local ip=$1
    echo "${ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        echo "::"
    else
        echo "0.0.0.0"
    fi
}

#################################
# 获取默认目标掩码
#################################
function get_desk_mask_default() {
    local ip=$1
    echo "${ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        echo "32"
    else
        echo "0.0.0.0"
    fi
}

#################################
# 修改ip
#################################
function modify_ip() {
    local nic=$1
    local ip_addr=$2
    local netmask=$3
    local ip_type=$(get_ip_type ${ip_addr})
    local ip_name="IPADDR"
    if [ "${linux_type}" = "SUSE" ]; then
        [[ "${ip_type}" == "IPv6" ]] && ip_name="IPADDR6"
    elif [ "${linux_type}" = "CentOS" -o "${linux_type}" = "Redhat" -o "${linux_type}" = "EulerOS" ]; then
        [[ "${ip_type}" == "IPv6" ]] && ip_name="IPV6ADDR"
    fi

    ip_config_file=${PATH_PERFIX}${nic}
    if [ ! -f ${ip_config_file} ]; then
        do_exit ${RETURN_AUTH_FAILED} "ERROR" "The arbitration config file ${file_name} does not exist, exit."
    fi
    local ret=0
    if [ "${ip_type}" = "IPv6" ]; then
        modify_ipv6 "${ip_config_file}" "${ip_name}" "${ip_addr}" "${netmask}" || let ret=ret+1
        enable_ipv6
    else
        modify_ipv4 "${ip_config_file}" "${ip_name}" "${ip_addr}" "${netmask}" || let ret=ret+1
    fi

    # 防止ssh未配置监听ip情况下配置新ip监听失败
    modify_ssh "${ip_addr}" || let ret=ret+1
    if [ ${ret} -ne 0 ]; then
        rollback_config
        do_exit ${RETURN_ERROR} "ERROR" "modify ${nic} ip address failed."
    fi
    write_logger "INFO" "modify ${nic} ip address success."
}

#################################
# 修改ipv4备用ip,保留原有ip情况下在同一个网口新增ip
#################################
function modify_ipv4() {
    local ip_config_file=$1
    local ip_name=$2
    local ip_addr=$3
    local netmask=$4
    local ip_config_tmp_file="${ip_config_file}:0"
    local ret=0
    cp -rp "${ip_config_file}" "${ip_config_tmp_file}" || let ret=ret+1
    sed -i "s#^${ip_name}.*#${ip_name}='${ip_addr}'#g" ${ip_config_tmp_file} 2>/dev/null || let ret=ret+1
    sed -i "s#^NETMASK.*#NETMASK='${netmask}'#g" ${ip_config_tmp_file} 2>/dev/null || let ret=ret+1
    return ${ret}
}

#################################
# 修改ipv6备用ip
#################################
function modify_ipv6() {
    local ip_config_file=$1
    local ip_name=$2
    local ip_addr=$3
    local netmask=$4
    cat "${ip_config_file}" | grep -q "IPV6ADDR_SECONDARIES"
    if [ $? -eq 0 ]; then
        sed -i "s#^${ip_name}_SECONDARIES=.*#${ip_name}_SECONDARIES=${ip_addr}/${netmask}#g" "${ip_config_file}" || let ret=ret+1
    else
        echo "${ip_name}_SECONDARIES=${ip_addr}/${netmask}" >> "${ip_config_file}" || let ret=ret+1
    fi
    return ${ret}
}

#################################
# 修改ssh配置,仅新增，不删除原有配置
#################################
function modify_ssh() {
    local ip_addr=$1
    local listen_address_list=$(grep "^\s*ListenAddress" "/etc/ssh/sshd_config" | egrep -wv "0.0.0.0|::|::1|127.0.0.1" | awk '{print $2}')
    if [ -z "${listen_address_list}" ]; then
        return 0
    fi
    echo ${listen_address_list} | grep -wq "${ip_addr}"
    if [ $? -eq 0 ]; then
        return 0
    fi
    local line_num=$(grep -n "^ListenAddress" /etc/ssh/sshd_config | head -n 1 | awk -F":" '{print $1}')
    sed -i "${line_num} iListenAddress ${ip_addr}" /etc/ssh/sshd_config
    return $?
}

#################################
# 获取ip类型
#################################
function get_ip_type() {
    local ip=$1
    echo "${ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        echo "IPv6"
    else
        echo "IPv4"
    fi
}

#################################
# 修改路由
#################################
function modify_route() {
    local nic=$1
    local gateway=$2
    local desk_ip=$3
    local desk_mask=$4
    if [ "${gateway}" = "None" ]; then
        write_logger "INFO" "gateway of ${nic} is None, no need to config."
        return
    fi
    echo "${desk_ip}" | grep ":" >/dev/null
    if [ $? -eq 0 ]; then
        bash /usr/local/ostools/bond_tools/config_netcard_ipv6.sh "route:${nic};${gateway},${desk_ip},${desk_mask}"
    else
        bash /usr/local/ostools/bond_tools/config_netcard.sh "route:${nic};${gateway},${desk_ip},${desk_mask}"
    fi
    if [ $? -ne 0 ]; then
        rollback_config
        do_exit ${RETURN_ERROR} "ERROR" "modify ${nic} route failed."
    fi
    write_logger "INFO" "modify ${nic} route success."
}

#################################
# 备份网络配置文件,网口文件Ipv4不备份，只备份ipv6
#################################
function backup_config() {
    local login_ip_type=$(get_ip_type ${LOGIN_IP})
    if [ "${login_ip_type}" = "IPv6" ]; then
        copy_file "${PATH_PERFIX}${LOGIN_NIC}" "${Login_IP_LOG_FILE}"
    fi
    copy_file "${PATH_ROUTE_PERFIX}${LOGIN_NIC}" "${LOGIN_ROUTE_FILE}"

    if [ "${LOGIN_NIC}" = "${INNER_NIC}" ]; then
        return
    fi
    local inner_ip_type=$(get_ip_type ${INNER_IP})
    if [ "${inner_ip_type}" = "IPv6" ]; then
        copy_file "${PATH_PERFIX}${INNER_NIC}" "${INNER_IP_FILE}"
    fi
    copy_file "${PATH_ROUTE_PERFIX}${INNER_NIC}" "${INNER_ROUTE_FILE}"

    copy_file /etc/ssh/sshd_config "${SSHD_CONFIG_FILE}"
}

function copy_file() {
    local src_file=$1
    local dest_file=$2
    if [ -f "${src_file}" ]; then
        cp -rp "${src_file}" "${dest_file}"
    fi
}

#################################
# 恢复网络配置文件
#################################
function rollback_config() {
    local login_ip_type=$(get_ip_type ${LOGIN_IP})
    if [ "${login_ip_type}" = "IPv6" ]; then
        move_file "${Login_IP_LOG_FILE}" "${PATH_PERFIX}${LOGIN_NIC}"
    else
        rm -f "${PATH_PERFIX}${LOGIN_NIC}:0" >/dev/null 2>&1
    fi
    move_file "${LOGIN_ROUTE_FILE}" "${PATH_ROUTE_PERFIX}${LOGIN_NIC}"

    local inner_ip_type=$(get_ip_type ${INNER_IP})
    if [ "${inner_ip_type}" = "IPv6" ]; then
        move_file "${INNER_IP_FILE}" "${PATH_PERFIX}${INNER_NIC}"
    else
        rm -f "${PATH_PERFIX}${INNER_NIC}:0" >/dev/null 2>&1
    fi
    move_file "${INNER_ROUTE_FILE}" "${PATH_ROUTE_PERFIX}${INNER_NIC}"

    move_file "${SSHD_CONFIG_FILE}" /etc/ssh/sshd_config

    restart_network
    restart_sshd
}

function move_file() {
    local dest_file=$1
    local src_file=$2
    if [ -f "${dest_file}" ]; then
        mv -f "${dest_file}" ${src_file}
    fi
}

#################################
# 成功后清除备份文件
#################################
function clear_backup_file() {
    for file_name in "${Login_IP_LOG_FILE}" "${Login_IP_LOG_FILE_first}" "${Login_IP_LOG_FILE_second}" "${LOGIN_ROUTE_FILE}" "${INNER_IP_FILE}" "${INNER_IP_FILE_first}" "${INNER_IP_FILE_second}" "${INNER_ROUTE_FILE}" "${SSHD_CONFIG_FILE}"; do
        rm -f "${file_name}" >/dev/null 2>&1
    done
}

#################################
# 重启sshd服务
#################################
function restart_sshd() {
    if [ ${linux_type} = "SUSE" ]; then
        service sshd restart >>"${LOG_FILE}" 2>&1
    elif [ ${linux_type} = "CentOS" -o ${linux_type} = "Redhat" -o ${linux_type} = "EulerOS" ]; then
        systemctl restart sshd.service >>"${LOG_FILE}" 2>&1
    fi
}

#################################
# 允许ipv6
#################################
function enable_ipv6() {
    local file="/etc/sysctl.conf"
    for line in "net.ipv6.conf.all.disable_ipv6" "net.ipv6.conf.default.disable_ipv6" "net.ipv6.conf.lo.disable_ipv6"; do
        cat "${file}" | grep "^${line}" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            sed -i "s/${line}.*/${line} = 0/g" "${file}"
        else
            echo "${line} = 0" >>"${file}"
        fi
    done
}

#################################
#后置处理，重启网络
#################################
function restart_network() {
    if [ ${linux_type} = "SUSE" ]; then
        service network restart >> "${LOG_FILE}" 2>&1
    elif [ ${linux_type} = "CentOS" -o ${linux_type} = "Redhat" -o ${linux_type} = "EulerOS" ]; then
        systemctl restart network.service >> "${LOG_FILE}" 2>&1
    fi
}

function config_third_ip() {
    backup_config
    modify_ip "${LOGIN_NIC}" "${LOGIN_IP}" "${LOGIN_MASK}"
    modify_route "${LOGIN_NIC}" "${LOGIN_GATEWAY}" "${LOGIN_DESK_IP}" "${LOGIN_DESK_MASK}"
    if [ "${LOGIN_NIC}" != "${INNER_NIC}" ]; then
        modify_ip "${INNER_NIC}" "${INNER_IP}" "${INNER_MASK}"
        modify_route "${INNER_NIC}" "${INNER_GATEWAY}" "${INNER_DESK_IP}" "${INNER_DESK_MASK}"
    fi

    restart_network
    if [ $? -ne 0 ]; then
        rollback_config
        do_exit ${RETURN_ERROR} "ERROR" "restart network failed."
    fi
    restart_sshd
}

function main() {
    TYPE=$1
    init_log
    init_env
    init_params "$@" || return 1
    echo "${TYPE}" "start to ${TYPE} the third ip" >>"${LOG_FILE}"
    case ${TYPE} in
    "config")
        config_third_ip
        return $?
        ;;
    "rollback")
        rollback_config
        return $?
        ;;
    esac
    echo "success to ${TYPE} the third ip" >>"${LOG_FILE}"
}

main "$@"
exit $?
