#!/bin/bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2020-2021. All rights reserved.
# File name: remote_sync_arbitration_file.sh
# Description: 同步A配置文件到omp_01
###########################################################
EXPECT=expect
# 源文件和目标都定义为数组，方便后续扩展
SRC_ARRAY=()
TARGET_ARRAY=()
SRC_ARRAY[0]="/opt/arbitration/conf/arbitration_conf.properties"
# 目标路径只支持拷贝到/opt/oss/share/manager/DRService/目录下，且前面路径省略
TARGET_ARRAY[0]="/configuration/arbitration_conf.properties"

function init() {
    . ./arbitration_opertate_lib
    . ./env.properties
    . ./public.sh

    rm -f /opt/arbitration_file/check_sync_file_result.sh

    # 获取maintaince IP
    OMP_IP_ADDRESS=$(get_omp_maintaince_ip)
    if [ -z "${OMP_IP_ADDRESS}" ]; then
        arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "Get OMP node maintaince ip failed."
        $(create_check_file 255)
        exit 255
    fi
}

function get_params() {
    for i in {1..2}; do
        [[ ${1#-} == "role" ]] && {
            site_role=$2
            shift 2
            continue
        }
        [[ ${1#-} == "ip" ]] && {
            local_ip_address=$2
            shift 2
            continue
        }
    done

    read -p "es_input: master_omp_node_uname" -r master_login_user_name
    read -p "es_input: master_omp_node_pwd" -s -r master_login_user_pwd
    read -p "es_input: master_omp_node_ossadm_pwd" -s -r master_ossadm_user_pwd

    read -p "es_input: slave_omp_node_uname" -r slave_login_user_name
    read -p "es_input: slave_omp_node_pwd" -s -r slave_login_user_pwd
    read -p "es_input: slave_omp_node_ossadm_pwd" -s -r slave_ossadm_user_pwd

    if [ "${site_role}" = "master" ]; then
        LOGIN_USER_NAME=${master_login_user_name}
        LOGIN_USER_PWD=${master_login_user_pwd}
        OSSADM_USER_PWD=${master_ossadm_user_pwd}
    else
        LOGIN_USER_NAME=${slave_login_user_name}
        LOGIN_USER_PWD=${slave_login_user_pwd}
        OSSADM_USER_PWD=${slave_ossadm_user_pwd}
    fi

    # 特殊字符转义
    LOGIN_USER_PWD=$(echo ${LOGIN_USER_PWD} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
    OSSADM_USER_PWD=$(echo ${OSSADM_USER_PWD} | sed 's/\\/\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\"/g')
}

function get_omp_maintaince_ip() {
    arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "Start to get the OMP maintenance IP address."
    omp_maintaince_ip=$(get_first_omp_ip)
    arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "Success to get the OMP maintenance IP address is ${omp_maintaince_ip}."
    echo ${omp_maintaince_ip}
}

function create_check_file() {
    result_code=$1
    create_check_result_file ${result_code} "check_sync_file_result.sh"
}

# 登录OMP节点，执行OMP上预置的脚本
function remote_exec() {
    ${EXPECT} <<ENF >/dev/null 2>&1
set timeout 30
spawn ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null ${LOGIN_USER_NAME}@${OMP_IP_ADDRESS}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${LOGIN_USER_PWD}\r"}
}

expect {
    "*${LOGIN_USER_NAME}@*" {
                            send "su - ossadm\r"
                            expect {
                                "*yes/no*" {send "yes\r";exp_continue}
                                "*assword:" {send -- "${OSSADM_USER_PWD}\r"}
                                "*ossadm@*" {send "\r"}
                            }
                            expect {
                                "*ossadm@*" {
                                    send "bash /opt/oss/manager/apps/DRService/shscript/sync_arbitration_file.sh ${local_ip_address} /opt/arbitration_tmp ${TARGET_ARRAY[@]}\r"
                                    expect {
                                        "*synchronize file failed*" { exit 255 }
                                        "*synchronize file successfully*" { exit 100 }
                                    }
                                }
                            }

                            expect {
                                "*ossadm@*" {send "exit\r"}
                            }

    }
}

expect {
    "*${LOGIN_USER_NAME}@*" {send "exit\r"}
}

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]

ENF
    return $?
}

# 本地OMP节点，执行OMP上预置的脚本
function local_exec() {
    ${EXPECT} <<ENF >/dev/null 2>&1
set timeout 30
spawn su - ossadm
expect {
    "*ossadm@*" {
        send "bash /opt/oss/manager/apps/DRService/shscript/sync_arbitration_file.sh ${local_ip_address} /opt/arbitration_tmp ${TARGET_ARRAY[@]}\r"
        expect {
            "*synchronize file failed*" { exit 255 }
            "*synchronize file successfully*" { exit 100 }
        }
    }
}
expect {
    "*ossadm@*" {send "exit\r"}
}

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]

ENF
    return $?
}

function omp_exec() {
    ip addr | grep -w ${OMP_IP_ADDRESS} >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "omp exec on local."
        local_exec
    else
        arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "omp exec on remote."
        remote_exec
    fi
}

function cp_file_to_local() {
    local file_name=$1
    local base_file=$(basename ${file_name})
    cp -f ${file_name} /opt/oss/tmp/manager/DRService
    chown ossadm: /opt/oss/tmp/manager/DRService/${base_file}
}

# 登录OMP节点，检查scp传文件的内容，企业环境偶现该配置文件为空
function check_remote_file_content() {
    check_target_filepath="$1/$(basename "$2")"
    ${EXPECT} <<ENF >/dev/null 2>&1
set timeout 30
spawn ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null ${LOGIN_USER_NAME}@${OMP_IP_ADDRESS}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${LOGIN_USER_PWD}\r"}
}

expect {
    "*${LOGIN_USER_NAME}@*" {
        send "su - ossadm\r"
        expect {
            "*yes/no*" {send "yes\r";exp_continue}
            "*assword:" {send -- "${OSSADM_USER_PWD}\r"}
            "*ossadm@*" {send "\r"}
        }
        expect {
            "*ossadm@*" {
                send "echo check_\\\$(grep MONITOR_PORT ${check_target_filepath} >/dev/null 2>&1 && echo success || echo failed)\r"
                expect {
                    "*check_failed*" { exit 255 }
                    "*check_success*" { exit 100 }
                }
            }
        }
        expect {
            "*ossadm@*" {send "exit\r"}
        }
    }
}

expect {
    "*${LOGIN_USER_NAME}@*" {send "exit\r"}
}

expect timeout { exit 1 }
expect eof

catch wait result
exit [lindex $result 3]

ENF
    if [ $? -ne 100 ]; then
        arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "Check sync file content failed. Target file: ${check_target_filepath}"
        return 1
    fi
    return 0
}

function scp_file_to_omp() {
    local file_name=$1
    local target_path=/opt/oss/tmp/manager/DRService
    # 转换IPV6格式
    local target_ip=${OMP_IP_ADDRESS}

    ip addr | grep -w ${OMP_IP_ADDRESS} >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        cp_file_to_local ${file_name}
        return $?
    fi

    echo ${OMP_IP_ADDRESS} | grep ":" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        local target_ip="\["${OMP_IP_ADDRESS}"\]"
    fi

    ${EXPECT} <<ENF >/dev/null 2>&1
set timeout 30
spawn scp -p -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null ${file_name} ossadm@${target_ip}:${target_path}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send -- "${OSSADM_USER_PWD}\r"}
}

expect {
    "100%" { exit 0 }
}
expect timeout {
    exit 1
}
expect eof

catch wait result
exit [lindex $result 3]
ENF
    if [ $? -ne 0 ]; then
        arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "SCP ERROR. Target ip: ${target_ip}"
        return 1
    fi

    check_remote_file_content "${target_path}" "${file_name}"
    if [ $? -ne 0 ]; then
        arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "Check sync file content failed. Target ip: ${target_ip}"
        return 1
    fi
    return 0
}

function copy_src_file() {
    [ -d "/opt/arbitration_tmp" ] && rm -rf /opt/arbitration_tmp
    mkdir -p /opt/arbitration_tmp

    # 拷贝file
    for src_file in ${SRC_ARRAY[@]}; do
        cp -f ${src_file} /opt/arbitration_tmp
    done

    # 控制属组和权限
    chown -R ossadm:ossgroup /opt/arbitration_tmp
    chmod 700 /opt/arbitration_tmp/*

    # 拷贝file至OMP
    for src_file in ${SRC_ARRAY[@]}; do
        src_filename=$(basename ${src_file})
        for i in {1..3}; do
            arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "start to scp file ${src_filename} to omp, the omp is ${OMP_IP_ADDRESS}"
            $(scp_file_to_omp ${src_file})
            scp_result=$?
            if [ ${scp_result} -eq 0 ]; then
                arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "success to scp file ${src_filename} to omp"
                break
            fi
            sleep 1
        done

        if [ ${scp_result} -ne 0 ]; then
            arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "failed to scp file ${src_filename} to omp"
            $(create_check_file 255)
            exit 255
        fi
    done
}

function synchronize_file_to_omps() {
    omp_exec
    if [ $? -ne 100 ]; then
        arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "Synchronize file failed."
        # 重试一次
        omp_exec
        if [ $? -ne 100 ]; then
            arbitration_log ERROR "REMOTE_SYNC_ARBITRATION_FILE" "Synchronize file failed again."
            [ -d "/opt/arbitration_tmp" ] && rm -rf /opt/arbitration_tmp
            $(create_check_file 255)
            exit
        fi
    fi

    arbitration_log INFO "REMOTE_SYNC_ARBITRATION_FILE" "Synchronize file successfully."
    [ -d "/opt/arbitration_tmp" ] && rm -rf /opt/arbitration_tmp
}

function main() {
    # 初始化环境变量
    init

    # 获取参数
    get_params "$@"

    # 复制目标文件并刷成ossadm属组，方便OMP节点拷贝
    copy_src_file

    # 执行脚本将file拷贝到OMP
    synchronize_file_to_omps

    # 返回成功的结果给easysuite
    $(create_check_file 100)
}

main "$@"
