#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: deal_easysuite_part.sh
# Description: 处理一体化安装easysuite产生的无用分区
###############################################################################


USELESS_MOUNT_POINT_LIST="/spare /opt/backup /opt/systembackup"
USELESS_PATH="/opt/systembackup /spare"
NEW_USELESS_PATH="/opt/databackup"
LOG_FILE='/root/install_config/deal_easysuite_part.log'



##################################################
# Description: 检查/opt/databackup分区是否存在
# Parameters:  无
##################################################
function check_databackup_existing() {
  df -h | grep ${NEW_USELESS_PATH} > /dev/null 2>&1
  if [[ $? -eq 0 ]]; then
    USELESS_MOUNT_POINT_LIST+=" ${NEW_USELESS_PATH}"
  fi
}


##################################################
# Description: 处理无用分区
# Parameters:  无
##################################################
function deal_useless_part() {
  obtain_and_umount_useless_lv
  remove_useless_lv
  remove_useless_opt
  remove_useless_vg
  remove_useless_path
}


##################################################
# Description: 获取lv名，卸载filesystem
# Parameters:  无
##################################################
function obtain_and_umount_useless_lv() {
  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Start to obtain useless LVs." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  for useless_mount_point in ${USELESS_MOUNT_POINT_LIST}; do
    useless_lv_name=$(df -h | grep ${useless_mount_point} | awk '{print $1}')
    useless_lv_name_list+=" ${useless_lv_name}"
    umount -l ${useless_mount_point} > /dev/null 2>&1
  done
  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Obtaining useless LVs finished." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
}


##################################################
# Description: 删除无用lv
# Parameters:  无
##################################################
function remove_useless_lv() {
  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Start to remove useless LVs." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  for useless_lv_name in ${useless_lv_name_list}; do
    lv_name=$(echo ${useless_lv_name} | awk -F '/' '{print $4}')
    echo y | lvremove -f ${useless_lv_name} > /dev/null 2>&1
    sed "/${lv_name}/d" /etc/fstab > /etc/fstab.bak
    mv /etc/fstab.bak /etc/fstab
    chmod 644 /etc/fstab
  done
  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Removing useless LVs finished." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
}


##################################################
# Description: 删除多余opt分区
# Parameters:  无
##################################################
function remove_useless_opt() {
  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Start to remove useless opt." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}

  opt_name_list=$(fdisk -l 2> /dev/null \
    | grep "Disk" \
    | grep "/dev/mapper" \
    | grep "opt:" \
    | awk '{print $2 $5}' \
    | awk -F ':' '{print $1}')

  for opt_name in ${opt_name_list}; do
    echo ${opt_name} | grep "/dev" > /dev/null 2>&1
    if [[ $? -eq 0 ]]; then
      useless_opt_name_list+=" ${opt_name}"
    fi
  done

  for useless_opt_name in ${useless_opt_name_list}; do
    umount -l ${useless_opt_name}
    echo y | lvremove -f ${useless_opt_name} > /dev/null 2>&1
    opt_name=$(echo ${useless_opt_name} | awk -F '/' '{print $4}')
    sed "/${opt_name}/d" /etc/fstab > /etc/fstab.bak
    mv /etc/fstab.bak /etc/fstab
    chmod 644 /etc/fstab
  done

  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Removing useless opt finished." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
}


##################################################
# Description: 删除无用vg
# Parameters:  无
##################################################
function remove_useless_vg() {
  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Start to remove useless VG." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}

  vg_list=$(vgdisplay | grep "VG Name" | grep "vg_" | awk '{print $3}')
  for vg_name in ${vg_list}; do
    echo ${vg_name} | grep "vg_root" > /dev/null 2>&1
    if [[ $? -ne 0 ]]; then
      echo y | vgremove -f ${vg_name} > /dev/null 2>&1
    fi
  done

  echo " " >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
  echo "Removing useless VG finished." >> ${LOG_FILE}
  echo "-----------------------------------------------------" >> ${LOG_FILE}
}


##################################################
# Description: 删除无用路径
# Parameters:  无
##################################################
function remove_useless_path() {
  for useless_path in ${USELESS_PATH}; do
    rm -rf ${useless_path}
  done
}

##################################################
# Description: 格式化sda之外的磁盘
# Parameters:  无
##################################################
function clean_void_disk() {
  DISKS=$(parted -l | grep Disk | grep -v "Disk Flags" | grep -v "/dev/sda" | awk '{print $2}' | awk -F':' '{print $1}')
  for disk in $DISKS; do
    echo "Start to mkfs ${disk}" >> ${LOG_FILE}
    mkfs.ext4 -F -F "${disk}" > /dev/null 2>&1
    if [[ $? -ne 0 ]]; then
      echo "Failed to mkfs ${disk}" >> ${LOG_FILE}
      exit 1
    fi
    echo "Success to mkfs ${disk}" >> ${LOG_FILE}
  done

  vgreduce -f --removemissing vg_root > /dev/null 2>&1
  [[ $? -ne 0 ]] && echo "Failed to do removemissing for vg_root" >> ${LOG_FILE} && exit 1

  echo "Success to do removemissing for vg_root" >> ${LOG_FILE}
}


##################################################
# Description: 主函数
# Parameters:  无
##################################################
function main_deal_easysuite_part() {
  check_databackup_existing
  deal_useless_part
  clean_void_disk
}


function main() {
    typeset -i fail_num=0

    # step1: 产品安装前处理easysuite无用分区
    main_deal_easysuite_part > /dev/null 2>&1
    if [[ $? -ne 0 ]]
    then
        ((fail_num=fail_num+1))
        echo "custom_config_easysuite_part"
    else
        echo "[successfully] Deal easysuite part successfully!" >> ${LOG_FILE}
        echo "custom_config_easysuite_part" >> ${LOG_FILE}
    fi

    if [[ ${fail_num} -gt 0 ]]
    then
        echo "==============================DEAL EASYSUITE PART FAILED================================="
        echo "===================================================================================="
        return 1
    fi
}
main $@