#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: disk_partition.sh
# Description: 安装操作系统补丁rpm包
###############################################################################
LOG_FILE="/root/install_config/install_os_patch_pkg.log"
OS_PATCH_PATH="/root/EasySuite/888/*Patch*.tar"
UNZIP_OS_PATCH_PATH="/opt/os_patch"

function logger() {
    log_level=$1
    log_info=$2
    if [[ ${log_level} == "info" ]]; then
        echo [INFO]${log_info} >> ${LOG_FILE}
    elif [[ ${log_level} == "warn" ]]; then
        echo [WARN]${log_info} >> ${LOG_FILE}
    else
        echo [ERROR]${log_info} >> ${LOG_FILE}
    fi
 }

function unzip_os_patch() {
    tar -xvf ${OS_PATCH_PATH} -C ${UNZIP_OS_PATCH_PATH} >/dev/null 2>&1
    cd ${UNZIP_OS_PATCH_PATH} >/dev/null 2>&1
    tar -xvf *.tar.gz >/dev/null 2>&1
    tar -xvf *.tar.gz >/dev/null 2>&1
    cd *Patch
    tar -xvf os*_pkg.tar >/dev/null 2>&1
    tar -xvf os_sudobin.tar >/dev/null 2>&1
}

function install_os_patch() {
    pkg_path=/usr/local/rpmpkg/install_extensions.sh
    if [[ ! -f ${pkg_path} ]];then
        logger "info" "Skip to install the OS Patch package"
        echo "The patch package has been installed."
        return 0
    fi
    bash ${pkg_path} --product carrier >/dev/null 2>&1
    if [[ $? != 0 ]];then
        logger "error" "Failed to install the OS Patch package"
        echo "[ERROR Message] Failed to find patch script" >&2
        echo "[Solution] Check whether /carrier and install_extensions.sh exists in /usr/local/rpmpkg/."
        return 1
    fi
    return 0
}

function scale_install_os_patch() {
    cd ${UNZIP_OS_PATCH_PATH}/*Patch/script
    bash update_*_do.sh --product carrier >/dev/null 2>&1
    if [[ $? != 0 ]];then
        return 1
    fi
    return 0
}

function install_dpdk_dependence_rpm(){
    cd /usr/local/rpmpkg
    rpm -qa |grep numactl-libs
    if [[ $? == 0 ]];then
        echo "[WARN] Dpdk dependence rpm package has been installed."
        return 0
    fi
    rpm -ivh numactl-libs-2.*.rpm >/dev/null 2>&1
}

function scale_os_patch() {
    logger "info" "Start to install the operating system patch package."

    unzip_os_patch
    if [[ $? != 0 ]];then
        logger "error" "Failed to decompress the OS patch package."
        echo "[ERROR Message] Failed to decompress the OS patch package." >&2
        echo "[Solution] Check the integrity of the patch package."
        return 1
    fi
    scale_install_os_patch
    if [[ $? != 0 ]];then
        logger "error" "Failed to install the OS Patch package"
        echo "[ERROR Message] Failed to decompress the OS patch package" >&2
        echo "[Solution] Check whether the patch package matches the OS."
        return 1
    fi
    return 0
}

function main() {
    os_type=$(cat /etc/os-release | grep '^NAME'|awk -F'=' '{print $2}')
    if [[ ${os_type} =~ "Kylin Linux Advanced Server" ]];then
        logger "info" "The Kirin OS does not need to be installed with the OS patch package."
        return 0
    fi
    if [[ ${os_type} =~ "CTyunOS" ]];then
        logger "info" "The CTyunOS does not need to be installed with the OS patch package."
        return 0
    fi
    if [[ ${os_type} =~ "SLES" ]];then
        logger "info" "The SUSE OS does not need to be installed with the OS patch package."
        return 0
    fi

    rm -rf ${UNZIP_OS_PATCH_PATH}
    mkdir -p ${UNZIP_OS_PATCH_PATH}
    type=$1
    if [[ ${type} == "install" ]];then
        install_os_patch
        if [[ $? != 0 ]];then
            return 1
        fi
    fi
    if [[ ${type} == "scale" ]];then
        scale_os_patch
        if [[ $? != 0 ]];then
            return 1
        fi
    fi

    install_dpdk_dependence_rpm
    if [[ $? != 0 ]];then
        logger "error" "Failed to install dpdk dependence package"
        echo "[ERROR Message] Failed to install dpdk dependence package."
        echo "[Solution] Check whether the RPM package numactl-libs on which DPDK depends exists."
    fi
    logger "info" "End to install the operating system patch package."
    return 0
}

main "$@"
