#!/bin/bash
FileName="`basename $0`"
MODULE_NAME="`echo ${FileName} | sed 's/.sh//g'`"
RELATIVE_PATH="`dirname $0`"
WAIT_TIME=3

# 检查操作系统类型是否支持
function check_support_os() {
    local support_os_list os_type
    support_os_list=("euleros" "sles")
    os_type=$(grep "^\s*ID=\"[a-zA-Z]\+\"" /etc/os-release | awk -F\" '{print $2}')
    if echo "${support_os_list[@]}" | grep -qw "${os_type}"; then
        return 0
    else
        print_log "ERROR" "Unsupported os type ${os_type}."
        return 1
    fi
}

function clean_vgs()
{
    vgs=$(vgs | grep -v "VG" | awk '{ print $1}' | xargs )
    for vg in ${vgs}
    do
       [ ${vg} != "vg_root" ] && vgremove -y ${vg}
    done
}

function clean_pvs()
{
    pvs=$(pvs | grep -v "PV" | awk '{ print $1}' | xargs )
    for pv in ${pvs}
    do
        echo ${pv}|grep "da" &>/dev/null
        [ $? -ne 0 ] && pvremove -y ${pv}
    done
}

function clean_part()
{
    disks=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}'| xargs)
    disk_1=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "1p")
    for disk in ${disks}
    do
        [ ${disk} == ${disk_1} ] && continue
        number=$(parted /dev/${disk} p 2>/dev/null |grep -n "Number" |awk -F: '{print $1}')
        [ -z ${number} ] && continue
        let number++
        numbers=$(parted /dev/${disk} p 2>/dev/null |tail -n +${number} | sed '/^\s*$/d' | awk '{print $1}' | xargs)
        for part in ${numbers}
        do
            echo ${part} | grep "^[0-9]\+" &>/dev/null
            [ $? -eq 0 ] && parted -s /dev/${disk} rm ${part} 
        done
    done        
}

#############################################
# 在已存在分区的磁盘上新建分区
#   已有磁盘分区数大于等于4时，无法继续分区
#############################################
function create_old_disk_partition()
{
    local disk_name="$1"
    local disk_path=/dev/${disk_name}

    # 当前磁盘分区大于等于4，无法继续分区，报错
    local part_count=$(lsblk -l ${disk_path} -o NAME,SIZE,TYPE | grep part | wc -l)
    if [ ${part_count} -ge 4 ]; then
        echo "The number of ${disk_name} partitions is greater than or equal to 4. No partition can be created. Skip this step."
        return 1
    fi

    local old_partition=$(lsblk -l ${disk_path} -o NAME,SIZE,TYPE | grep part | awk '{print $1}' | xargs)
    if [ ${IS_UEFI} -eq 0 ]; then
        expect<<END
      spawn parted ${disk_path} p
      expect {
          "*Fix/Ignore?*" { send "f\r"; }
      }
      expect eof
END
        start_point=$(parted -s ${disk_path} unit b print free | sed '/^$/d' | tail -1 | awk '{print $1}' | tr -cd [0-9])
        [ -n "${start_point}" ] || { echo "Failed to obtain the disk space start point."; return 1; }
        parted -s ${disk_path} unit b mkpart primary ext4 ${start_point} 100% || { echo "Failed to create the partition. 3"; return 1; }
    else
        expect<<EOF
    set timeout 10
    spawn fdisk ${disk_path}
    expect "*help):"
    send "n\r"
    expect {
      "*primary*" { send "p\r";exp_continue}
      "*Partition number*" {send "\r"}
    }
    expect "*First sector*"
    send "\r"
    expect "*Last sector*"
    send "\r"
    expect "*help):"
    send "w\r"
    expect eof
EOF
    fi
    sleep "${WAIT_TIME}"
    partprobe
    local new_partition=""
    for one_partition in $(lsblk -l ${disk_path} -o NAME,SIZE,TYPE | grep part | awk '{print $1}' | xargs); do
        echo "${old_partition[@]}" | grep -qw ${one_partition}
        [ $? -ne 0 ] && new_partition=${one_partition}
    done
    [ -n "${new_partition}" ] || { echo "Failed to query the new partition name."; return 1; }
    NEW_PARTITION_NAME="/dev/${new_partition}"
    return 0
}

function create_lv()
{
    if lvs | grep -qw opt; then
        LOG "WARNING" "LVM: opt already exists."
        return 0
    fi

    echo "Start to create the opt LVM."
    # 将新创建的PV中的所有空间都划给opt分区
    lvcreate -y -l +100%FREE -n opt vg_root ${NEW_PARTITION_NAME} || { echo "Failed to create the lv"; return 1; }
    echo "The opt LVM is created successfully."
    echo "Start to format the opt partition."
    mkfs.ext4 /dev/mapper/vg_root-opt || { echo "Failed to format the partition."; return 1; }
    echo "The opt partition is formatted successfully."
    [ ! -d /opt ] && mkdir -p /opt
    echo "Start to mount the opt partition."
    mount /dev/mapper/vg_root-opt /opt || { echo "Failed to mount the opt partition."; return 1; }
    if ! grep -q "^/dev/mapper/vg_root-opt" /etc/fstab; then
        echo "/dev/mapper/vg_root-opt /opt ext4 defaults 1 2" >>/etc/fstab
        if [ $? -ne 0 ]; then
            echo "Failed to write partition information."
            return 1
        fi
    fi

    echo "The opt file is mounted successfully."
}

function create_opt()
{
    local disk_name=$(lsblk -o NAME,TYPE | grep sd | grep disk | grep -v ^fd | awk '{print $1}' | head -1)
    if [ -z "${disk_name}" ]; then
        echo "Failed to get disk name."
        return 1
    fi

    # 将第一块磁盘的剩余空间划给opt分区
    create_old_disk_partition "${disk_name}" || return 1

    if [ -n "${NEW_PARTITION_NAME}" ]; then
        echo "Start to expand the space of ${NEW_PARTITION_NAME} to vg_root."
        pvcreate -y ${NEW_PARTITION_NAME} || { echo "Failed to pvcreate the ${NEW_PARTITION_NAME}."; return 1; }
        vgextend vg_root ${NEW_PARTITION_NAME} || { echo "Failed to vgextend the vg_root."; return 1; }
        echo "The vg_root space is expanded successfully."
    fi

    create_lv || return 1
}

function extend_vg_root() {
    local disk_name disk_path start_point end_point left_space gigabit new_partition_num

    disk_name=$(lsblk -o NAME,TYPE | grep sd | grep disk | grep -v ^fd | awk '{print $1}' | head -1)
    disk_path=/dev/${disk_name}

    # 执行parted命令打印磁盘剩余的空间
    start_point=$(parted -s ${disk_path} unit b print free | grep -i "free" | tail -1 | awk '{print $1}' | tr -cd "[0-9]")
    end_point=$(parted -s ${disk_path} unit b print free | grep -i "free" | tail -1 | awk '{print $2}' | tr -cd "[0-9]")
    left_space=$(parted -s ${disk_path} unit b print free | grep -i "free" | tail -1 | awk '{print $3}' | tr -cd "[0-9]")
    gigabit=$((10**9))
    # 剩余空间不足1G时跳过扩容
    if [ "${left_space}" -le "${gigabit}" ]; then
        echo "If the remaining space is less than 1 GB, skip the expansion step."
        return 0
    fi

    parted -s ${disk_path} unit b mkpart primary ext4 ${start_point} ${end_point}
    if [ $? -ne 0 ]; then
        echo "Failed to extend the vg_root volume group."
        return 1
    fi

    new_partition_num=$(parted -s ${disk_path} print | grep "primary" | tail -1 | awk '{print $1}')
    if [ -z "${new_partition_num}" ]; then
        echo "Check new partition number failed."
        return 1
    fi
    sleep "${WAIT_TIME}"
    pvcreate -y "${disk_path}${new_partition_num}" || { echo "Failed to create new pv."; return 1; }
    vgextend vg_root "${disk_path}${new_partition_num}" || { echo "Failed to vgextend the vg_root."; return 1; }
}

function makelvm_uefi()
{
    diskcount=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | wc -l)
    disk_1=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "1p")
    systemd-detect-virt | grep -i "none" >/dev/null 2>&1
    if [ $? -eq 0 ]
    then
        if [ ${diskcount} -eq 1 -o ${diskcount} -eq 2 ]
        then
            if [ -e /etc/SuSE-release ]; then
                IS_UEFI=0
                create_opt
                return $?
            fi
            lvcreate -y -l 100%VG -n opt vg_root
            mkfs.ext4 /dev/mapper/vg_root-opt
            [ ! -d /opt ] && mkdir -p /opt
            mount /dev/mapper/vg_root-opt /opt || return 1
            grep "^/dev/mapper/vg_root-opt" /etc/fstab &>/dev/null
            [ $? -ne 0 ] && echo "/dev/mapper/vg_root-opt /opt ext4 defaults 1 2" >> /etc/fstab
            echo "mount /dev/mapper/vg_root-opt to /opt successfully."
        elif [ ${diskcount} -eq 3 -o ${diskcount} -eq 4 -o ${diskcount} -ge 5 ]
        then
            extend_vg_root || return 1
            disk_2=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "2p")
            disk_3=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "3p")
            parted -s /dev/${disk_2} mklabel gpt
            parted -s /dev/${disk_2} mkpart primary 1 100%
            sleep "${WAIT_TIME}"
            partprobe
            pvdb_1="${disk_2}1"
            pvcreate -y /dev/${pvdb_1}
            vgextend vg_root /dev/${pvdb_1}
            # 若vg_root剩余空间大于100G，则预留100G空间
            left_space=$(vgs --units G | grep -w "vg_root" | awk '{print $7}' | sed "s/G//")
            if [ $(echo "${left_space} > 100" | bc) -eq 1 ]; then
                lvcreate -y -L 100G -n spare vg_root
            fi
            lvcreate -y -l 100%VG -n opt vg_root

            mkfs.ext4 /dev/mapper/vg_root-opt
            [ ! -d /opt ] && mkdir -p /opt
            mount /dev/mapper/vg_root-opt /opt || return 1
            grep "^/dev/mapper/vg_root-opt" /etc/fstab &>/dev/null
            [ $? -ne 0 ] && echo "/dev/mapper/vg_root-opt /opt ext4 defaults 1 2" >> /etc/fstab
            echo "mount /dev/mapper/vg_root-opt to /opt successfully."

            parted -s /dev/${disk_3} mklabel gpt
            parted -s /dev/${disk_3} mkpart primary 1 100%
            sleep "${WAIT_TIME}"
            partprobe
            pvdc_1="${disk_3}1"
            pvcreate -y /dev/${pvdc_1}
            vgcreate vg_sdc /dev/${pvdc_1}
            lvcreate -y -l 100%VG -n systembackup vg_sdc
            mkfs.ext4 /dev/mapper/vg_sdc-systembackup
            [ ! -d /opt/systembackup ] && mkdir -p /opt/systembackup
            mount /dev/mapper/vg_sdc-systembackup /opt/systembackup || return 1
            grep "^/dev/mapper/vg_sdc-systembackup" /etc/fstab &>/dev/null
            [ $? -ne 0 ] && echo "/dev/mapper/vg_sdc-systembackup /opt/systembackup ext4 defaults 1 2" >> /etc/fstab     
            echo "mount /dev/mapper/vg_sdc-systembackup to /opt/systembackup successfully."
            disk_4=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "4p")
            if [ -n "${disk_4}" ]
            then
                parted -s /dev/${disk_4} mklabel gpt
                parted -s /dev/${disk_4} mkpart primary 1 100%
                sleep "${WAIT_TIME}"
                partprobe
                pvdd_1="${disk_4}1"
                pvcreate -y /dev/${pvdd_1}
                vgcreate vg_sdd /dev/${pvdd_1}
                lvcreate -y -l 100%VG -n databackup vg_sdd
                mkfs.ext4 /dev/mapper/vg_sdd-databackup
                [ ! -d /opt/databackup ] && mkdir -p /opt/databackup
                mount /dev/mapper/vg_sdd-databackup /opt/databackup || return 1
                grep "^/dev/mapper/vg_sdd-databackup" /etc/fstab &>/dev/null
                [ $? -ne 0 ] && echo "/dev/mapper/vg_sdd-databackup /opt/databackup ext4 defaults 1 2" >> /etc/fstab
                echo "mount /dev/mapper/vg_sdd-databackup to /opt/databackup successfully."
            fi
        else
            echo "Error:The number of disk groups is not supported,exit!"
            exit 1
        fi
    fi
}


function makelvm_bios()
{
    diskcount=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | wc -l)
    disk_1=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "1p")
    systemd-detect-virt | grep -i "none" >/dev/null 2>&1
    if [ $? -eq 0 ]
    then
        if [ ${diskcount} -eq 1 -o ${diskcount} -eq 2 ]
        then
            if [ -e /etc/SuSE-release ]; then
                IS_UEFI=1
                create_opt
                return $?
            fi
            lvcreate -y -l 100%VG -n opt vg_root
            mkfs.ext4 /dev/mapper/vg_root-opt
            [ ! -d /opt ] && mkdir -p /opt
            mount /dev/mapper/vg_root-opt /opt || return 1
            grep "^/dev/mapper/vg_root-opt" /etc/fstab &>/dev/null
            [ $? -ne 0 ] && echo "/dev/mapper/vg_root-opt /opt ext4 defaults 1 2" >> /etc/fstab
            echo "mount /dev/mapper/vg_root-opt to /opt successfully."
        elif [ ${diskcount} -eq 3 -o ${diskcount} -eq 4 -o ${diskcount} -ge 5 ]
        then
            extend_vg_root || return 1
            disk_2=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "2p")
            disk_3=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "3p")
            /sbin/fdisk /dev/${disk_2}  <<EOF
n
p
1


t

8e
w
EOF
            sleep "${WAIT_TIME}"
            partprobe
            pvdb_1="${disk_2}1"
            pvcreate -y /dev/${pvdb_1}
            vgextend vg_root /dev/${pvdb_1}
            # 若vg_root剩余空间大于100G，则预留100G空间
            left_space=$(vgs --units G | grep -w "vg_root" | awk '{print $7}' | sed "s/G//")
            if [ $(echo "${left_space} > 100" | bc) -eq 1 ]; then
                lvcreate -y -L 100G -n spare vg_root
            fi
            lvcreate -y -l 100%VG -n opt vg_root

            mkfs.ext4 /dev/mapper/vg_root-opt
            [ ! -d /opt ] && mkdir -p /opt
            mount /dev/mapper/vg_root-opt /opt || return 1
            grep "^/dev/mapper/vg_root-opt" /etc/fstab &>/dev/null
            [ $? -ne 0 ] && echo "/dev/mapper/vg_root-opt /opt ext4 defaults 1 2" >> /etc/fstab
            echo "mount /dev/mapper/vg_root-opt to /opt successfully."


            /sbin/fdisk /dev/${disk_3}  <<EOF
n
p
1


t

8e
w
EOF
            sleep "${WAIT_TIME}"
            partprobe
            pvdc_1="${disk_3}1"
            pvcreate -y /dev/${pvdc_1}
            vgcreate vg_sdc /dev/${pvdc_1}
            lvcreate -y -l 100%VG -n systembackup vg_sdc
            mkfs.ext4 /dev/mapper/vg_sdc-systembackup
            [ ! -d /opt/systembackup ] && mkdir -p /opt/systembackup
            mount /dev/mapper/vg_sdc-systembackup /opt/systembackup || return 1
            grep "^/dev/mapper/vg_sdc-systembackup" /etc/fstab &>/dev/null
            [ $? -ne 0 ] && echo "/dev/mapper/vg_sdc-systembackup /opt/systembackup ext4 defaults 1 2" >> /etc/fstab
            echo "mount /dev/mapper/vg_sdc-systembackup to /opt/systembackup successfully."
            disk_4=$(lsblk -o name,rm,ro,type -n|awk '{if ($2==0 && $3==0 && $4=="disk"){print $1}}' | sed -n "4p")
            if [ -n "${disk_4}" ]
            then
                /sbin/fdisk /dev/${disk_4}  <<EOF
n
p
1


t

8e
w
EOF
                sleep "${WAIT_TIME}"
                partprobe
                pvdd_1="${disk_4}1"
                pvcreate -y /dev/${pvdd_1}
                vgcreate vg_sdd /dev/${pvdd_1}
                lvcreate -y -l 100%VG -n databackup vg_sdd
                mkfs.ext4 /dev/mapper/vg_sdd-databackup
                [ ! -d /opt/databackup ] && mkdir -p /opt/databackup
                mount /dev/mapper/vg_sdd-databackup /opt/databackup || return 1
                grep "^/dev/mapper/vg_sdd-databackup" /etc/fstab &>/dev/null
                [ $? -ne 0 ] && echo "/dev/mapper/vg_sdd-databackup /opt/databackup ext4 defaults 1 2" >> /etc/fstab
                echo "mount /dev/mapper/vg_sdd-databackup to /opt/databackup successfully."
            fi
        else
            echo "Error:The number of disk groups is not supported,exit!"
            exit 1
        fi
    fi
}


function makelvm()
{
    # 若标志文件已存在，则跳过划分区步骤
    [ -f /var/log/partition_success_flag ] && return 0
    check_support_os || return 0
    clean_vgs
    clean_pvs
    clean_part
    lsblk | grep "/boot/efi$" &>/dev/null
    if [ $? -eq 0 ]
    then
        makelvm_uefi || { echo "Create partition failed, please use 'parted {disk_name}' command to clear disk first."; return 1; }
    else
        makelvm_bios  || { echo "Create partition failed, please use 'parted {disk_name}' command to clear disk first."; return 1; }
    fi
    touch /var/log/partition_success_flag
}
makelvm $@
RET_MAIN=$?
if [[ "${RET_MAIN}" == "0" ]]
then
    logger -t $USER -p local0.info "$FileName;Successful;${OPERIP:-127.0.0.1};Excute execution $FileName."
else
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Excute execution $FileName."
fi
exit ${RET_MAIN}
