#!/bin/bash
################################################################################
# Copyright © Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
#
# Description: 操作系统主备分区倒换并重启OS
#
# Usage： bash switch_os.sh
#
# Exitcode：0成功，1失败
#
#################################################################################

source "${WORKSPACE}/online/env.propertity"
source "${WORKSPACE}/online/common.sh"
source "${WORKSPACE}/online/backup.sh"

#################################################################################
# Description： 设置新版本选项为默认启动项，确保重启后进入新系统
# @Return: 0成功 1失败
################################################################################
function set_default_grub(){
    log "INFO" "Start to set default grub menu..." "${UPGRADE_LOG_FILE}"
    grub2-set-default "${NEW_VERSION_GRUB_MENU}" 
    if [ $? -eq 0 ]; then
        log "INFO" "Default grub menu has been set as ${NEW_VERSION_GRUB_MENU} successfully!" "${UPGRADE_LOG_FILE}"
        return 0
    fi

    log "ERROR" "Set default grub menu as ${NEW_VERSION_GRUB_MENU} failed! Please check os is running normally or not!" "${UPGRADE_LOG_FILE}"
    return 1
}

#################################################################################
# Description： 日志压缩转储，每次执行完将本次执行日志压缩转储
# @Return: 0成功 1失败
################################################################################
function log_compress(){
    local file_name="${UPGRADE_LOG_FILE%.*}"
    local zip_file="${file_name}_$(date '+%Y%m%d%H%M%S').zip"
    zip -moj "${zip_file}" "${UPGRADE_LOG_FILE}" || return 1
}


#################################################################################
# Description： 清理冗余文件，包括调测工具、编译工具等
# @Return: 0成功 1失败
################################################################################
function clean_redundant_file(){
    local file_list=$(cat "${WORKSPACE}/cfg/clean/post_clean.cfg")
    for line in ${file_list}; do
        [ -n "${line}" ] && rm -f "/back/${line}" 2>&1 >> "${UPGRADE_LOG_FILE}" 
    done 
}

#################################################################################
# Description： OS主备切换入口
# @Return: 0成功 1失败
################################################################################
function main(){
    init_log "${UPGRADE_LOG_FILE}"

    log "INFO" "Start to switch OS to backup partition..." "${UPGRADE_LOG_FILE}"
    check_is_upgrade
    local upgrade_status=$? 
    if [ ${upgrade_status} -eq ${UPGRADE_REBOOTED_CODE} ]; then
        log "INFO" "The patch has been installed in this environment, skip the reboot progress!" "${UPGRADE_LOG_FILE}"
        return 0
    elif [ ${upgrade_status} -eq ${NOT_UPGRADE_NOT_REBOOTED_CODE} ]; then
        log "ERROR" "The backup partition not upgrade yet, please execute the upgrade progress first!" "${UPGRADE_LOG_FILE}"
        return 1
    fi
    
    post_backup || return 1

    clean_redundant_file || return 1

    set_default_grub || return 1

    log_compress || return 1
}

main